/*
 * Copyright (C) 2019-2020 Alexandros Theodotou <alex at zrythm dot org>
 *
 * This file is part of Zrythm
 *
 * Zrythm is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Zrythm is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Zrythm.  If not, see <https://www.gnu.org/licenses/>.
 */

/**
 * \file
 *
 * Localization utils.
 *
 */

#ifndef __UTILS_LOCALIZATION_H__
#define __UTILS_LOCALIZATION_H__

#include <stdbool.h>

#include <gtk/gtk.h>

/**
 * @addtogroup utils
 *
 * @{
 */

typedef enum LocalizationLanguage
{
  /* note: this part will be auto-generated by
   * meson */
  LL_AF_ZA,
  LL_AR,
  LL_CA,
  LL_DE,
  LL_EL,
  LL_EN,
  LL_EN_GB,
  LL_ES,
  LL_FA,
  LL_FR,
  LL_GL,
  LL_HE,
  LL_HI,
  LL_HU,
  LL_ID,
  LL_IT,
  LL_JA,
  LL_KO,
  LL_NB_NO,
  LL_MK,
  LL_NL,
  LL_PL,
  LL_PT,
  LL_PT_BR,
  LL_RU,
  LL_SV,
  LL_TH,
  LL_TR,
  LL_UK,
  LL_VI,
  LL_ZH_CN,
  LL_ZH_TW,

  NUM_LL_LANGUAGES,
} LocalizationLanguage;

/**
 * Returns the localized language name (e.g.
 * "Ελληνικά").
 */
const char *
localization_get_localized_name (
  LocalizationLanguage lang);

/**
 * Returns the character string code for the
 * language (e.g. "fr").
 */
const char *
localization_get_string_code (
  LocalizationLanguage lang);

/**
 * Returns the localized language name with the
 * code (e.g. "Ελληνικά [el]").
 */
const char *
localization_get_string_w_code (
  LocalizationLanguage lang);

const char **
localization_get_language_codes (void);

const char **
localization_get_language_strings (void);

const char **
localization_get_language_strings_w_codes (void);

/**
 * Returns the first locale found matching the given
 * language, or NULL if a locale for the given
 * language does not exist.
 *
 * Must be free'd with g_free().
 */
char *
localization_locale_exists (
  LocalizationLanguage lang);

/**
 * Sets the locale to the currently selected one and
 * inits gettext.
 *
 * @param use_locale Use the user's local instead of
 *   the Zrythm settings.
 *
 * Returns if a locale for the selected language
 * exists on the system or not.
 */
bool
localization_init (
  bool use_locale,
  bool print_debug_messages,
  bool queue_error_if_not_installed);

/**
 * @}
 */

#endif
