"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SearchFilterCollection = void 0;
const ArgumentException_1 = require("../../Exceptions/ArgumentException");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const LogicalOperator_1 = require("../../Enumerations/LogicalOperator");
const ServiceValidationException_1 = require("../../Exceptions/ServiceValidationException");
const Strings_1 = require("../../Strings");
const SearchFilter_1 = require("./SearchFilter");
/**
 * Represents a collection of search filters linked by a logical operator. Applications can use SearchFilterCollection to define complex search filters such as "Condition1 AND Condition2".
 */
class SearchFilterCollection extends SearchFilter_1.SearchFilter {
    constructor(logicalOperator, _searchFilters) {
        super();
        this.searchFilters = [];
        this.logicalOperator = LogicalOperator_1.LogicalOperator.And;
        var searchFilters = [];
        if (arguments.length <= 2) {
            if (ExtensionMethods_1.ArrayHelper.isArray(_searchFilters)) {
                searchFilters = _searchFilters;
            }
            else if (typeof SearchFilter_1.SearchFilter[_searchFilters] !== 'undefined') {
                searchFilters.push(arguments[1]);
            }
        }
        else {
            for (var _i = 1; _i < arguments.length; _i++) {
                searchFilters[_i - 1] = arguments[_i];
            }
        }
        this.logicalOperator = logicalOperator || this.logicalOperator;
        this.AddRange(searchFilters);
    }
    /**
     * Gets the total number of search filters in the collection.
     */
    get Count() {
        return this.searchFilters.length;
    }
    /**
     * Gets or sets the logical operator that links the serach filters in this collection.
     */
    get LogicalOperator() {
        return this.logicalOperator;
    }
    set LogicalOperator(value) {
        this.logicalOperator = value;
    }
    /**
     * Gets or sets the search filter at the specified index.
     *
     * @param   {number}   index   The zero-based index of the search filter to get or set.
     * @return  {SearchFilter}           The search filter at the specified index.
     */
    _getItem(index) {
        if (index < 0 || index >= this.Count) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("index", Strings_1.Strings.IndexIsOutOfRange);
        }
        return this.searchFilters[index];
    }
    /**
     * Gets or sets the search filter at the specified index.
     *
     * @param   {number}   index   The zero-based index of the search filter to get or set.
     * @return  {SearchFilter}           The search filter at the specified index.
     */
    _setItem(index, value) {
        if (index < 0 || index >= this.Count) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("index", Strings_1.Strings.IndexIsOutOfRange);
        }
        this.searchFilters[index] = value;
    }
    /**
     * Adds a search filter of any type to the collection.
     *
     * @param   {SearchFilter}   searchFilter   The search filter to add. Available search filter classes include SearchFilter.IsEqualTo, SearchFilter.ContainsSubstring and SearchFilter.SearchFilterCollection.
     */
    Add(searchFilter) {
        if (searchFilter == null) {
            throw new ArgumentException_1.ArgumentNullException("searchFilter");
        }
        searchFilter.OnChange.push(this.SearchFilterChanged.bind(this));
        this.searchFilters.push(searchFilter);
        this.Changed();
    }
    /**
     * Adds multiple search filters to the collection.
     *
     * @param   {SearchFilter[]}   searchFilters   The search filters to add. Available search filter classes include SearchFilter.IsEqualTo, SearchFilter.ContainsSubstring and SearchFilter.SearchFilterCollection.
     */
    AddRange(searchFilters) {
        if (searchFilters == null) {
            throw new ArgumentException_1.ArgumentNullException("searchFilters");
        }
        for (let searchFilter of searchFilters) {
            searchFilter.OnChange.push(this.SearchFilterChanged.bind(this));
        }
        ExtensionMethods_1.ArrayHelper.AddRange(this.searchFilters, searchFilters);
        this.Changed();
    }
    /**
     * Clears the collection.
     */
    Clear() {
        if (this.Count > 0) {
            for (let searchFilter of this.searchFilters) {
                ExtensionMethods_1.ArrayHelper.RemoveEntry(searchFilter.OnChange, this.SearchFilterChanged);
                //searchFilter.OnChange -= this.SearchFilterChanged;
            }
            this.searchFilters.splice(0);
            this.Changed();
        }
    }
    /**
     * Determines whether a specific search filter is in the collection.
     *
     * @param   {SearchFilter} 	searchFilter   The search filter to locate in the collection.
     * @return  {boolean}		True is the search filter was found in the collection, false otherwise.
     */
    Contains(searchFilter) {
        return this.searchFilters.indexOf(searchFilter) >= 0;
    }
    /**
     *  Returns an enumerator that iterates through the collection. this case this.items
     */
    GetEnumerator() {
        return this.searchFilters;
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @return  {string}      XML element name.
     */
    GetXmlElementName() { return LogicalOperator_1.LogicalOperator[this.LogicalOperator]; }
    /**
     * @internal Validate instance.
     */
    InternalValidate() {
        for (let i = 0; i < this.Count; i++) {
            try {
                this._getItem(i).InternalValidate();
            }
            catch (e) {
                if (e instanceof ServiceValidationException_1.ServiceValidationException) {
                    throw new ServiceValidationException_1.ServiceValidationException(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.SearchFilterAtIndexIsInvalid, i), e);
                }
            }
        }
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}                 jsObject                Json Object converted from XML.
     * @param   {ExchangeService}     service                 The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        for (var key in jsObject) {
            if (key.indexOf("__") === 0)
                continue;
            let filter = SearchFilter_1.SearchFilter.LoadFromXmlJsObject(jsObject[key], service, key);
            if (filter) {
                this.Add(filter);
            }
        }
    }
    /**
     * Removes a search filter from the collection.
     *
     * @param   {SearchFilter}   searchFilter   The search filter to remove.
     */
    Remove(searchFilter) {
        if (searchFilter == null) {
            throw new ArgumentException_1.ArgumentNullException("searchFilter");
        }
        if (this.Contains(searchFilter)) {
            ExtensionMethods_1.ArrayHelper.RemoveEntry(searchFilter.OnChange, this.SearchFilterChanged);
            ExtensionMethods_1.ArrayHelper.RemoveEntry(this.searchFilters, searchFilter);
            this.Changed();
        }
    }
    /**
     * Removes the search filter at the specified index from the collection.
     *
     * @param   {number}   index   The zero-based index of the search filter to remove.
     */
    RemoveAt(index) {
        if (index < 0 || index >= this.Count) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("index", Strings_1.Strings.IndexIsOutOfRange);
        }
        ExtensionMethods_1.ArrayHelper.RemoveEntry(this._getItem(index).OnChange, this.SearchFilterChanged);
        this.searchFilters.splice(index, 1);
        this.Changed();
    }
    /**
     * A search filter has changed.
     *
     * @param   {ComplexProperty}   complexProperty   The complex property.
     */
    SearchFilterChanged(complexProperty) { this.Changed(); }
    /**
     * @internal Writes the elements to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        for (let searchFilter of this.searchFilters) {
            searchFilter.WriteToXml(writer);
        }
    }
    /**
     * @internal Writes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteToXml(writer) {
        // If there is only one filter in the collection, which developers tend to do,
        // we need to not emit the collection and instead only emit the one filter within
        // the collection. This is to work around the fact that EWS does not allow filter
        // collections that have less than two elements.
        if (this.Count == 1) {
            this._getItem(0).WriteToXml(writer);
        }
        else {
            super.WriteToXml(writer);
        }
    }
}
exports.SearchFilterCollection = SearchFilterCollection;
//# sourceMappingURL=SearchFilter.SearchFilterCollection.js.map