"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainsSubstring = void 0;
const ComparisonMode_1 = require("../../Enumerations/ComparisonMode");
const ContainmentMode_1 = require("../../Enumerations/ContainmentMode");
const ServiceValidationException_1 = require("../../Exceptions/ServiceValidationException");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const Strings_1 = require("../../Strings");
const XmlAttributeNames_1 = require("../../Core/XmlAttributeNames");
const XmlElementNames_1 = require("../../Core/XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const SearchFilter_PropertyBasedFilter_1 = require("./SearchFilter.PropertyBasedFilter");
/**
 * Represents a search filter that checks for the presence of a substring inside a text property. Applications can use ContainsSubstring to define conditions such as "Field CONTAINS Value" or "Field IS PREFIXED WITH Value".
 */
class ContainsSubstring extends SearchFilter_PropertyBasedFilter_1.PropertyBasedFilter {
    constructor(propertyDefinition = null, value = null, containmentMode = ContainmentMode_1.ContainmentMode.Substring, comparisonMode = ComparisonMode_1.ComparisonMode.IgnoreCase) {
        arguments.length === 0 ? super() : super(propertyDefinition);
        this.value = value;
        this.containmentMode = containmentMode;
        this.comparisonMode = comparisonMode;
    }
    /**
     * Gets or sets the containment mode.
     */
    get ContainmentMode() {
        return this.containmentMode;
    }
    set ContainmentMode(value) {
        this.SetFieldValue({ getValue: () => this.containmentMode, setValue: (updateValue) => { this.containmentMode = updateValue; } }, value);
    }
    /**
     * Gets or sets the comparison mode.
     */
    get ComparisonMode() {
        return this.comparisonMode;
    }
    set ComparisonMode(value) {
        this.SetFieldValue({ getValue: () => this.comparisonMode, setValue: (updateValue) => { this.comparisonMode = updateValue; } }, value);
    }
    /**
     * Gets or sets the value to compare the specified property with.
     */
    get Value() {
        return this.value;
    }
    set Value(value) {
        this.SetFieldValue({ getValue: () => this.value, setValue: (updateValue) => { this.value = updateValue; } }, value);
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @return  {string}      XML element name.
     */
    GetXmlElementName() { return XmlElementNames_1.XmlElementNames.Contains; }
    /**
     * @internal Validate instance.
     */
    InternalValidate() {
        super.InternalValidate();
        if (ExtensionMethods_1.StringHelper.IsNullOrEmpty(this.value)) {
            throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.ValuePropertyMustBeSet);
        }
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}				jsObject	Json Object converted from XML.
     * @param   {ExchangeService}	service	The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        super.LoadFromXmlJsObject(jsObject, service);
        this.value = jsObject[XmlElementNames_1.XmlElementNames.Constant][XmlElementNames_1.XmlElementNames.Value];
        this.containmentMode = ContainmentMode_1.ContainmentMode[jsObject[XmlAttributeNames_1.XmlAttributeNames.ContainmentMode]];
        this.comparisonMode = ComparisonMode_1.ComparisonMode[jsObject[XmlAttributeNames_1.XmlAttributeNames.ContainmentComparison]];
    }
    /**
     * @internal Writes the attributes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteAttributesToXml(writer) {
        super.WriteAttributesToXml(writer);
        writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.ContainmentMode, ContainmentMode_1.ContainmentMode[this.ContainmentMode]);
        writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.ContainmentComparison, ComparisonMode_1.ComparisonMode[this.ComparisonMode]);
    }
    /**
     * @internal Writes the elements to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        super.WriteElementsToXml(writer);
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Constant);
        writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.Value, this.Value);
        writer.WriteEndElement(); // Constant
    }
}
exports.ContainsSubstring = ContainsSubstring;
//# sourceMappingURL=SearchFilter.ContainsSubstring.js.map