"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CalendarView = void 0;
const ArgumentException_1 = require("../Exceptions/ArgumentException");
const ItemTraversal_1 = require("../Enumerations/ItemTraversal");
const ServiceObjectType_1 = require("../Enumerations/ServiceObjectType");
const ServiceValidationException_1 = require("../Exceptions/ServiceValidationException");
const Strings_1 = require("../Strings");
const XmlAttributeNames_1 = require("../Core/XmlAttributeNames");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const ViewBase_1 = require("./ViewBase");
/**
 * Represents a date range view of appointments in calendar folder search operations.
 */
class CalendarView extends ViewBase_1.ViewBase {
    constructor(startDate, endDate, maxItemsReturned = null) {
        super();
        this.traversal = ItemTraversal_1.ItemTraversal.Shallow;
        this.maxItemsReturned = null;
        this.startDate = null;
        this.endDate = null;
        this.startDate = startDate;
        this.endDate = endDate;
        this.MaxItemsReturned = maxItemsReturned;
    }
    /**
     * Gets or sets the start date.
     */
    get StartDate() {
        return this.startDate;
    }
    set StartDate(value) {
        this.startDate = value;
    }
    /**
     * Gets or sets the end date.
     */
    get EndDate() {
        return this.endDate;
    }
    set EndDate(value) {
        this.endDate = value;
    }
    /**
     * The maximum number of items the search operation should return.
     */
    get MaxItemsReturned() {
        return this.maxItemsReturned;
    }
    set MaxItemsReturned(value) {
        if (value !== null && value <= 0) {
            throw new ArgumentException_1.ArgumentException(Strings_1.Strings.ValueMustBeGreaterThanZero);
        }
        this.maxItemsReturned = value;
    }
    /**
     * Gets or sets the search traversal mode. Defaults to ItemTraversal.Shallow.
     */
    get Traversal() {
        return this.traversal;
    }
    set Traversal(value) {
        this.traversal = value;
    }
    /**
     * @internal Gets the maximum number of items or folders the search operation should return.
     *
     * @return  {number}      The maximum number of items the search operation should return.
     */
    GetMaxEntriesReturned() { return this.MaxItemsReturned; }
    /**
     * @internal Gets the type of service object this view applies to.
     *
     * @return  {ServiceObjectType}      A ServiceObjectType value.
     */
    GetServiceObjectType() { return ServiceObjectType_1.ServiceObjectType.Item; }
    /**
     * @internal Gets the name of the view XML element.
     *
     * @return  {string}      XML element name.
     */
    GetViewXmlElementName() { return XmlElementNames_1.XmlElementNames.CalendarView; }
    /**
     * @internal Validate instance.
     *
     * @param   {ServiceRequestBase}   request   The request using this view.
     */
    InternalValidate(request) {
        super.InternalValidate(request);
        if (this.endDate < this.StartDate) {
            throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.EndDateMustBeGreaterThanStartDate);
        }
    }
    /**
     * @internal Writes the search settings to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer    The writer.
     * @param   {Grouping}              groupBy   The group by clause.
     */
    InternalWriteSearchSettingsToXml(writer, groupBy) { }
    /**
     * @internal Write to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    InternalWriteViewToXml(writer) {
        super.InternalWriteViewToXml(writer);
        writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.StartDate, this.StartDate);
        writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.EndDate, this.EndDate);
    }
    /**
     * @internal Writes the attributes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteAttributesToXml(writer) { writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.Traversal, ItemTraversal_1.ItemTraversal[this.Traversal]); }
    /**
     * @internal Writes OrderBy property to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer
     */
    WriteOrderByToXml(writer) { }
}
exports.CalendarView = CalendarView;
//# sourceMappingURL=CalendarView.js.map