"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GetStreamingEventsResults = void 0;
const EventType_1 = require("../Enumerations/EventType");
const EwsServiceJsonReader_1 = require("../Core/EwsServiceJsonReader");
const FolderEvent_1 = require("./FolderEvent");
const GetEventsResults_1 = require("./GetEventsResults");
const ItemEvent_1 = require("./ItemEvent");
const NotificationGroup_1 = require("./NotificationGroup");
const XmlElementNames_1 = require("../Core/XmlElementNames");
/**
 * @internal Represents a collection of notification events.
 *
 * @sealed
 */
class GetStreamingEventsResults {
    /**
     * @internal Initializes a new instance of the **GetStreamingEventsResults** class.
     */
    constructor() {
        /**
         * Collection of notification events.
         */
        this.events = [];
    }
    /**
     * @internal Gets the notification collection.
     *
     * @value	The notification collection.
     */
    get Notifications() { return this.events; }
    /**
     * Loads the events from XML.
     *
     * @param   {any[]}               	jsEventsArray         	The json events array.
     * @param   {string}     		  	xmlElementName			Name of the element.
     * @param   {NotificationGroup}		notifications			Collection of notifications.
     * @param   {ExchangeService}     	service               	The service.
     */
    LoadNotificationEventFromXmlJsObject(jsEventsArray, eventElementName, notifications, service) {
        for (let jsEvent of jsEventsArray) {
            let eventType = GetEventsResults_1.GetEventsResults.XmlElementNameToEventTypeMap.get(eventElementName);
            if (eventType == EventType_1.EventType.Status) {
                continue;
            }
            let timeStamp = service.ConvertUniversalDateTimeStringToLocalDateTime(jsEvent[XmlElementNames_1.XmlElementNames.TimeStamp]);
            let notificationEvent;
            if (jsEvent[XmlElementNames_1.XmlElementNames.FolderId]) {
                notificationEvent = new FolderEvent_1.FolderEvent(eventType, timeStamp);
            }
            else {
                notificationEvent = new ItemEvent_1.ItemEvent(eventType, timeStamp);
            }
            notificationEvent.LoadFromXmlJsObject(jsEvent, service);
            notifications.Events.push(notificationEvent);
        }
    }
    /**
     * @internal Loads from XML.
     *
     * @param   {any}                 eventsResponse         The events response Object converted from XML.
     * @param   {ExchangeService}     service                The service.
     */
    LoadFromXmlJsObject(eventsResponse, service) {
        let streamingNotifications = EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(eventsResponse, XmlElementNames_1.XmlElementNames.Notification);
        for (let streamingNotification of streamingNotifications) {
            let notifications = new NotificationGroup_1.NotificationGroup();
            notifications.SubscriptionId = streamingNotification[XmlElementNames_1.XmlElementNames.SubscriptionId];
            notifications.Events = [];
            this.events.push(notifications);
            for (let key in streamingNotification) {
                switch (key) {
                    default:
                        if (GetEventsResults_1.GetEventsResults.XmlElementNameToEventTypeMap.containsKey(key)) {
                            this.LoadNotificationEventFromXmlJsObject(EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(streamingNotification, key), key, notifications, service);
                        }
                        break;
                }
            }
        }
    }
}
exports.GetStreamingEventsResults = GetStreamingEventsResults;
//# sourceMappingURL=GetStreamingEventsResults.js.map