"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Time = void 0;
const ArgumentException_1 = require("../Exceptions/ArgumentException");
const ExtensionMethods_1 = require("../ExtensionMethods");
const Strings_1 = require("../Strings");
/**
 * @internal Represents a time.
 * @sealed
 */
class Time {
    constructor(minutesOrDateTimeOrHours, minutes, seconds) {
        this.hours = 0;
        this.minutes = 0;
        this.seconds = 0;
        if (arguments.length === 1) {
            if (typeof minutesOrDateTimeOrHours === 'number') {
                if (minutes < 0 || minutes >= 1440) {
                    throw new ArgumentException_1.ArgumentException(Strings_1.Strings.MinutesMustBeBetween0And1439, "minutes");
                }
                this.Hours = minutesOrDateTimeOrHours / 60;
                this.Minutes = minutesOrDateTimeOrHours % 60;
                this.Seconds = 0;
            }
            else {
                this.Hours = minutesOrDateTimeOrHours.Hour;
                this.Minutes = minutesOrDateTimeOrHours.Minute;
                this.Seconds = minutesOrDateTimeOrHours.Second;
            }
        }
        if (arguments.length === 3) {
            this.Hours = minutesOrDateTimeOrHours;
            this.Minutes = minutes;
            this.Seconds = seconds;
        }
    }
    /**
     * @internal Gets or sets the hours.
     */
    get Hours() {
        return this.hours;
    }
    set Hours(value) {
        if (value >= 0 && value < 24) {
            this.hours = value;
            return;
        }
        throw new ArgumentException_1.ArgumentException(Strings_1.Strings.HourMustBeBetween0And23);
    }
    /**
     * @internal Gets or sets the minutes.
     */
    get Minutes() {
        return this.minutes;
    }
    set Minutes(value) {
        if (value >= 0 && value < 60) {
            this.minutes = value;
            return;
        }
        throw new ArgumentException_1.ArgumentException(Strings_1.Strings.MinuteMustBeBetween0And59);
    }
    /**
     * @internal Gets or sets the seconds.
     */
    get Seconds() {
        return this.seconds;
    }
    set Seconds(value) {
        if (value >= 0 && value < 60) {
            this.seconds = value;
            return;
        }
        throw new ArgumentException_1.ArgumentException(Strings_1.Strings.SecondMustBeBetween0And59);
    }
    /**
     * @internal Converts the time into a number of minutes since 12:00AM.
     *
     * @return  {number}      The number of minutes since 12:00AM the time represents.
     */
    ConvertToMinutes() {
        return this.Minutes + (this.Hours * 60);
    }
    /**
     * @internal Convert Time to XML Schema time.
     *
     * @return  {string}      String in XML Schema time format.
     */
    ToXSTime() {
        return ExtensionMethods_1.StringHelper.Format("{0:00}:{1:00}:{2:00}", this.Hours, this.Minutes, this.Seconds);
    }
}
exports.Time = Time;
//# sourceMappingURL=Time.js.map