"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SoapFaultDetails = void 0;
const ExtensionMethods_1 = require("../ExtensionMethods");
const AltDictionary_1 = require("../AltDictionary");
const EwsLogging_1 = require("../Core/EwsLogging");
const EwsServiceJsonReader_1 = require("../Core/EwsServiceJsonReader");
const Exception_1 = require("../Exceptions/Exception");
const ServiceError_1 = require("../Enumerations/ServiceError");
const XmlAttributeNames_1 = require("../Core/XmlAttributeNames");
const XmlElementNames_1 = require("../Core/XmlElementNames");
/**
 * Represents SoapFault details.
 *
 * /remarks/    ews-javascript-api -> removing internal modifier to, this class will be used to pass on to error delegate of promise instead of Exceptions.
 *
 * /remarks/    ews-javascript-api -> 0.9 - Extending from Error object to avoid BlueBird errors when promise is rejected without and Error object
 * /remarks/    ews-javascript-api -> 0.9 - extending from Exception which looks like Error to BlueBird. can not extend from Error. Typescript 1.8+ can not extend builtin objects property, it swallows inheriting properties see  https://github.com/Microsoft/TypeScript/wiki/Breaking-Changes#extending-built-ins-like-error-array-and-map-may-no-longer-work
 */
class SoapFaultDetails extends Exception_1.Exception {
    constructor() {
        super(...arguments);
        this.faultCode = null;
        this.faultString = null;
        this.faultActor = null;
        /**
         * Response code returned by EWS requests.
         * Default to InternalServerError.
         */
        this.responseCode = ServiceError_1.ServiceError.ErrorInternalServerError;
        /**
         * This is returned by Availability requests.
         */
        this.errorCode = ServiceError_1.ServiceError.NoError;
        /**
         * This is returned by UM requests. It's the name of the exception that was raised.
         */
        this.exceptionType = null;
        /**
         * When a schema validation error is returned, this is the line number in the request where the error occurred.
         */
        this.lineNumber = 0;
        /**
         * When a schema validation error is returned, this is the offset into the line of the request where the error occurred.
         */
        this.positionWithinLine = 0;
        /**
         * Dictionary of key/value pairs from the MessageXml node in the fault. Usually empty but there are a few cases where SOAP faults may include MessageXml details (e.g. CASOverBudgetException includes BackoffTime value).
         */
        this.errorDetails = new AltDictionary_1.DictionaryWithStringKey();
        /**
         * ews-javascript-api specific: HTTP status code
         */
        this.HttpStatusCode = 200;
    }
    /**
     * @internal Gets or sets the SOAP fault code.
     *
     * @value   The SOAP fault code.
     */
    get FaultCode() {
        return this.faultCode;
    }
    set FaultCode(value) {
        this.faultCode = value;
    }
    /**
     * @internal Gets or sets the SOAP fault string.
     *
     * @value   The fault string.
     */
    get FaultString() {
        return this.faultString;
    }
    set FaultString(value) {
        this.faultString = value;
    }
    /**
     * @internal Gets or sets the SOAP fault actor.
     *
     * @value   The fault actor.
     */
    get FaultActor() {
        return this.faultActor;
    }
    set FaultActor(value) {
        this.faultActor = value;
    }
    /**
     * @internal Gets or sets the response code.
     *
     * @value   The response code.
     */
    get ResponseCode() {
        return this.responseCode;
    }
    set ResponseCode(value) {
        this.responseCode = value;
    }
    /**
     * @internal Gets or sets the message.
     *
     * @value   The message.
     */
    get Message() {
        return this.message;
    }
    set Message(value) {
        this.message = value;
    }
    /**
     * @internal Gets or sets the error code.
     *
     * @value   The error code.
     */
    get ErrorCode() {
        return this.errorCode;
    }
    set ErrorCode(value) {
        this.errorCode = value;
    }
    /**
     * @internal Gets or sets the type of the exception.
     *
     * @value   The type of the exception.
     */
    get ExceptionType() {
        return this.exceptionType;
    }
    set ExceptionType(value) {
        this.exceptionType = value;
    }
    /**
     * @internal Gets or sets the line number.
     *
     * @value   The line number.
     */
    get LineNumber() {
        return this.lineNumber;
    }
    set LineNumber(value) {
        this.lineNumber = value;
    }
    /**
     * @internal Gets or sets the position within line.
     *
     * @value   The position within line.
     */
    get PositionWithinLine() {
        return this.positionWithinLine;
    }
    set PositionWithinLine(value) {
        this.positionWithinLine = value;
    }
    /**
     * @internal Gets or sets the error details dictionary.
     *
     * @value   The error details dictionary.
     */
    get ErrorDetails() {
        return this.errorDetails;
    }
    set ErrorDetails(value) {
        this.errorDetails = value;
    }
    // /**
    //  * @private Initializes a new instance of the **SoapFaultDetails** class.
    //  */
    // // constructor() {
    // //     super();
    // // }
    /**
     * @internal Parses the soap:Fault content.
     *
     * @param   {any}   jsObject        The converted Xml JsObject.
     * @return  {SoapFaultDetails}      SOAP fault details.
     */
    static Parse(jsObject) {
        var soapFaultDetails = new SoapFaultDetails();
        for (let key in jsObject) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.SOAPFaultCodeElementName:
                    soapFaultDetails.FaultCode = jsObject[key];
                    break;
                case XmlElementNames_1.XmlElementNames.SOAPFaultStringElementName:
                    soapFaultDetails.FaultString = jsObject[key];
                    break;
                case XmlElementNames_1.XmlElementNames.SOAPFaultActorElementName:
                    soapFaultDetails.FaultActor = jsObject[key];
                    break;
                case XmlElementNames_1.XmlElementNames.SOAPDetailElementName:
                    soapFaultDetails.ParseDetailNode(jsObject[key]);
                    break;
                default:
                    break;
            }
        }
        return soapFaultDetails;
    }
    /**
     * Parses the detail node.
     *
     * @param   {any}   jsObject   The detail node.
     */
    ParseDetailNode(jsObject) {
        for (let key in jsObject) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.EwsResponseCodeElementName:
                    // ServiceError couldn't be mapped to enum value, treat as an ISE
                    this.ResponseCode = ServiceError_1.ServiceError[jsObject[key]] || ServiceError_1.ServiceError.ErrorInternalServerError;
                    ;
                    break;
                case XmlElementNames_1.XmlElementNames.EwsMessageElementName:
                    this.Message = jsObject[key];
                    break;
                case XmlElementNames_1.XmlElementNames.EwsLineElementName:
                    this.LineNumber = ExtensionMethods_1.Convert.toNumber(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.EwsPositionElementName:
                    this.PositionWithinLine = ExtensionMethods_1.Convert.toNumber(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.EwsErrorCodeElementName:
                    // ServiceError couldn't be mapped to enum value, treat as an ISE
                    this.ErrorCode = ServiceError_1.ServiceError[jsObject[key]] || ServiceError_1.ServiceError.ErrorInternalServerError;
                    break;
                case XmlElementNames_1.XmlElementNames.EwsExceptionTypeElementName:
                    this.ExceptionType = jsObject[key];
                    break;
                case XmlElementNames_1.XmlElementNames.MessageXml:
                    this.ParseMessageXml(jsObject[key]);
                    break;
                default:
                    // Ignore any other details
                    break;
            }
        }
    }
    /**
     * Parses the message XML.
     *
     * @param   {any}   jsObject   The message Xml object.
     */
    ParseMessageXml(jsObject) {
        for (var key in jsObject) {
            if (key.indexOf("__") === 0) {
                continue;
            }
            switch (key) {
                case XmlElementNames_1.XmlElementNames.Value:
                    let values = EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(jsObject, key);
                    values.forEach((value, index) => {
                        let name = value[XmlAttributeNames_1.XmlAttributeNames.Name];
                        if (name) {
                            if (this.ErrorDetails.containsKey(name)) {
                                name = name + " - " + (index + 1);
                            }
                            this.errorDetails.Add(name, value[key]);
                        }
                    });
                case XmlElementNames_1.XmlElementNames.EwsLineElementName:
                case "LineNumber":
                    this.LineNumber = ExtensionMethods_1.Convert.toNumber(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.EwsPositionElementName:
                case "LinePosition":
                    this.PositionWithinLine = ExtensionMethods_1.Convert.toNumber(jsObject[key]);
                    break;
                default:
                    if (typeof jsObject[key] === "string") {
                        this.errorDetails.addUpdate(key, jsObject[key]);
                    }
                    EwsLogging_1.EwsLogging.Assert(false, "SoapFaultDetails.ParseMessageXml", "Element: " + key + " - Please report example of this operation to ews-javascript-api repo to improve SoapFault parsing");
                    break;
            }
        }
    }
}
exports.SoapFaultDetails = SoapFaultDetails;
//# sourceMappingURL=SoapFaultDetails.js.map