"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceResponseException = void 0;
const ServiceError_1 = require("../Enumerations/ServiceError");
const ExtensionMethods_1 = require("../ExtensionMethods");
const Strings_1 = require("../Strings");
const ServiceRemoteException_1 = require("./ServiceRemoteException");
/**
 * Represents a remote service exception that has a single response.
 */
class ServiceResponseException extends ServiceRemoteException_1.ServiceRemoteException {
    /**
     * @internal Initializes a new instance of the **ServiceResponseException** class.
     *
     * @param   {ServiceResponse}   response   The ServiceResponse when service operation failed remotely.
     */
    constructor(response) {
        super(response && response.ErrorMessage ? response.ErrorMessage : "");
        this.response = response;
    }
    /**
     * Gets the ServiceResponse for the exception.
     */
    get Response() {
        return this.response;
    }
    /**
     * Gets the service error code.
     */
    get ErrorCode() {
        return this.response ? this.response.ErrorCode : null;
    }
    /**
     * Gets a message that describes the current exception.
     *
     * @returns The error message that explains the reason for the exception.
     */
    get Message() {
        // Special case for Internal Server Error. If the server returned
        // stack trace information, include it in the exception message.
        if (this.Response.ErrorCode == ServiceError_1.ServiceError.ErrorInternalServerError) {
            var exceptionClass = this.Response.ErrorDetails.get(ServiceResponseException.ExceptionClassKey);
            var exceptionMessage = this.Response.ErrorDetails.get(ServiceResponseException.ExceptionMessageKey);
            var stackTrace = this.Response.ErrorDetails.get(ServiceResponseException.StackTraceKey);
            if (!ExtensionMethods_1.StringHelper.IsNullOrEmpty(exceptionClass) && !ExtensionMethods_1.StringHelper.IsNullOrEmpty(exceptionMessage) && !ExtensionMethods_1.StringHelper.IsNullOrEmpty(stackTrace)) {
                return ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.ServerErrorAndStackTraceDetails, this.Response.ErrorMessage, exceptionClass, exceptionMessage, stackTrace);
            }
        }
        return this.Response.ErrorMessage;
    }
}
exports.ServiceResponseException = ServiceResponseException;
/**
 * Error details Value keys
 */
ServiceResponseException.ExceptionClassKey = "ExceptionClass";
ServiceResponseException.ExceptionMessageKey = "ExceptionMessage";
ServiceResponseException.StackTraceKey = "StackTrace";
//# sourceMappingURL=ServiceResponseException.js.map