"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.unitOfTime = exports.DateTimeStyles = exports.DateTime = exports.DateTimeKind = exports.msToEpoch = void 0;
const moment = require("moment-timezone");
const ArgumentException_1 = require("./Exceptions/ArgumentException");
const TimeSpan_1 = require("./TimeSpan");
const ticksToEpoch = 621355968000000000; //can be used when calculating ticks/ms from Windows date to unix date
exports.msToEpoch = 62135596800000;
const invalidDateTimeMessage = {
    "years": "year is less than 1 or greater than 9999.",
    "months": "month is less than 1 or greater than 12.",
    "days": "day is less than 1 or greater than the number of days in month.",
    "hours": "hour is less than 0 or greater than 23.",
    "minutes": "minute is less than 0 or greater than 59.",
    "seconds": "second is less than 0 or greater than 59.",
    "milliseconds": "millisecond is less than 0 or greater than 999."
};
var DateTimeKind;
(function (DateTimeKind) {
    DateTimeKind[DateTimeKind["Unspecified"] = 0] = "Unspecified";
    DateTimeKind[DateTimeKind["Utc"] = 1] = "Utc";
    DateTimeKind[DateTimeKind["Local"] = 2] = "Local";
})(DateTimeKind = exports.DateTimeKind || (exports.DateTimeKind = {}));
/**
 * DateTime - basic date time based on moment.js
 */
class DateTime {
    constructor(msOrDateOrMomentOrYear, monthOrKind, day, hour, minute, second, millisecond, kind = DateTimeKind.Unspecified) {
        this.kind = DateTimeKind.Unspecified;
        this.originalDateInput = null;
        let argsLength = arguments.length;
        let momentdate = moment();
        this.kind = kind;
        if (argsLength === 1) {
            if (msOrDateOrMomentOrYear instanceof DateTime) {
                momentdate = msOrDateOrMomentOrYear.MomentDate.clone();
                this.kind = msOrDateOrMomentOrYear.kind;
            }
            else {
                momentdate = moment(msOrDateOrMomentOrYear);
                this.originalDateInput = msOrDateOrMomentOrYear;
            }
        }
        else if (argsLength === 2) {
            if (monthOrKind === DateTimeKind.Utc && !(msOrDateOrMomentOrYear instanceof moment)) {
                momentdate = moment.utc(msOrDateOrMomentOrYear);
            }
            else {
                momentdate = moment(msOrDateOrMomentOrYear);
            }
            this.kind = monthOrKind;
            if (this.kind === DateTimeKind.Unspecified && !(msOrDateOrMomentOrYear instanceof moment)) {
                this.originalDateInput = msOrDateOrMomentOrYear;
            }
        }
        else {
            let momentInput = {};
            if (argsLength >= 3) {
                momentInput.year = msOrDateOrMomentOrYear;
                momentInput.month = monthOrKind - 1;
                momentInput.day = day;
            }
            if (argsLength >= 6) {
                momentInput.hour = hour;
                momentInput.minute = minute;
                momentInput.second = second;
            }
            if (argsLength >= 7) {
                momentInput.millisecond = millisecond;
            }
            momentdate = moment(momentInput);
        }
        if (momentdate && !momentdate.isValid()) {
            let invalid = momentdate.invalidAt();
            throw new ArgumentException_1.ArgumentOutOfRangeException(momentValidity[invalid], invalidDateTimeMessage[momentValidity[invalid]]);
        }
        // if (momentdate.isUtc()) {
        //     this.kind = DateTimeKind.Utc
        // }
        // else if (momentdate.isLocal()) {
        //     this.kind = DateTimeKind.Local;
        // }
        this.getMomentDate = () => momentdate;
        this.setMomentDate = (value) => momentdate = value;
    }
    get MomentDate() { return this.momentDate; }
    get currentUtcOffset() { return this.momentDate.utcOffset(); }
    get momentDate() { return this.getMomentDate(); }
    static get Now() {
        return new DateTime(moment());
    }
    static get UtcNow() {
        return new DateTime(moment.utc());
    }
    get TotalMilliSeconds() {
        return this.momentDate.valueOf();
    }
    Add(quantity, unit = "ms") {
        if (typeof quantity !== 'number') {
            quantity = quantity.TotalMilliseconds;
            unit = "ms";
        }
        var date = moment(this.momentDate);
        date.add(quantity, unit);
        return new DateTime(date);
    }
    static Compare(x, y) {
        var diff = x.momentDate.diff(y.momentDate);
        if (diff === 0)
            return 0;
        if (diff < 0)
            return -1;
        return 1;
    }
    CompareTo(toDate) {
        return DateTime.Compare(this, toDate);
    }
    Difference(toDate) {
        return new TimeSpan_1.TimeSpan(toDate.momentDate.diff(this.momentDate));
    }
    Format(formatting) {
        return this.momentDate.format(formatting);
    }
    static getKindfromMoment(m) {
        if (m.isUTC()) {
            return DateTimeKind.Utc;
        }
        if (m.isLocal()) {
            return DateTimeKind.Local;
        }
        return DateTimeKind.Unspecified;
    }
    static Parse(value, kind = DateTimeKind.Unspecified) {
        let mdate = moment(value);
        let tempDate = null;
        if (mdate.isValid()) {
            switch (kind) {
                case DateTimeKind.Local:
                    tempDate = new DateTime(mdate.local());
                    tempDate.kind = kind;
                    return tempDate;
                case DateTimeKind.Utc:
                    tempDate = new DateTime(moment.utc(value));
                    tempDate.kind = kind;
                    return tempDate;
                default:
                    tempDate = new DateTime(mdate);
                    tempDate.originalDateInput = value;
                    tempDate.kind = kind;
                    return tempDate;
            }
        }
        else {
            throw new ArgumentException_1.ArgumentException("invalid date value");
        }
    }
    ToISOString() {
        return this.momentDate.toISOString();
    }
    toString() {
        return this.momentDate.toString();
    }
    utcOffset(value) {
        this.momentDate.utcOffset(value);
    }
    static DateimeStringToTimeZone(dtStr, zoneStr) {
        return new DateTime(moment.tz(dtStr, zoneStr));
    }
    static DateTimeToXSDateTime(dateTime) {
        var format = 'YYYY-MM-DDTHH:mm:ss.SSSZ'; //using moment format for c#->"yyyy-MM-ddTHH:mm:ss.fff";
        // switch (dateTime.Kind) {
        // 	case DateTimeKind.Utc:
        // 		format += "Z";
        // 		break;
        // 	case DateTimeKind.Local:
        // 		format += "zzz";
        // 		break;
        // 	default:
        // 		break;
        // }
        // Depending on the current culture, DateTime formatter will replace ':' with 
        // the DateTimeFormatInfo.TimeSeparator property which may not be ':'. Force the proper string
        // to be used by using the InvariantCulture.
        return dateTime.Format(format); //, CultureInfo.InvariantCulture);
    }
    static DateTimeToXSDate(date) {
        var format = 'YYYY-MM-DDZ'; //using moment format for c#->"yyyy-MM-dd";
        // switch (date.Kind) {
        // 	case DateTimeKind.Utc:
        // 		format = "yyyy-MM-ddZ";
        // 		break;
        // 	case DateTimeKind.Unspecified:
        // 		format = "yyyy-MM-dd";
        // 		break;
        // 	default: // DateTimeKind.Local is remaining
        // 		format = "yyyy-MM-ddzzz";
        // 		break;
        // }
        // Depending on the current culture, DateTime formatter will 
        // translate dates from one culture to another (e.g. Gregorian to Lunar).  The server
        // however, considers all dates to be in Gregorian, so using the InvariantCulture will
        // ensure this.
        return date.Format(format); //, CultureInfo.InvariantCulture);
    }
    /* c# DateTime properties */
    get Date() {
        if (this === DateTime.MaxValue || this === DateTime.MinValue) {
            return new DateTime(this.momentDate.utc().format("YYYY-MM-DD"));
        }
        return new DateTime(this.momentDate.format("YYYY-MM-DD"));
    }
    get Day() {
        return this.momentDate.date();
    }
    get DayOfWeek() {
        return this.momentDate.day();
    }
    get DayOfYear() {
        return this.momentDate.dayOfYear();
    }
    get Hour() {
        return this.momentDate.hour();
    }
    get Kind() {
        return this.kind;
    }
    get Millisecond() {
        return this.momentDate.millisecond();
    }
    get Minute() {
        return this.momentDate.minute();
    }
    get Month() {
        return this.momentDate.month() + 1;
    }
    get Second() {
        return this.momentDate.second();
    }
    //  public get Ticks(): {
    //      return this.
    //  }
    get TimeOfDay() {
        return TimeSpan_1.TimeSpan.FromMilliseconds(this.momentDate.millisecond());
    }
    get Today() {
        return new DateTime(moment(this.momentDate.format("LL"), "LL"));
    }
    get Year() {
        return this.momentDate.year();
    }
    /* c# DateTime Methods */
    //CompareTo
    AddDays(days) {
        return this.Add(days, exports.unitOfTime.days);
    }
    AddHours(hours) {
        return this.Add(hours, exports.unitOfTime.hours);
    }
    AddMilliseconds(ms) {
        return this.Add(ms, exports.unitOfTime.ms);
    }
    AddMinutes(minutes) {
        return this.Add(minutes, exports.unitOfTime.minutes);
    }
    AddMonths(months) {
        return this.Add(months, exports.unitOfTime.months);
    }
    AddSeconds(seconds) {
        return this.Add(seconds, exports.unitOfTime.seconds);
    }
    // public AddTicks(ticks: number): DateTime {
    //     return this.Add(ticks, unitOfTime.);
    // }
    AddYears(years) {
        return this.Add(years, exports.unitOfTime.years);
    }
    static DaysInMonth(year, month) {
        if (month < 1 || month > 12)
            throw new ArgumentException_1.ArgumentOutOfRangeException("month", invalidDateTimeMessage["months"]);
        // IsLeapYear checks the year argument
        let days = DateTime.IsLeapYear(year) ? DateTime.DaysToMonth366 : DateTime.DaysToMonth365;
        return days[month] - days[month - 1];
    }
    Equals(value) {
        if (value instanceof DateTime) {
            return value.TotalMilliSeconds === this.TotalMilliSeconds;
        }
        return false;
    }
    static Equals(t1, t2) {
        return t1.TotalMilliSeconds === t2.TotalMilliSeconds;
    }
    // FromBinary
    // FromFileTime
    // FromFileTimeUtc
    // FromOADate
    // GetHashCode
    IsDaylightSavingTime() {
        return this.momentDate.isDST();
    }
    /**
     *  Checks whether a given year is a leap year. This method returns true if year is a leap year, or false if not.
     * @param {number}  year
     */
    static IsLeapYear(year) {
        if (year < 1 || year > 9999) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("year", invalidDateTimeMessage["years"]);
        }
        return year % 4 == 0 && (year % 100 != 0 || year % 400 == 0);
    }
    // ParseExact
    static SpecifyKind(value, kind) {
        return new DateTime(value.TotalMilliSeconds, kind);
    }
    Subtract(dateTime) {
        if (dateTime instanceof DateTime) {
            return new TimeSpan_1.TimeSpan(this.TotalMilliSeconds - dateTime.TotalMilliSeconds);
        }
        else {
            return new DateTime(this.TotalMilliSeconds - dateTime.TotalMilliseconds);
        }
    }
    // ToBinary
    // ToFileTime
    // ToFileTimeUtc
    ToLocalTime() {
        return new DateTime(this.momentDate.local());
    }
    ToLongDateString() {
        return this.momentDate.format("dddd, MMMM D, YYYY");
    }
    ToLongTimeString() {
        return this.momentDate.format("LTS");
    }
    // ToOADate
    ToShortDateString() {
        return this.MomentDate.format("l");
    }
    ToShortTimeString() {
        return this.MomentDate.format("LT");
    }
    ToString() {
        return this.toString();
    }
    ToUniversalTime() {
        return new DateTime(this.MomentDate.utc());
    }
    static TryParse(s, outDate) {
        try {
            outDate.outValue = DateTime.Parse(s);
            outDate.outValue.kind = this.getKindfromMoment(outDate.outValue.momentDate);
            return true;
        }
        catch (error) {
            outDate.exception = error;
        }
        return false;
    }
    // TryParseExact
    valueOf() {
        return this.TotalMilliSeconds;
    }
}
exports.DateTime = DateTime;
DateTime.DaysToMonth365 = [0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365];
DateTime.DaysToMonth366 = [0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366];
DateTime.MinValue = new DateTime('0001-01-01T00:00:00+00:00');
DateTime.MaxValue = new DateTime("9999-12-31T23:59:59.9999999+00:00");
//
// Summary:
//     Defines the formatting options that customize string parsing for some date and
//     time parsing methods.
var DateTimeStyles;
(function (DateTimeStyles) {
    //
    // Summary:
    //     Default formatting options must be used. This value represents the default style
    //     for the System.DateTime.Parse(System.String), System.DateTime.ParseExact(System.String,System.String,System.IFormatProvider),
    //     and System.DateTime.TryParse(System.String,System.DateTime@) methods.
    DateTimeStyles[DateTimeStyles["None"] = 0] = "None";
    //
    // Summary:
    //     Leading white-space characters must be ignored during parsing, except if they
    //     occur in the System.Globalization.DateTimeFormatInfo format patterns.
    DateTimeStyles[DateTimeStyles["AllowLeadingWhite"] = 1] = "AllowLeadingWhite";
    //
    // Summary:
    //     Trailing white-space characters must be ignored during parsing, except if they
    //     occur in the System.Globalization.DateTimeFormatInfo format patterns.
    DateTimeStyles[DateTimeStyles["AllowTrailingWhite"] = 2] = "AllowTrailingWhite";
    /**
     * Summary:
     *	    Extra white-space characters in the middle of the string must be ignored during
     *	    parsing, except if they occur in the System.Globalization.DateTimeFormatInfo
     *	    format patterns.
     */
    DateTimeStyles[DateTimeStyles["AllowInnerWhite"] = 4] = "AllowInnerWhite";
    /**
     * 	Summary:
     *	    Extra white-space characters anywhere in the string must be ignored during parsing,
     *	    except if they occur in the System.Globalization.DateTimeFormatInfo format patterns.
     *	    This value is a combination of the System.Globalization.DateTimeStyles.AllowLeadingWhite,
     *	    System.Globalization.DateTimeStyles.AllowTrailingWhite, and System.Globalization.DateTimeStyles.AllowInnerWhite
     *	    values.
     */
    DateTimeStyles[DateTimeStyles["AllowWhiteSpaces"] = 7] = "AllowWhiteSpaces";
    //
    // Summary:
    //     If the parsed string contains only the time and not the date, the parsing methods
    //     assume the Gregorian date with year = 1, month = 1, and day = 1. If this value
    //     is not used, the current date is assumed.
    DateTimeStyles[DateTimeStyles["NoCurrentDateDefault"] = 8] = "NoCurrentDateDefault";
    //
    // Summary:
    //     Date and time are returned as a Coordinated Universal Time (UTC). If the input
    //     string denotes a local time, through a time zone specifier or System.Globalization.DateTimeStyles.AssumeLocal,
    //     the date and time are converted from the local time to UTC. If the input string
    //     denotes a UTC time, through a time zone specifier or System.Globalization.DateTimeStyles.AssumeUniversal,
    //     no conversion occurs. If the input string does not denote a local or UTC time,
    //     no conversion occurs and the resulting System.DateTime.Kind property is System.DateTimeKind.Unspecified.
    DateTimeStyles[DateTimeStyles["AdjustToUniversal"] = 16] = "AdjustToUniversal";
    //
    // Summary:
    //     If no time zone is specified in the parsed string, the string is assumed to denote
    //     a local time.
    DateTimeStyles[DateTimeStyles["AssumeLocal"] = 32] = "AssumeLocal";
    //
    // Summary:
    //     If no time zone is specified in the parsed string, the string is assumed to denote
    //     a UTC.
    DateTimeStyles[DateTimeStyles["AssumeUniversal"] = 64] = "AssumeUniversal";
    //
    // Summary:
    //     The System.DateTimeKind field of a date is preserved when a System.DateTime object
    //     is converted to a string using the "o" or "r" standard format specifier, and
    //     the string is then converted back to a System.DateTime object.
    DateTimeStyles[DateTimeStyles["RoundtripKind"] = 128] = "RoundtripKind";
})(DateTimeStyles = exports.DateTimeStyles || (exports.DateTimeStyles = {}));
exports.unitOfTime = {
    "year": "year",
    "years": "years",
    "y": "y",
    "month": "month",
    "months": "months",
    "M": "M",
    "week": "week",
    "weeks": "weeks",
    "w": "w",
    "day": "day",
    "days": "days",
    "d": "d",
    "hour": "hour",
    "hours": "hours",
    "h": "h",
    "minute": "minute",
    "minutes": "minutes",
    "m": "m",
    "second": "second",
    "seconds": "seconds",
    "s": "s",
    "millisecond": "millisecond",
    "milliseconds": "milliseconds",
    "ms": "ms",
};
var momentValidity;
(function (momentValidity) {
    momentValidity[momentValidity["years"] = 0] = "years";
    momentValidity[momentValidity["months"] = 1] = "months";
    momentValidity[momentValidity["days"] = 2] = "days";
    momentValidity[momentValidity["hours"] = 3] = "hours";
    momentValidity[momentValidity["minutes"] = 4] = "minutes";
    momentValidity[momentValidity["seconds"] = 5] = "seconds";
    momentValidity[momentValidity["milliseconds"] = 6] = "milliseconds";
})(momentValidity || (momentValidity = {}));
//# sourceMappingURL=DateTime.js.map