"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Item = void 0;
const ExtensionMethods_1 = require("../../../ExtensionMethods");
const EwsLogging_1 = require("../../EwsLogging");
const ExchangeVersion_1 = require("../../../Enumerations/ExchangeVersion");
const FolderId_1 = require("../../../ComplexProperties/FolderId");
const MessageDisposition_1 = require("../../../Enumerations/MessageDisposition");
const PropertySet_1 = require("../../PropertySet");
const Schemas_1 = require("../Schemas/Schemas");
const ServiceErrorHandling_1 = require("../../../Enumerations/ServiceErrorHandling");
const ServiceObjectSchema_1 = require("../Schemas/ServiceObjectSchema");
const ServiceVersionException_1 = require("../../../Exceptions/ServiceVersionException");
const Strings_1 = require("../../../Strings");
const TypeContainer_1 = require("../../../TypeContainer");
const XmlElementNames_1 = require("../../XmlElementNames");
const ServiceObject_1 = require("../ServiceObject");
/**
 * Represents a generic **Item**. Properties available on items are defined in the *ItemSchema* class.
 *
 */
class Item extends ServiceObject_1.ServiceObject {
    constructor(obj) {
        super(obj instanceof TypeContainer_1.TypeContainer.ExchangeService ? obj : obj instanceof TypeContainer_1.TypeContainer.ItemAttachment ? obj.Service : null); //info: cannot check instanceof to avoid circular dependency in js. TypeContainer is workaround 
        this.parentAttachment = null;
        if (obj instanceof TypeContainer_1.TypeContainer.ItemAttachment) {
            var parentAttachment = obj;
            EwsLogging_1.EwsLogging.Assert(parentAttachment != null, "Item.ctor", "parentAttachment is null");
            this.parentAttachment = parentAttachment;
        }
    }
    /** required to check [Attachable] attribute, AttachmentCollection.AddItemAttachment<TItem>() checks for non inherited [Attachable] attribute. */
    static get Attachable() { return this.name === "Item"; }
    ;
    /**
     * @internal Gets the parent attachment of this item.
     *
     */
    get ParentAttachment() {
        return this.parentAttachment;
    }
    /**
     * @internal Gets Id of the root item for this item.
     *
     */
    get RootItemId() {
        if (this.IsAttachment && this.ParentAttachment.Owner !== null) {
            return this.ParentAttachment.Owner.RootItemId;
        }
        return this.Id;
    }
    /**
     * Gets a value indicating whether the item is an attachment.
     *
     */
    get IsAttachment() { return this.parentAttachment != null && typeof this.parentAttachment !== 'undefined'; }
    /**
     * Gets a value indicating whether this object is a real store item, or if it's a local object that has yet to be saved.
     *
     */
    get IsNew() {
        // Item attachments don't have an Id, need to check whether the
        // parentAttachment is new or not.
        if (this.IsAttachment) {
            return this.ParentAttachment.IsNew;
        }
        else {
            var id = this.GetId();
            return id == null ? true : !id.IsValid;
        }
    }
    /**
     * Gets the Id of this item.
     *
     */
    get Id() { return this.PropertyBag._getItem(this.GetIdPropertyDefinition()); }
    /**
     * Get or sets the MIME content of this item.
     *
     */
    get MimeContent() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.MimeContent);
    }
    set MimeContent(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.MimeContent, value);
    }
    /**
     * Gets the Id of the parent folder of this item.
     *
     */
    get ParentFolderId() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.ParentFolderId);
    }
    /**
     * Gets or sets the sensitivity of this item.
     *
     */
    get Sensitivity() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.Sensitivity);
    }
    set Sensitivity(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.Sensitivity, value);
    }
    /**
     * Gets a list of the attachments to this item.
     *
     */
    get Attachments() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.Attachments);
    }
    /**
     * Gets the time when this item was received.
     *
     */
    get DateTimeReceived() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.DateTimeReceived);
    }
    /**
     * Gets the size of this item.
     *
     */
    get Size() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.Size);
    }
    /**
     * Gets or sets the list of categories associated with this item.
     *
     */
    get Categories() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.Categories);
    }
    set Categories(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.Categories, value);
    }
    /**
     * Gets or sets the culture associated with this item.
     *
     */
    get Culture() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.Culture);
    }
    set Culture(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.Culture, value);
    }
    /**
     * Gets or sets the importance of this item.
     *
     */
    get Importance() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.Importance);
    }
    set Importance(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.Importance, value);
    }
    /**
     * Gets or sets the In-Reply-To reference of this item.
     *
     */
    get InReplyTo() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.InReplyTo);
    }
    set InReplyTo(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.InReplyTo, value);
    }
    /**
     * Gets a value indicating whether the message has been submitted to be sent.
     *
     */
    get IsSubmitted() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.IsSubmitted);
    }
    /**
     * Gets a value indicating whether this is an associated item.
     *
     */
    get IsAssociated() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.IsAssociated);
    }
    /**
     * Gets a value indicating whether the item is is a draft. An item is a draft when it has not yet been sent.
     *
     */
    get IsDraft() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.IsDraft);
    }
    /**
     * Gets a value indicating whether the item has been sent by the current authenticated user.
     *
     */
    get IsFromMe() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.IsFromMe);
    }
    /**
     * Gets a value indicating whether the item is a resend of another item.
     *
     */
    get IsResend() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.IsResend);
    }
    /**
     * Gets a value indicating whether the item has been modified since it was created.
     *
     */
    get IsUnmodified() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.IsUnmodified);
    }
    /**
     * Gets a list of Internet headers for this item.
     *
     */
    get InternetMessageHeaders() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.InternetMessageHeaders);
    }
    /**
     * Gets the date and time this item was sent.
     *
     */
    get DateTimeSent() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.DateTimeSent);
    }
    /**
     * Gets the date and time this item was created.
     *
     */
    get DateTimeCreated() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.DateTimeCreated);
    }
    /**
     * Gets a value indicating which response actions are allowed on this item. Examples of response actions are Reply and Forward.
     *
     */
    get AllowedResponseActions() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.AllowedResponseActions);
    }
    /**
     * Gets or sets the date and time when the reminder is due for this item.
     *
     */
    get ReminderDueBy() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.ReminderDueBy);
    }
    set ReminderDueBy(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.ReminderDueBy, value);
    }
    /**
     * Gets or sets a value indicating whether a reminder is set for this item.
     *
     */
    get IsReminderSet() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.IsReminderSet);
    }
    set IsReminderSet(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.IsReminderSet, value);
    }
    /**
     * Gets or sets the number of minutes before the start of this item when the reminder should be triggered.
     *
     */
    get ReminderMinutesBeforeStart() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.ReminderMinutesBeforeStart);
    }
    set ReminderMinutesBeforeStart(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.ReminderMinutesBeforeStart, value);
    }
    /**
     * Gets a text summarizing the Cc receipients of this item.
     *
     */
    get DisplayCc() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.DisplayCc);
    }
    /**
     * Gets a text summarizing the To recipients of this item.
     *
     */
    get DisplayTo() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.DisplayTo);
    }
    /**
     * Gets a value indicating whether the item has attachments.
     *
     */
    get HasAttachments() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.HasAttachments);
    }
    /**
     * Gets or sets the body of this item.
     *
     */
    get Body() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.Body);
    }
    set Body(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.Body, value);
    }
    /**
     * Gets or sets the custom class name of this item.
     *
     */
    get ItemClass() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.ItemClass);
    }
    set ItemClass(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.ItemClass, value);
    }
    /**
     * Gets or sets the subject of this item.
     *
     */
    get Subject() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.Subject);
    }
    set Subject(value) {
        this.SetSubject(value);
    }
    /**
     * Gets the query string that should be appended to the Exchange Web client URL to open this item using the appropriate read form in a web browser.
     *
     */
    get WebClientReadFormQueryString() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.WebClientReadFormQueryString);
    }
    /**
     * Gets the query string that should be appended to the Exchange Web client URL to open this item using the appropriate edit form in a web browser.
     *
     */
    get WebClientEditFormQueryString() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.WebClientEditFormQueryString);
    }
    /**
     * Gets a list of extended properties defined on this item.
     *
     */
    get ExtendedProperties() {
        return this.PropertyBag._getItem(ServiceObjectSchema_1.ServiceObjectSchema.ExtendedProperties);
    }
    /**
     * Gets a value indicating the effective rights the current authenticated user has on this item.
     *
     */
    get EffectiveRights() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.EffectiveRights);
    }
    /**
     * Gets the name of the user who last modified this item.
     *
     */
    get LastModifiedName() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.LastModifiedName);
    }
    /**
     * Gets the date and time this item was last modified.
     *
     */
    get LastModifiedTime() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.LastModifiedTime);
    }
    /**
     * Gets the Id of the conversation this item is part of.
     *
     */
    get ConversationId() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.ConversationId);
    }
    /**
     * Gets the body part that is unique to the conversation this item is part of.
     *
     */
    get UniqueBody() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.UniqueBody);
    }
    /**
     * Gets the store entry id.
     *
     */
    get StoreEntryId() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.StoreEntryId);
    }
    /**
     * Gets the item instance key.
     *
     */
    get InstanceKey() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.InstanceKey);
    }
    /**
     * Get or set the Flag value for this item.
     *
     */
    get Flag() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.Flag);
    }
    set Flag(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.Flag, value);
    }
    /**
     * Gets the normalized body of the item.
     *
     */
    get NormalizedBody() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.NormalizedBody);
    }
    /**
     * Gets the EntityExtractionResult of the item.
     *
     */
    get EntityExtractionResult() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.EntityExtractionResult);
    }
    /**
     * Gets or sets the policy tag.
     *
     */
    get PolicyTag() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.PolicyTag);
    }
    set PolicyTag(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.PolicyTag, value);
    }
    /**
     * Gets or sets the archive tag.
     *
     */
    get ArchiveTag() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.ArchiveTag);
    }
    set ArchiveTag(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.ArchiveTag, value);
    }
    /**
     * Gets the retention date.
     *
     */
    get RetentionDate() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.RetentionDate);
    }
    /**
     * Gets the item Preview.
     *
     */
    get Preview() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.Preview);
    }
    /**
     * Gets the text body of the item.
     *
     */
    get TextBody() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.TextBody);
    }
    /**
     * Gets the icon index.
     *
     */
    get IconIndex() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ItemSchema.IconIndex);
    }
    /**
     * @internal Gets the default setting for how to treat affected task occurrences on Delete.
     * Subclasses will override this for different default behavior.
     *
     */
    get DefaultAffectedTaskOccurrences() {
        return null;
    }
    /**
     * @internal Gets the default setting for sending cancellations on Delete.
     * Subclasses will override this for different default behavior.
     *
     */
    get DefaultSendCancellationsMode() {
        return null;
    }
    /**
     * @internal Gets the default settings for sending invitations on Save.
     * Subclasses will override this for different default behavior.
     *
     */
    get DefaultSendInvitationsMode() {
        return null;
    }
    /**
     * @internal Gets the default settings for sending invitations or cancellations on Update.
     * Subclasses will override this for different default behavior.
     *
     */
    get DefaultSendInvitationsOrCancellationsMode() {
        return null;
    }
    static Bind(service, id, propertySet = PropertySet_1.PropertySet.FirstClassProperties) {
        return service.BindToItem(id, propertySet, Item);
    }
    Copy(destinationFolderIdOrName) {
        this.ThrowIfThisIsNew();
        this.ThrowIfThisIsAttachment();
        var folderId = null;
        if (destinationFolderIdOrName instanceof FolderId_1.FolderId) {
            folderId = destinationFolderIdOrName;
        }
        else {
            folderId = new FolderId_1.FolderId(destinationFolderIdOrName);
        }
        //todo: EwsUtilities.ValidateParam(destinationFolderId, "destinationFolderId");
        return this.Service.CopyItem(this.Id, folderId);
    }
    Delete(deleteMode, suppressReadReceipts = false) {
        return this.InternalDelete(deleteMode, null, null, suppressReadReceipts);
    }
    /**
     * @internal Gets a list of extended properties defined on this object.
     *
     * @return  {ExtendedPropertyCollection}      Extended properties collection.
     */
    GetExtendedProperties() { return this.ExtendedProperties; }
    /**
     * @inrtnal The property definition for the Id of this object.
     *
     * @return  {PropertyDefinition}      A PropertyDefinition instance.
     */
    GetIdPropertyDefinition() { return Schemas_1.Schemas.ItemSchema.Id; }
    /**
     * Gets a value indicating whether a time zone SOAP header should be emitted in a CreateItem or UpdateItem request so this item can be property saved or updated.
     *
     * @param   {boolean}   isUpdateOperation   Indicates whether the operation being petrformed is an update operation.
     * @return  {boolean}                       true if a time zone SOAP header should be emitted; otherwise, false.
     */
    GetIsTimeZoneHeaderRequired(isUpdateOperation) {
        // Starting E14SP2, attachment will be sent along with CreateItem requests. 
        // if the attachment used to require the Timezone header, CreateItem request should do so too.
        //
        //debugger;//todo: filtering of specific type needed.
        if (!isUpdateOperation &&
            (this.Service.RequestedServerVersion >= ExchangeVersion_1.ExchangeVersion.Exchange2010_SP2)) {
            for (var itemAttachment of ExtensionMethods_1.ArrayHelper.OfType(this.Attachments.Items, (a) => a instanceof TypeContainer_1.TypeContainer.ItemAttachment)) //.OfType<ItemAttachment>()) //info: cannot check instanceof to avoid circular dependency in js. TypeContainer is workaround
             {
                if ((itemAttachment.Item != null) && itemAttachment.Item.GetIsTimeZoneHeaderRequired(false /* isUpdateOperation */)) {
                    return true;
                }
            }
        }
        return super.GetIsTimeZoneHeaderRequired(isUpdateOperation);
    }
    /**
     * @internal Gets the minimum required server version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this service object type is supported.
     */
    GetMinimumRequiredServerVersion() { return ExchangeVersion_1.ExchangeVersion.Exchange2007_SP1; }
    /**
     * @internal Internal method to return the schema associated with this type of object.
     *
     * @return  {ServiceObjectSchema}      The schema associated with this type of object.
     */
    GetSchema() { return Schemas_1.Schemas.ItemSchema.Instance; }
    /**
     * @internal Gets the element name of item in XML
     *
     * @return  {string} name of elelment
     */
    GetXmlElementName() { return XmlElementNames_1.XmlElementNames.Item; }
    /**
     * @internal Gets a value indicating whether this instance has unprocessed attachment collection changes.
     *
     * @return  {boolean}      true or false.
     *
     */
    HasUnprocessedAttachmentChanges() { return this.Attachments.HasUnprocessedChanges(); }
    /**
     * @internal Create item.
     *
     * @param   {FolderId}              parentFolderId        The parent folder id.
     * @param   {MessageDisposition}    messageDisposition    The message disposition.
     * @param   {SendInvitationsMode}   sendInvitationsMode   The send invitations mode.
     */
    InternalCreate(parentFolderId, messageDisposition, sendInvitationsMode) {
        this.ThrowIfThisIsNotNew();
        this.ThrowIfThisIsAttachment();
        if (this.IsNew || this.IsDirty) {
            return this.Service.CreateItem(this, parentFolderId, messageDisposition, sendInvitationsMode !== null ? sendInvitationsMode : this.DefaultSendInvitationsMode)
                .then((response) => {
                return this.Attachments.Save();
            });
        }
        return;
    }
    InternalDelete(deleteMode, sendCancellationsMode = this.DefaultSendCancellationsMode, affectedTaskOccurrences = this.DefaultAffectedTaskOccurrences, suppressReadReceipts = false) {
        this.ThrowIfThisIsNew();
        this.ThrowIfThisIsAttachment();
        // If sendCancellationsMode is null, use the default value that's appropriate for item type.
        if ((0, ExtensionMethods_1.isNullOrUndefined)(sendCancellationsMode)) {
            sendCancellationsMode = this.DefaultSendCancellationsMode;
        }
        // If affectedTaskOccurrences is null, use the default value that's appropriate for item type.
        if ((0, ExtensionMethods_1.isNullOrUndefined)(affectedTaskOccurrences)) {
            affectedTaskOccurrences = this.DefaultAffectedTaskOccurrences;
        }
        return this.Service.DeleteItem(this.Id, deleteMode, sendCancellationsMode, affectedTaskOccurrences, suppressReadReceipts);
    }
    /**
     * @internal Loads the specified set of properties on the object.
     *
     * @param   {PropertySet}   propertySet   The properties to load.
     */
    InternalLoad(propertySet) {
        this.ThrowIfThisIsNew();
        this.ThrowIfThisIsAttachment();
        return this.Service.InternalLoadPropertiesForItems([this], //new Item[] { this },
        propertySet, ServiceErrorHandling_1.ServiceErrorHandling.ThrowOnError);
    }
    InternalUpdate(parentFolderId, conflictResolutionMode, messageDisposition, sendInvitationsOrCancellationsMode, suppressReadReceipts = false) {
        this.ThrowIfThisIsNew();
        this.ThrowIfThisIsAttachment();
        var returnedPromise = null;
        var returnedItem = null;
        if (this.IsDirty && this.PropertyBag.GetIsUpdateCallNecessary()) {
            returnedPromise = this.Service.UpdateItem(this, parentFolderId, conflictResolutionMode, messageDisposition, sendInvitationsOrCancellationsMode !== null ? sendInvitationsOrCancellationsMode : this.DefaultSendInvitationsOrCancellationsMode, suppressReadReceipts);
        }
        return Promise.resolve(returnedPromise).then((item) => {
            // Regardless of whether item is dirty or not, if it has unprocessed
            // attachment changes, validate them and process now.
            if (this.HasUnprocessedAttachmentChanges()) {
                this.Attachments.Validate();
                return this.Attachments.Save().then(() => {
                    return item;
                });
            }
            return item;
        });
        //return Promise.resolve(returnedItem);
    }
    Move(destinationFolderIdOrName) {
        this.ThrowIfThisIsNew();
        this.ThrowIfThisIsAttachment();
        var folderId = null;
        if (destinationFolderIdOrName instanceof FolderId_1.FolderId) {
            folderId = destinationFolderIdOrName;
        }
        else {
            folderId = new FolderId_1.FolderId(destinationFolderIdOrName);
        }
        //EwsUtilities.ValidateParam(destinationFolderId, "destinationFolderId");
        return this.Service.MoveItem(this.Id, folderId);
    }
    /**
     * Removes an extended property.
     *
     * @param   {ExtendedPropertyDefinition}    extendedPropertyDefinition   The extended property definition.
     * @return  {boolean}                       True if property was removed.
     */
    RemoveExtendedProperty(extendedPropertyDefinition) { return this.ExtendedProperties.RemoveExtendedProperty(extendedPropertyDefinition); }
    Save(parentFolderIdOrName = null) {
        var parentFolderId = null;
        if (parentFolderIdOrName !== null) {
            parentFolderId = parentFolderIdOrName;
            if (typeof parentFolderIdOrName === 'number') {
                parentFolderId = new FolderId_1.FolderId(parentFolderIdOrName);
            }
        }
        return this.InternalCreate(parentFolderId, MessageDisposition_1.MessageDisposition.SaveOnly, null);
    }
    /**
     * Sets the extended property.
     *
     * @param   {ExtendedPropertyDefinition}    extendedPropertyDefinition   The extended property definition.
     * @param   {value}                         value                        The value.
     */
    SetExtendedProperty(extendedPropertyDefinition, value) { this.ExtendedProperties.SetExtendedProperty(extendedPropertyDefinition, value); }
    /**
     * @internal Sets the subject.
     *
     */
    SetSubject(subject) { this.PropertyBag._setItem(Schemas_1.Schemas.ItemSchema.Subject, subject); }
    /**
     * Throws exception if this is attachment.
     *
     */
    ThrowIfThisIsAttachment() {
        if (this.IsAttachment) {
            throw new Error(Strings_1.Strings.OperationDoesNotSupportAttachments); //InvalidOperationException
        }
    }
    Update(conflictResolutionMode, suppressReadReceipts = false) {
        return this.InternalUpdate(null /* parentFolder */, conflictResolutionMode, MessageDisposition_1.MessageDisposition.SaveOnly, null, suppressReadReceipts);
    }
    /**
     * @internal Validates this instance.
     *
     */
    Validate() {
        super.Validate();
        this.Attachments.Validate();
        // Flag parameter is only valid for Exchange2013 or higher
        //
        var flag = { outValue: null };
        if (this.TryGetProperty(Schemas_1.Schemas.ItemSchema.Flag, flag) && flag.outValue != null) {
            if (this.Service.RequestedServerVersion < ExchangeVersion_1.ExchangeVersion.Exchange2013) {
                throw new ServiceVersionException_1.ServiceVersionException(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.ParameterIncompatibleWithRequestVersion, "Flag", ExchangeVersion_1.ExchangeVersion[ExchangeVersion_1.ExchangeVersion.Exchange2013]));
            }
            flag.outValue.Validate();
        }
    }
}
exports.Item = Item;
//# sourceMappingURL=Item.js.map