"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubscribeToPushNotificationsRequest = void 0;
const ArgumentException_1 = require("../../Exceptions/ArgumentException");
const EwsUtilities_1 = require("../EwsUtilities");
const ExchangeVersion_1 = require("../../Enumerations/ExchangeVersion");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const Strings_1 = require("../../Strings");
const SubscribeResponse_1 = require("../Responses/SubscribeResponse");
const XmlElementNames_1 = require("../XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const PushSubscription_1 = require("../../Notifications/PushSubscription");
const SubscribeRequest_1 = require("./SubscribeRequest");
/**
 * @internal Represents a "push" Subscribe request.
 */
class SubscribeToPushNotificationsRequest extends SubscribeRequest_1.SubscribeRequest {
    /**
     * @internal Initializes a new instance of the **SubscribeToPushNotificationsRequest** class.
     *
     * @param   {ExchangeService}   service   The service.
     */
    constructor(service) {
        super(service);
        this.frequency = 30;
        this.url = null;
        this.callerData = null;
    }
    /**
     * Gets or sets the frequency.
     *
     * @value	The frequency.
     */
    get Frequency() {
        return this.frequency;
    }
    set Frequency(value) {
        this.frequency = value;
    }
    /**
     * Gets or sets the URL.
     *
     * @value	The URL.
     */
    get Url() {
        return this.url;
    }
    set Url(value) {
        this.url = value;
    }
    /**
     * Gets or sets the CallerData.
     *
     * @value	The CallerData.
     */
    get CallerData() {
        return this.callerData;
    }
    set CallerData(value) {
        this.callerData = value;
    }
    /**
     * @internal Creates the service response.
     *
     * @param   {ExchangeService}   service         The service.
     * @param   {number}   			responseIndex   Index of the response.
     * @return  {SubscribeResponse<PushSubscription>}		Service response.
     */
    CreateServiceResponse(service, responseIndex) {
        return new SubscribeResponse_1.SubscribeResponse(new PushSubscription_1.PushSubscription(service));
    }
    /**
     * @internal Gets the request version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this request is supported.
     */
    GetMinimumRequiredServerVersion() {
        return ExchangeVersion_1.ExchangeVersion.Exchange2007_SP1;
    }
    /**
     * @internal Gets the name of the subscription XML element.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this request is supported.
     */
    GetSubscriptionXmlElementName() {
        return XmlElementNames_1.XmlElementNames.PushSubscriptionRequest;
    }
    /**
     * @internal Internal method to write XML elements.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    InternalWriteElementsToXml(writer) {
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.StatusFrequency, this.Frequency);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.URL, this.Url.ToString());
        if (this.Service.RequestedServerVersion >= ExchangeVersion_1.ExchangeVersion.Exchange2013
            && !ExtensionMethods_1.StringHelper.IsNullOrEmpty(this.callerData)) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.CallerData, this.CallerData);
        }
    }
    /**
     * @internal Validate request.
     */
    Validate() {
        super.Validate();
        EwsUtilities_1.EwsUtilities.ValidateParam(this.Url, "Url");
        if ((this.Frequency < 1) || (this.Frequency > 1440)) {
            throw new ArgumentException_1.ArgumentException(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.InvalidFrequencyValue, this.Frequency));
        }
    }
}
exports.SubscribeToPushNotificationsRequest = SubscribeToPushNotificationsRequest;
//# sourceMappingURL=SubscribeToPushNotificationsRequest.js.map