"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GetConversationItemsRequest = void 0;
const ConversationSortOrder_1 = require("../../Enumerations/ConversationSortOrder");
const ExchangeVersion_1 = require("../../Enumerations/ExchangeVersion");
const MailboxSearchLocation_1 = require("../../Enumerations/MailboxSearchLocation");
const ServiceObjectType_1 = require("../../Enumerations/ServiceObjectType");
const ServiceVersionException_1 = require("../../Exceptions/ServiceVersionException");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const Strings_1 = require("../../Strings");
const XmlElementNames_1 = require("../XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const GetConversationItemsResponse_1 = require("../Responses/GetConversationItemsResponse");
const MultiResponseServiceRequest_1 = require("./MultiResponseServiceRequest");
/**
 * @internal Represents a request to a GetConversationItems operation
 *
 * @sealed
 */
class GetConversationItemsRequest extends MultiResponseServiceRequest_1.MultiResponseServiceRequest {
    /**
     * @internal Initializes a new instance of the **GetConversationItemsRequest** class.
     *
     * @param   {ExchangeService}       service   The service.
     * @param   {ServiceErrorHandling}  errorHandlingMode   Indicates how errors should be handled.
     */
    constructor(service, errorHandlingMode) {
        super(service, errorHandlingMode);
        /**
         * @internal Gets or sets the conversations.
         */
        this.Conversations = null;
        /**
         * Gets or sets the item properties.
         */
        this.ItemProperties = null;
        /**
         * @internal Gets or sets the folders to ignore.
         */
        this.FoldersToIgnore = null;
        /**
         * @internal Gets or sets the maximum number of items to return.
         *
         * @Nullable
         */
        this.MaxItemsToReturn = null;
        /**
         * @internal Gets or sets the conversation sort order.
         *
         * @Nullable
         */
        this.SortOrder = null;
        /**
         * @internal Gets or sets the mailbox search location to include in the search.
         *
         * @Nullable
         */
        this.MailboxScope = null;
    }
    /**
     * @internal Creates service response.
     *
     * @param   {ExchangeService}   service         The service.
     * @param   {number}   			responseIndex   Index of the response.
     * @return  {GetConversationItemsResponse}	    Service response.
     */
    CreateServiceResponse(service, responseIndex) {
        return new GetConversationItemsResponse_1.GetConversationItemsResponse(this.ItemProperties);
    }
    /**
     * @internal Gets the expected response message count.
     *
     * @return  {number}      Number of expected response messages.
     */
    GetExpectedResponseMessageCount() {
        return this.Conversations ? this.Conversations.length : 0;
    }
    /**
     * @internal Gets the request version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this request is supported.
     */
    GetMinimumRequiredServerVersion() {
        return ExchangeVersion_1.ExchangeVersion.Exchange2013;
    }
    /**
     * @internal Gets the name of the response message XML element.
     *
     * @return  {string}      Xml element name.
     */
    GetResponseMessageXmlElementName() {
        return XmlElementNames_1.XmlElementNames.GetConversationItemsResponseMessage;
    }
    /**
     * @internal Gets the name of the response XML element.
     *
     * @return  {string}      Xml element name.
     */
    GetResponseXmlElementName() {
        return XmlElementNames_1.XmlElementNames.GetConversationItemsResponse;
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @return  {string}      Xml element name.
     */
    GetXmlElementName() {
        return XmlElementNames_1.XmlElementNames.GetConversationItems;
    }
    /**
     * @internal Validate the request.
     */
    Validate() {
        super.Validate();
        // SearchScope is only valid for Exchange2013 or higher
        //
        if (this.MailboxScope &&
            this.Service.RequestedServerVersion < ExchangeVersion_1.ExchangeVersion.Exchange2013) {
            throw new ServiceVersionException_1.ServiceVersionException(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.ParameterIncompatibleWithRequestVersion, "MailboxScope", ExchangeVersion_1.ExchangeVersion[ExchangeVersion_1.ExchangeVersion.Exchange2013]));
        }
    }
    /**
     * @internal Writes XML attributes.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteAttributesToXml(writer) {
        super.WriteAttributesToXml(writer);
    }
    /**
     * @internal Writes XML elements.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        this.ItemProperties.WriteToXml(writer, ServiceObjectType_1.ServiceObjectType.Item);
        //this.FoldersToIgnore.WriteToXml(writer, XmlNamespace.Messages, XmlElementNames.FoldersToIgnore);
        this.FoldersToIgnore.WriteToXml(writer, XmlElementNames_1.XmlElementNames.FoldersToIgnore, XmlNamespace_1.XmlNamespace.Messages); //info: temp workaround github #52 
        if (this.MaxItemsToReturn) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.MaxItemsToReturn, this.MaxItemsToReturn);
        }
        if ((0, ExtensionMethods_1.hasValue)(this.SortOrder)) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.SortOrder, ConversationSortOrder_1.ConversationSortOrder[this.SortOrder]);
        }
        if ((0, ExtensionMethods_1.hasValue)(this.MailboxScope)) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.MailboxScope, MailboxSearchLocation_1.MailboxSearchLocation[this.MailboxScope]);
        }
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.Conversations);
        this.Conversations.forEach((conversation) => conversation.WriteToXml(writer, XmlElementNames_1.XmlElementNames.Conversation));
        writer.WriteEndElement();
    }
}
exports.GetConversationItemsRequest = GetConversationItemsRequest;
//# sourceMappingURL=GetConversationItemsRequest.js.map