"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RuleCollection = void 0;
const ArgumentException_1 = require("../Exceptions/ArgumentException");
const EwsServiceJsonReader_1 = require("../Core/EwsServiceJsonReader");
const Rule_1 = require("./Rule");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const ComplexProperty_1 = require("./ComplexProperty");
/**
 * Represents a collection of rules.
 *
 * @sealed
 */
class RuleCollection extends ComplexProperty_1.ComplexProperty {
    /**
     * @internal Initializes a new instance of the **RuleCollection** class.
     */
    constructor() {
        super();
        /**
         * The OutlookRuleBlobExists flag.
         */
        this.outlookRuleBlobExists = false;
        /**
         * The rules in the rule collection.
         */
        this.rules = null;
        this.rules = [];
    }
    /**
     * Gets a value indicating whether an Outlook rule blob exists in the user's mailbox. To update rules with EWS when the Outlook rule blob exists, call SetInboxRules passing true as the value of the removeOutlookBlob parameter.
     */
    get OutlookRuleBlobExists() {
        return this.outlookRuleBlobExists;
    }
    set OutlookRuleBlobExists(value) {
        this.outlookRuleBlobExists = value;
    }
    /**
     * Gets the number of rules in this collection.
     */
    get Count() {
        return this.rules.length;
    }
    /**
     * Gets the rule at the specified index in the collection.
     *
     * @param   {number}   index   The index of the rule to get.
     * @return  {Rule}	The rule at the specified index.
     */
    _getItem(index) {
        if (index < 0 || index >= this.rules.length) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("Index");
        }
        return this.rules[index];
    }
    /**
     *  Returns an enumerator that iterates through the collection. this case this.rules
     */
    GetEnumerator() {
        return this.rules;
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}                 jsObject                Json Object converted from XML.
     * @param   {ExchangeService}     service                 The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        let jsRules = EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(jsObject, XmlElementNames_1.XmlElementNames.Rule);
        for (let jsRule of jsRules) {
            let rule = new Rule_1.Rule();
            rule.LoadFromXmlJsObject(jsRule, service);
            this.rules.push(rule);
        }
    }
}
exports.RuleCollection = RuleCollection;
//# sourceMappingURL=RuleCollection.js.map