"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WeeklyPattern = void 0;
const DayOfTheWeekCollection_1 = require("../DayOfTheWeekCollection");
const DayOfWeek_1 = require("../../../Enumerations/DayOfWeek");
const EwsUtilities_1 = require("../../../Core/EwsUtilities");
const ExchangeVersion_1 = require("../../../Enumerations/ExchangeVersion");
const ExtensionMethods_1 = require("../../../ExtensionMethods");
const ServiceValidationException_1 = require("../../../Exceptions/ServiceValidationException");
const Strings_1 = require("../../../Strings");
const XmlElementNames_1 = require("../../../Core/XmlElementNames");
const XmlNamespace_1 = require("../../../Enumerations/XmlNamespace");
const Recurrence_IntervalPattern_1 = require("./Recurrence.IntervalPattern");
/**
 * Represents a recurrence pattern where each occurrence happens on specific days a specific number of weeks after the previous one.
 */
class WeeklyPattern extends Recurrence_IntervalPattern_1.IntervalPattern {
    constructor(startDate, interval, daysOfTheWeek = []) {
        if (arguments.length === 0) {
            super();
            this.daysOfTheWeek = new DayOfTheWeekCollection_1.DayOfTheWeekCollection();
            this.daysOfTheWeek.OnChange.push(this.DaysOfTheWeekChanged.bind(this));
        }
        else {
            super(startDate, interval);
            this.daysOfTheWeek = new DayOfTheWeekCollection_1.DayOfTheWeekCollection();
        }
        this.firstDayOfWeek = null;
        this.daysOfTheWeek.AddRange(daysOfTheWeek);
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @value   The name of the XML element.
     */
    get XmlElementName() {
        return XmlElementNames_1.XmlElementNames.WeeklyRecurrence;
    }
    /**
     * Gets the list of the days of the week when occurrences happen.
     */
    get DaysOfTheWeek() {
        return this.daysOfTheWeek;
    }
    /**
     * Gets or sets the first day of the week for this recurrence.
     */
    get FirstDayOfWeek() {
        return super.GetFieldValueOrThrowIfNull(this.firstDayOfWeek, "FirstDayOfWeek");
    }
    set FirstDayOfWeek(value) {
        this.SetFieldValue({ getValue: () => this.firstDayOfWeek, setValue: (updateValue) => { this.firstDayOfWeek = updateValue; } }, value);
    }
    /**
     * Change event handler.
     *
     * @param   {ComplexProperty}   complexProperty   The complex property.
     */
    DaysOfTheWeekChanged(complexProperty) { this.Changed(); }
    /**
     * @internal Validates this instance.
     */
    InternalValidate() {
        super.InternalValidate();
        if (this.DaysOfTheWeek.Count == 0) {
            throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.DaysOfTheWeekNotSpecified);
        }
    }
    /**
     * @internal Write properties to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    InternalWritePropertiesToXml(writer) {
        super.InternalWritePropertiesToXml(writer);
        this.DaysOfTheWeek.WriteToXml(writer, XmlElementNames_1.XmlElementNames.DaysOfWeek);
        if ((0, ExtensionMethods_1.hasValue)(this.firstDayOfWeek)) {
            //  We only allow the "FirstDayOfWeek" parameter for the Exchange2010_SP1 schema
            //  version.
            //
            EwsUtilities_1.EwsUtilities.ValidatePropertyVersion(writer.Service, ExchangeVersion_1.ExchangeVersion.Exchange2010_SP1, "FirstDayOfWeek");
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.FirstDayOfWeek, DayOfWeek_1.DayOfWeek[this.firstDayOfWeek]);
        }
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}                 jsObject                Json Object converted from XML.
     * @param   {ExchangeService}     service                 The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        super.LoadFromXmlJsObject(jsObject, service);
        for (let key in jsObject) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.DaysOfWeek:
                    this.DaysOfTheWeek.LoadFromXmlJsObjectValue(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.FirstDayOfWeek:
                    this.FirstDayOfWeek = DayOfWeek_1.DayOfWeek[jsObject[key]];
                    break;
                default:
                    break;
            }
        }
    }
}
exports.WeeklyPattern = WeeklyPattern;
//# sourceMappingURL=Recurrence.WeeklyPattern.js.map