"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RelativeYearlyPattern = void 0;
const DayOfTheWeekIndex_1 = require("../../../Enumerations/DayOfTheWeekIndex");
const DayOfTheWeek_1 = require("../../../Enumerations/DayOfTheWeek");
const Month_1 = require("../../../Enumerations/Month");
const ServiceValidationException_1 = require("../../../Exceptions/ServiceValidationException");
const Strings_1 = require("../../../Strings");
const XmlElementNames_1 = require("../../../Core/XmlElementNames");
const XmlNamespace_1 = require("../../../Enumerations/XmlNamespace");
const Recurrence_1 = require("./Recurrence");
/**
 * Represents a recurrence pattern where each occurrence happens on a relative day every year.
 */
class RelativeYearlyPattern extends Recurrence_1.Recurrence {
    constructor(startDate, month = null, dayOfTheWeek = null, dayOfTheWeekIndex = null) {
        if (arguments.length === 0) {
            super();
            this.dayOfTheWeek = null;
            this.dayOfTheWeekIndex = null;
            return;
        }
        super(startDate);
        this.Month = month;
        this.DayOfTheWeek = dayOfTheWeek;
        this.DayOfTheWeekIndex = dayOfTheWeekIndex;
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @value   The name of the XML element.
     */
    get XmlElementName() {
        return XmlElementNames_1.XmlElementNames.RelativeYearlyRecurrence;
    }
    /**
     * Gets or sets the relative position of the day specified in DayOfTheWeek within the month.
     */
    get DayOfTheWeekIndex() {
        return super.GetFieldValueOrThrowIfNull(this.dayOfTheWeekIndex, "DayOfTheWeekIndex");
    }
    set DayOfTheWeekIndex(value) {
        this.SetFieldValue({ getValue: () => this.dayOfTheWeekIndex, setValue: (updateValue) => { this.dayOfTheWeekIndex = updateValue; } }, value);
    }
    /**
     * Gets or sets the day of the week when each occurrence happens.
     */
    get DayOfTheWeek() {
        return super.GetFieldValueOrThrowIfNull(this.dayOfTheWeek, "DayOfTheWeek");
    }
    set DayOfTheWeek(value) {
        this.SetFieldValue({ getValue: () => this.dayOfTheWeek, setValue: (updateValue) => { this.dayOfTheWeek = updateValue; } }, value);
    }
    /**
     * Gets or sets the month of the year when each occurrence happens.
     */
    get Month() {
        return super.GetFieldValueOrThrowIfNull(this.month, "Month");
    }
    set Month(value) {
        this.SetFieldValue({ getValue: () => this.month, setValue: (updateValue) => { this.month = updateValue; } }, value);
    }
    /**
     * @internal Validates this instance.
     */
    InternalValidate() {
        super.InternalValidate();
        if (this.dayOfTheWeekIndex === null) {
            throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.DayOfWeekIndexMustBeSpecifiedForRecurrencePattern);
        }
        if (this.dayOfTheWeek === null) {
            throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.DayOfTheWeekMustBeSpecifiedForRecurrencePattern);
        }
        if (this.month === null) {
            throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.MonthMustBeSpecifiedForRecurrencePattern);
        }
    }
    /**
     * @internal Write properties to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    InternalWritePropertiesToXml(writer) {
        super.InternalWritePropertiesToXml(writer);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.DaysOfWeek, DayOfTheWeek_1.DayOfTheWeek[this.DayOfTheWeek]);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.DayOfWeekIndex, DayOfTheWeekIndex_1.DayOfTheWeekIndex[this.DayOfTheWeekIndex]);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Month, Month_1.Month[this.Month]);
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}                 jsObject                Json Object converted from XML.
     * @param   {ExchangeService}     service                 The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        super.LoadFromXmlJsObject(jsObject, service);
        for (let key in jsObject) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.DaysOfWeek:
                    this.dayOfTheWeek = DayOfTheWeek_1.DayOfTheWeek[jsObject[key]];
                    break;
                case XmlElementNames_1.XmlElementNames.DayOfWeekIndex:
                    this.dayOfTheWeekIndex = DayOfTheWeekIndex_1.DayOfTheWeekIndex[jsObject[key]];
                    break;
                case XmlElementNames_1.XmlElementNames.Month:
                    this.month = Month_1.Month[jsObject[key]];
                    break;
                default:
                    break;
            }
        }
    }
}
exports.RelativeYearlyPattern = RelativeYearlyPattern;
//# sourceMappingURL=Recurrence.RelativeYearlyPattern.js.map