"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MonthlyPattern = void 0;
const ArgumentException_1 = require("../../../Exceptions/ArgumentException");
const ExtensionMethods_1 = require("../../../ExtensionMethods");
const ServiceValidationException_1 = require("../../../Exceptions/ServiceValidationException");
const Strings_1 = require("../../../Strings");
const XmlElementNames_1 = require("../../../Core/XmlElementNames");
const XmlNamespace_1 = require("../../../Enumerations/XmlNamespace");
const Recurrence_IntervalPattern_1 = require("./Recurrence.IntervalPattern");
/**
 * Represents a recurrence pattern where each occurrence happens on a specific day a specific number of months after the previous one.
 */
class MonthlyPattern extends Recurrence_IntervalPattern_1.IntervalPattern {
    constructor(startDate, interval, dayOfMonth = null) {
        if (arguments.length === 0) {
            super();
            this.dayOfMonth = null;
            return;
        }
        super(startDate, interval);
        this.DayOfMonth = dayOfMonth;
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @value   The name of the XML element.
     */
    get XmlElementName() {
        return XmlElementNames_1.XmlElementNames.AbsoluteMonthlyRecurrence;
    }
    /**
     * Gets or sets the day of the month when each occurrence happens. DayOfMonth must be between 1 and 31.
     */
    get DayOfMonth() {
        return super.GetFieldValueOrThrowIfNull(this.dayOfMonth, "DayOfMonth");
    }
    set DayOfMonth(value) {
        if (value < 1 || value > 31) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("DayOfMonth", Strings_1.Strings.DayOfMonthMustBeBetween1And31);
        }
        this.SetFieldValue({ getValue: () => this.dayOfMonth, setValue: (updateValue) => { this.dayOfMonth = updateValue; } }, value);
    }
    /**
     * @internal Validates this instance.
     */
    InternalValidate() {
        super.InternalValidate();
        if (this.dayOfMonth === null) {
            throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.DayOfMonthMustBeBetween1And31);
        }
    }
    /**
     * @internal Write properties to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    InternalWritePropertiesToXml(writer) {
        super.InternalWritePropertiesToXml(writer);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.DayOfMonth, this.DayOfMonth);
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}                 jsObject                Json Object converted from XML.
     * @param   {ExchangeService}     service                 The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        super.LoadFromXmlJsObject(jsObject, service);
        for (let key in jsObject) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.DayOfMonth:
                    this.dayOfMonth = ExtensionMethods_1.Convert.toNumber(jsObject[key]);
                    break;
                default:
                    break;
            }
        }
    }
}
exports.MonthlyPattern = MonthlyPattern;
//# sourceMappingURL=Recurrence.MonthlyPattern.js.map