"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DayOfTheWeekCollection = void 0;
const ArgumentException_1 = require("../../Exceptions/ArgumentException");
const DayOfTheWeek_1 = require("../../Enumerations/DayOfTheWeek");
const EwsUtilities_1 = require("../../Core/EwsUtilities");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const Strings_1 = require("../../Strings");
const XmlElementNames_1 = require("../../Core/XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const ComplexProperty_1 = require("../ComplexProperty");
/**
 * Represents a collection of DayOfTheWeek values.
 */
class DayOfTheWeekCollection extends ComplexProperty_1.ComplexProperty {
    /**
     * @internal Initializes a new instance of the **DayOfTheWeekCollection** class.
     */
    constructor() {
        super();
        this.items = [];
    }
    get Count() {
        return this.items.length;
    }
    /**
     * Gets the DayOfTheWeek at a specific index in the collection.
     *
     * @param   {number}        index   Index
     * @return  {DayOfTheWeek}  DayOfTheWeek at index
     */
    _getItem(index) {
        return this.items[index];
    }
    /**
     * Adds a day to the collection if it is not already present.
     *
     * @param   {DayOfTheWeek}   dayOfTheWeek   The day to add.
     */
    Add(dayOfTheWeek) {
        if (this.items.indexOf(dayOfTheWeek) < 0) {
            this.items.push(dayOfTheWeek);
            this.Changed();
        }
    }
    /**
     * Adds multiple days to the collection if they are not already present.
     *
     * @param   {DayOfTheWeek[]}   daysOfTheWeek   The days to add.
     */
    AddRange(daysOfTheWeek) {
        for (let dayOfTheWeek of daysOfTheWeek) {
            this.Add(dayOfTheWeek);
        }
    }
    /**
     * Clears the collection.
     */
    Clear() {
        if (this.Count > 0) {
            this.items.splice(0);
            this.Changed();
        }
    }
    /**
     *  Returns an enumerator that iterates through the collection. this case this.items
     */
    GetEnumerator() {
        return this.items;
    }
    /**
     * @internal Loads service object from XMLJsObject Value.
     *
     * @param   {string}            jsObject                Json Object converted from XML.
     */
    LoadFromXmlJsObjectValue(jsObjectValue) {
        EwsUtilities_1.EwsUtilities.ParseEnumValueList(this.items, jsObjectValue, ' ', DayOfTheWeek_1.DayOfTheWeek);
    }
    /**
     * Remove a specific day from the collection.
     *
     * @param   {DayOfTheWeek}   dayOfTheWeek   The day to remove.
     * @return  {boolean}       True if the day was removed from the collection, false otherwise.
     */
    Remove(dayOfTheWeek) {
        if (this.items.indexOf(dayOfTheWeek)) {
            let result = this.items.splice(this.items.indexOf(dayOfTheWeek));
            if (result.length > 0) {
                this.Changed();
            }
            return result.length > 0;
        }
        return false;
    }
    /**
     * Removes the day at a specific index.
     *
     * @param   {number}   index   The index of the day to remove.
     */
    RemoveAt(index) {
        if (index < 0 || index >= this.Count) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("index", Strings_1.Strings.IndexIsOutOfRange);
        }
        this.items.splice(index);
        this.Changed();
    }
    ToString(separator = " ") {
        if (this.Count == 0) {
            return ExtensionMethods_1.StringHelper.Empty;
        }
        else {
            let daysOfTheWeekArray = new Array(this.Count);
            for (let i = 0; i < this.Count; i++) {
                daysOfTheWeekArray[i] = DayOfTheWeek_1.DayOfTheWeek[this.items[i]];
            }
            return daysOfTheWeekArray.join(separator);
        }
    }
    /**
     * @internal Writes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer           The writer.
     * @param   {string}   xmlElementName   Name of the XML element.
     */
    WriteToXml(writer, xmlElementName) {
        let daysOfWeekAsString = this.ToString(" ");
        if (!ExtensionMethods_1.StringHelper.IsNullOrEmpty(daysOfWeekAsString)) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.DaysOfWeek, daysOfWeekAsString);
        }
    }
}
exports.DayOfTheWeekCollection = DayOfTheWeekCollection;
//# sourceMappingURL=DayOfTheWeekCollection.js.map