"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DelegatePermissions = void 0;
const DelegateFolderPermissionLevel_1 = require("../Enumerations/DelegateFolderPermissionLevel");
const AltDictionary_1 = require("../AltDictionary");
const ServiceValidationException_1 = require("../Exceptions/ServiceValidationException");
const Strings_1 = require("../Strings");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
const ComplexProperty_1 = require("./ComplexProperty");
/**
 * Represents the permissions of a delegate user.
 *
 * @sealed
 */
class DelegatePermissions extends ComplexProperty_1.ComplexProperty {
    /**
     * @internal Initializes a new instance of the **DelegatePermissions** class.
     */
    constructor() {
        super();
        this.delegateFolderPermissions = null;
        var dictionary = new AltDictionary_1.DictionaryWithStringKey();
        dictionary.Add(XmlElementNames_1.XmlElementNames.CalendarFolderPermissionLevel, new DelegateFolderPermission());
        dictionary.Add(XmlElementNames_1.XmlElementNames.TasksFolderPermissionLevel, new DelegateFolderPermission());
        dictionary.Add(XmlElementNames_1.XmlElementNames.InboxFolderPermissionLevel, new DelegateFolderPermission());
        dictionary.Add(XmlElementNames_1.XmlElementNames.ContactsFolderPermissionLevel, new DelegateFolderPermission());
        dictionary.Add(XmlElementNames_1.XmlElementNames.NotesFolderPermissionLevel, new DelegateFolderPermission());
        dictionary.Add(XmlElementNames_1.XmlElementNames.JournalFolderPermissionLevel, new DelegateFolderPermission());
        this.delegateFolderPermissions = dictionary;
    }
    /**
     * Gets or sets the delegate user's permission on the principal's calendar.
     */
    get CalendarFolderPermissionLevel() {
        return this.delegateFolderPermissions.get(XmlElementNames_1.XmlElementNames.CalendarFolderPermissionLevel).PermissionLevel;
    }
    set CalendarFolderPermissionLevel(value) {
        this.delegateFolderPermissions.get(XmlElementNames_1.XmlElementNames.CalendarFolderPermissionLevel).PermissionLevel = value;
    }
    /**
     * Gets or sets the delegate user's permission on the principal's tasks folder.
     */
    get TasksFolderPermissionLevel() {
        return this.delegateFolderPermissions.get(XmlElementNames_1.XmlElementNames.TasksFolderPermissionLevel).PermissionLevel;
    }
    set TasksFolderPermissionLevel(value) {
        this.delegateFolderPermissions.get(XmlElementNames_1.XmlElementNames.TasksFolderPermissionLevel).PermissionLevel = value;
    }
    /**
     * Gets or sets the delegate user's permission on the principal's inbox.
     */
    get InboxFolderPermissionLevel() {
        return this.delegateFolderPermissions.get(XmlElementNames_1.XmlElementNames.InboxFolderPermissionLevel).PermissionLevel;
    }
    set InboxFolderPermissionLevel(value) {
        this.delegateFolderPermissions.get(XmlElementNames_1.XmlElementNames.InboxFolderPermissionLevel).PermissionLevel = value;
    }
    /**
     * Gets or sets the delegate user's permission on the principal's contacts folder.
     */
    get ContactsFolderPermissionLevel() {
        return this.delegateFolderPermissions.get(XmlElementNames_1.XmlElementNames.ContactsFolderPermissionLevel).PermissionLevel;
    }
    set ContactsFolderPermissionLevel(value) {
        this.delegateFolderPermissions.get(XmlElementNames_1.XmlElementNames.ContactsFolderPermissionLevel).PermissionLevel = value;
    }
    /**
     * Gets or sets the delegate user's permission on the principal's notes folder.
     */
    get NotesFolderPermissionLevel() {
        return this.delegateFolderPermissions.get(XmlElementNames_1.XmlElementNames.NotesFolderPermissionLevel).PermissionLevel;
    }
    set NotesFolderPermissionLevel(value) {
        this.delegateFolderPermissions.get(XmlElementNames_1.XmlElementNames.NotesFolderPermissionLevel).PermissionLevel = value;
    }
    /**
     * Gets or sets the delegate user's permission on the principal's journal folder.
     */
    get JournalFolderPermissionLevel() {
        return this.delegateFolderPermissions.get(XmlElementNames_1.XmlElementNames.JournalFolderPermissionLevel).PermissionLevel;
    }
    set JournalFolderPermissionLevel(value) {
        this.delegateFolderPermissions.get(XmlElementNames_1.XmlElementNames.JournalFolderPermissionLevel).PermissionLevel = value;
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}                 jsObject                Json Object converted from XML.
     * @param   {ExchangeService}     service                 The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        for (let key in jsObject) {
            let delegateFolderPermission = null;
            if (this.delegateFolderPermissions.containsKey(key)) {
                delegateFolderPermission = this.delegateFolderPermissions.get(key);
                delegateFolderPermission.Initialize(DelegateFolderPermissionLevel_1.DelegateFolderPermissionLevel[jsObject[key]]);
            }
        }
    }
    /**
     * @internal Resets this instance.
     */
    Reset() {
        for (let delegateFolderPermission of this.delegateFolderPermissions.Values) {
            delegateFolderPermission.Reset();
        }
    }
    /**
     * @internal Validates this instance for AddDelegate.
     */
    ValidateAddDelegate() {
        // If any folder permission is Custom, throw
        //
        this.delegateFolderPermissions.Values.forEach(permission => {
            if (permission.PermissionLevel == DelegateFolderPermissionLevel_1.DelegateFolderPermissionLevel.Custom) {
                throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.CannotSetDelegateFolderPermissionLevelToCustom);
            }
        });
    }
    /**
     * @internal Validates this instance for UpdateDelegate.
     */
    ValidateUpdateDelegate() {
        // If any folder permission was changed to custom, throw
        //
        this.delegateFolderPermissions.Values.forEach(permission => {
            if (permission.PermissionLevel == DelegateFolderPermissionLevel_1.DelegateFolderPermissionLevel.Custom && !permission.IsExistingPermissionLevelCustom) {
                throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.CannotSetDelegateFolderPermissionLevelToCustom);
            }
        });
    }
    /**
     * @internal Writes the elements to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        this.WritePermissionToXml(writer, XmlElementNames_1.XmlElementNames.CalendarFolderPermissionLevel);
        this.WritePermissionToXml(writer, XmlElementNames_1.XmlElementNames.TasksFolderPermissionLevel);
        this.WritePermissionToXml(writer, XmlElementNames_1.XmlElementNames.InboxFolderPermissionLevel);
        this.WritePermissionToXml(writer, XmlElementNames_1.XmlElementNames.ContactsFolderPermissionLevel);
        this.WritePermissionToXml(writer, XmlElementNames_1.XmlElementNames.NotesFolderPermissionLevel);
        this.WritePermissionToXml(writer, XmlElementNames_1.XmlElementNames.JournalFolderPermissionLevel);
    }
    /**
     * Write permission to Xml.
     *
     * @param   {EwsServiceXmlWriter}   writer           The writer.
     * @param   {string}                xmlElementName   The element name.
     */
    WritePermissionToXml(writer, xmlElementName) {
        let delegateFolderPermissionLevel = this.delegateFolderPermissions.get(xmlElementName).PermissionLevel;
        // UpdateDelegate fails if Custom permission level is round tripped
        //
        if (delegateFolderPermissionLevel != DelegateFolderPermissionLevel_1.DelegateFolderPermissionLevel.Custom) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, xmlElementName, DelegateFolderPermissionLevel_1.DelegateFolderPermissionLevel[delegateFolderPermissionLevel]);
        }
    }
}
exports.DelegatePermissions = DelegatePermissions;
/**
 * @internal Represents a folder's DelegateFolderPermissionLevel
 */
class DelegateFolderPermission {
    constructor() {
        /**
         * @internal Gets or sets the delegate user's permission on a principal's folder.
         */
        this.PermissionLevel = DelegateFolderPermissionLevel_1.DelegateFolderPermissionLevel.None;
        /**
         * @internal Gets IsExistingPermissionLevelCustom.
         */
        this.IsExistingPermissionLevelCustom = false;
    }
    /**
     * Intializes this DelegateFolderPermission.
     *
     * @param   {DelegateFolderPermissionLevel}   permissionLevel   The DelegateFolderPermissionLevel
     */
    Initialize(permissionLevel) {
        this.PermissionLevel = permissionLevel;
        this.IsExistingPermissionLevelCustom = (permissionLevel === DelegateFolderPermissionLevel_1.DelegateFolderPermissionLevel.Custom);
    }
    /**
     * @internal Resets this DelegateFolderPermission.
     */
    Reset() {
        this.Initialize(DelegateFolderPermissionLevel_1.DelegateFolderPermissionLevel.None);
    }
}
//# sourceMappingURL=DelegatePermissions.js.map