"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AttachmentCollection = void 0;
const ArgumentException_1 = require("../Exceptions/ArgumentException");
const ExtensionMethods_1 = require("../ExtensionMethods");
const CreateAttachmentException_1 = require("../Exceptions/CreateAttachmentException");
const DeleteAttachmentException_1 = require("../Exceptions/DeleteAttachmentException");
const EwsLogging_1 = require("../Core/EwsLogging");
const EwsUtilities_1 = require("../Core/EwsUtilities");
const ExchangeVersion_1 = require("../Enumerations/ExchangeVersion");
const FileAttachment_1 = require("./FileAttachment");
const ItemAttachment_1 = require("./ItemAttachment");
const ItemAttachmentOf_1 = require("./ItemAttachmentOf");
const ItemInfo_1 = require("../Core/ServiceObjects/Items/ItemInfo");
const ServiceResult_1 = require("../Enumerations/ServiceResult");
const ServiceValidationException_1 = require("../Exceptions/ServiceValidationException");
const Strings_1 = require("../Strings");
const TypeContainer_1 = require("../TypeContainer");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const Attachment_1 = require("./Attachment");
const ComplexPropertyCollection_1 = require("./ComplexPropertyCollection");
/**
 * Represents an item's attachment collection.
 */
class AttachmentCollection extends ComplexPropertyCollection_1.ComplexPropertyCollection {
    /**
     * @internal Initializes a new instance of AttachmentCollection.
     */
    constructor() {
        super();
        this.___typeGenerics = ["ComplexProperty"];
        /**
         * The item owner that owns this attachment collection
         */
        this.owner = null;
    }
    /**
     * @interface:IOwnedProperty The owner of this attachment collection.
     */
    get Owner() { return this.owner; }
    ;
    set Owner(value) {
        EwsLogging_1.EwsLogging.Assert(value != null && (value instanceof TypeContainer_1.TypeContainer.Item), // instanceof Item), //info: can not check instanceof to avoid circular dependency in js. TypeContainer is workaround
        "AttachmentCollection.IOwnedProperty.set_Owner", "value is not a descendant of ItemBase");
        this.owner = value;
    }
    AddFileAttachmentXXXXX(nameOrFileName, fileNameOrContent, isContent = false) {
        let argsLength = arguments.length;
        if (argsLength == 1) {
            let name = nameOrFileName.split('\\').pop().split('/').pop();
            return this.AddFileAttachment(name, fileNameOrContent);
        }
        if (argsLength === 2) {
            throw new Error("AttachmentCollection.ts - Can only use this method with base64 content");
            let fileAttachment = new FileAttachment_1.FileAttachment(this.owner);
            fileAttachment.Name = 'name';
            fileAttachment.FileName = fileNameOrContent;
            this.InternalAdd(fileAttachment);
            return fileAttachment;
        }
        if (argsLength == 3) {
            if (isContent === true) {
                let fileAttachment = new FileAttachment_1.FileAttachment(this.owner);
                fileAttachment.Name = 'name';
                fileAttachment.Base64Content = fileNameOrContent;
                this.InternalAdd(fileAttachment);
                return fileAttachment;
            }
            else {
                return this.AddFileAttachment(nameOrFileName, fileNameOrContent);
            }
        }
        new Error("AttachmentCollection.ts - AddFileAttachment - incorrect count of parameters");
    }
    /**
     * Adds a file attachment to the collection. - isContent parameter is required to be true to be able to use bas64 content directly
     *
     * @param   {string}    name       The display name of the new attachment.
     * @param   {string}    fileContent   base64 ontent of the file representing the content of the attachment.
     * @return  {FileAttachment}      A FileAttachment instance.
     */
    AddFileAttachment(name, content) {
        let fileAttachment = new FileAttachment_1.FileAttachment(this.owner);
        fileAttachment.Name = name;
        fileAttachment.Base64Content = content;
        this.InternalAdd(fileAttachment);
        return fileAttachment;
    }
    /**
     * Adds an item attachment to the collection
     *
     * @type <TItem>    The type of the item to attach.
     *
     * @param   {any*}      TItem    Item type, not instance
     * @param   {string}    TItemElementName    XML Element Name of the Item class
     * @return  {ItemAttachmentOf<TItem>}      An ItemAttachment instance.
     */
    AddItemAttachment(TItem, TItemElementName) {
        if (typeof TItem.Attachable === 'undefined' || TItem.Attachable === false) {
            throw new Error(ExtensionMethods_1.StringHelper.Format("Items of type {0} are not supported as attachments.", TItem["name"])); //InvalidOperationException
        }
        let itemAttachment = new ItemAttachmentOf_1.ItemAttachmentOf(this.owner); //ref: //info: ItemAttachment can not be generic when same name non generic version exhist. TypeScript limitation
        itemAttachment.Item = (new ItemInfo_1.ItemInfo()).CreateItemFromItemClass(itemAttachment, TItemElementName, true); //todo: needs to implement Reflector metadata for Type to class creation map
        this.InternalAdd(itemAttachment);
        return itemAttachment;
    }
    /**
     * Removes all attachments from this collection.
     */
    Clear() { this.InternalClear(); }
    /**
     * @internal Disables the change log clearing mechanism. Attachment collections are saved separately from the items they belong to.
     */
    ClearChangeLog() { }
    /**
     * @internal Instantiate the appropriate attachment type depending on the current XML element name.
     *
     * @param   {string}   xmlElementName   The XML element name from which to determine the type of attachment to create.
     * @return  {Attachment}        An Attachment instance.
     */
    CreateComplexProperty(xmlElementName) {
        switch (xmlElementName) {
            case XmlElementNames_1.XmlElementNames.FileAttachment:
                return new FileAttachment_1.FileAttachment(this.owner);
            case XmlElementNames_1.XmlElementNames.ItemAttachment:
                return new ItemAttachment_1.ItemAttachment(this.owner);
            default:
                return null;
        }
    }
    //JsonDeserializationNotImplementedException
    CreateDefaultComplexProperty() { EwsLogging_1.EwsLogging.DebugLog("AttachmentCollection.ts - CreateDefaultComplexProperty : Not implemented."); return null; }
    /**
     * @internal Determines the name of the XML element associated with the complexProperty parameter.
     *
     * @param   {Attachment}   complexProperty   The attachment object for which to determine the XML element name with.
     * @return  {string}        The XML element name associated with the complexProperty parameter.
     */
    GetCollectionItemXmlElementName(complexProperty) {
        if (complexProperty instanceof FileAttachment_1.FileAttachment) {
            return XmlElementNames_1.XmlElementNames.FileAttachment;
        }
        else {
            return XmlElementNames_1.XmlElementNames.ItemAttachment;
        }
    }
    /**
     * @internal Determines whether there are any unsaved attachment collection changes.
     *
     * @return  {boolean}      True if attachment adds or deletes haven't been processed yet.
     */
    HasUnprocessedChanges() {
        for (let attachment of this.Items) {
            if (attachment.IsNew) {
                return true;
            }
        }
        // Any pending deletions?
        for (let attachment of this.RemovedItems) {
            if (!attachment.IsNew) {
                return true;
            }
        }
        // Recurse: process item attachments to check for new or deleted sub-attachments.
        for (let itemAttachment of ExtensionMethods_1.ArrayHelper.OfType(this.Items, (attach) => { return attach instanceof ItemAttachment_1.ItemAttachment; })) {
            if (itemAttachment.Item != null) {
                if (itemAttachment.Item.Attachments.HasUnprocessedChanges()) {
                    return true;
                }
            }
        }
        return false;
    }
    /**
     * Calls the CreateAttachment web method to create a list of attachments.
     *
     * @param   {string}        parentItemId   The Id of the parent item of the new attachments.
     * @param   {Attachment[]}  attachments    The attachments to create.
     */
    InternalCreateAttachments(parentItemId, attachments) {
        return this.owner.Service.CreateAttachments(parentItemId, attachments)
            .then((responses) => {
            for (let response of responses.Responses) {
                // We remove all attachments that were successfully deleted from the change log. We should never
                // receive a warning from EWS, so we ignore them.
                if (response.Result != ServiceResult_1.ServiceResult.Error) {
                    this.RemoveFromChangeLog(response.Attachment);
                }
            }
            // TODO : Should we throw for warnings as well?
            if (responses.OverallResult == ServiceResult_1.ServiceResult.Error) {
                throw new CreateAttachmentException_1.CreateAttachmentException(responses, Strings_1.Strings.AttachmentCreationFailed);
            }
        });
    }
    /**
     * Calls the DeleteAttachment web method to delete a list of attachments.
     *
     * @param   {Attachment[]}   attachments   The attachments to delete.
     */
    InternalDeleteAttachments(attachments) {
        return this.owner.Service.DeleteAttachments(attachments)
            .then((responses) => {
            for (let response of responses.Responses) {
                // We remove all attachments that were successfully deleted from the change log. We should never
                // receive a warning from EWS, so we ignore them.
                if (response.Result != ServiceResult_1.ServiceResult.Error) {
                    this.RemoveFromChangeLog(response.Attachment);
                }
            }
            // TODO : Should we throw for warnings as well?
            if (responses.OverallResult == ServiceResult_1.ServiceResult.Error) {
                throw new DeleteAttachmentException_1.DeleteAttachmentException(responses, Strings_1.Strings.AtLeastOneAttachmentCouldNotBeDeleted);
            }
        });
    }
    /**
     * Removes the specified attachment.
     *
     * @param   {Attachment}    attachment   The attachment to remove.
     * @return  {boolean}       True if the attachment was successfully removed from the collection, false otherwise.
     */
    Remove(attachment) {
        EwsUtilities_1.EwsUtilities.ValidateParam(attachment, "attachment");
        return this.InternalRemove(attachment);
    }
    /**
     * Removes the attachment at the specified index.
     *
     * @param   {number}   index   Index of the attachment to remove.
     */
    RemoveAt(index) {
        if (index < 0 || index >= this.Count) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("index", Strings_1.Strings.IndexIsOutOfRange);
        }
        this.InternalRemoveAt(index);
    }
    /**
     * @internal Saves this collection by creating new attachment and deleting removed ones.
     */
    Save() {
        let attachments = [];
        // Retrieve a list of attachments that have to be deleted.
        for (let attachment of this.RemovedItems) {
            if (!attachment.IsNew) {
                attachments.push(attachment);
            }
        }
        // If any, delete them by calling the DeleteAttachment web method.
        //promise resolves with null to keep chaining
        return Promise.resolve(attachments.length > 0 ? this.InternalDeleteAttachments(attachments) : void 0)
            .then(() => {
            attachments.splice(0);
        })
            .then(() => {
            // Retrieve a list of attachments that have to be created.
            for (let attachment of this.Items) {
                if (attachment.IsNew) {
                    attachments.push(attachment);
                }
            }
            // If there are any, create them by calling the CreateAttachment web method.
            if (attachments.length > 0) {
                let parentId = this.owner.IsAttachment ? this.owner.ParentAttachment.Id : this.owner.Id.UniqueId;
                return this.InternalCreateAttachments(parentId, attachments);
            }
            else {
                return Promise.resolve();
            }
        }).then(() => {
            // Process all of the item attachments in this collection.
            let itemAttachments = ExtensionMethods_1.ArrayHelper.OfType(attachments, (attachment) => attachment instanceof ItemAttachment_1.ItemAttachment);
            return itemAttachments.reduce((prev, curr, index) => {
                return prev.then(() => {
                    return curr.Item.Attachments.Save().then(() => {
                        curr.Item.ClearChangeLog();
                    });
                });
            }, Promise.resolve());
        }).then(() => {
            super.ClearChangeLog();
        });
    }
    /**
     * @internal Validates this instance.
     */
    Validate() {
        // Validate all added attachments
        let contactPhotoFound = false;
        for (let attachmentIndex = 0; attachmentIndex < this.AddedItems.length; attachmentIndex++) {
            let attachment = this.AddedItems[attachmentIndex];
            if (attachment.IsNew) {
                // At the server side, only the last attachment with IsContactPhoto is kept, all other IsContactPhoto
                // attachments are removed. CreateAttachment will generate AttachmentId for each of such attachments (although
                // only the last one is valid).
                // 
                // With E14 SP2 CreateItemWithAttachment, such request will only return 1 AttachmentId; but the client
                // expects to see all, so let us prevent such "invalid" request in the first place. 
                // 
                // The IsNew check is to still let CreateAttachmentRequest allow multiple IsContactPhoto attachments.
                // 
                if (this.owner.IsNew && this.owner.Service.RequestedServerVersion >= ExchangeVersion_1.ExchangeVersion.Exchange2010_SP2) {
                    let fileAttachment = attachment;
                    if (fileAttachment instanceof Attachment_1.Attachment && fileAttachment.IsContactPhoto) {
                        if (contactPhotoFound) {
                            throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.MultipleContactPhotosInAttachment);
                        }
                        contactPhotoFound = true;
                    }
                }
                attachment.Validate(attachmentIndex);
            }
        }
    }
    /**
     * @internal Validates and saves this instance. **Not in official EWS source, to workaround some promise errors with validate and save**
     */
    ValidateAndSave() {
        this.Validate();
        return this.Save();
    }
}
exports.AttachmentCollection = AttachmentCollection;
//# sourceMappingURL=AttachmentCollection.js.map