'use strict';

var electron = require('electron');
var redux = require('redux');
var detectBrowsers = require('detect-browsers');
var fs = require('fs');
var path = require('path');

function _interopDefault (e) { return e && e.__esModule ? e : { default: e }; }

var fs__default = /*#__PURE__*/_interopDefault(fs);
var path__default = /*#__PURE__*/_interopDefault(path);

const handle = (channel, handler) => {
  const listener = async (_, id, ...args) => {
    try {
      const resolved = await handler(...args);
      electron.ipcRenderer.send(`${channel}@${id}`, {
        resolved
      });
    } catch (error) {
      error instanceof Error && electron.ipcRenderer.send(`${channel}@${id}`, {
        rejected: {
          name: error.name,
          message: error.message,
          stack: error.stack
        }
      });
    }
  };
  electron.ipcRenderer.on(channel, listener);
  return () => {
    electron.ipcRenderer.removeListener(channel, listener);
  };
};
const invoke = (channel, ...args) => electron.ipcRenderer.invoke(channel, ...args);

const jitsiDomain = window.location.origin;
const desktopCapturer = {
  getSources: opts => electron.ipcRenderer.invoke('jitsi-desktop-capturer-get-sources', [opts, jitsiDomain])
};
const JitsiMeetElectron = {
  async obtainDesktopStreams(callback, errorCallback, options) {
    try {
      const sources = (await desktopCapturer.getSources(options)).map(source => ({
        id: source.id,
        name: source.name,
        display_id: source.display_id,
        thumbnail: {
          toDataURL: () => source.thumbnail.toDataURL()
        },
        appIcon: {
          toDataURL: () => source.appIcon.toDataURL()
        }
      }));
      callback(sources);
    } catch (error) {
      error instanceof Error && errorCallback(error);
      console.log(error);
    }
  }
};

let getAbsoluteUrl;
let serverUrl;
const setServerUrl = _serverUrl => {
  serverUrl = _serverUrl;
};
const getServerUrl = () => serverUrl;
const setUrlResolver = _getAbsoluteUrl => {
  getAbsoluteUrl = _getAbsoluteUrl;
};

const isFSA = action => typeof action === 'object' && action !== null && !Array.isArray(action) && 'type' in action && typeof action.type === 'string';
const hasMeta = action => 'meta' in action && typeof action.meta === 'object' && action.meta !== null;
const isResponse = action => hasMeta(action) && action.meta.response === true;
const isLocallyScoped = action => hasMeta(action) && action.meta.scope === 'local';
const isErrored = action => 'meta' in action && action.error === true && action.payload instanceof Error;
const hasPayload = action => 'payload' in action;
const isResponseTo = (id, ...types) => action => isResponse(action) && types.includes(action.type) && action.meta.id === id;

const getInitialState = () => invoke('redux/get-initial-state');
const forwardToMain = api => {
  handle('redux/action-dispatched', async action => {
    api.dispatch(action);
  });
  return next => action => {
    if (!isFSA(action) || isLocallyScoped(action)) {
      return next(action);
    }
    invoke('redux/action-dispatched', action);
    return action;
  };
};

const APP_PATH_SET = 'app/path-set';
const APP_VERSION_SET = 'app/version-set';
const APP_SETTINGS_LOADED = 'app/settings-loaded';
const APP_ALLOWED_NTLM_CREDENTIALS_DOMAINS_SET = 'app/allowed-ntlm-credentials-domains-set';
const APP_MAIN_WINDOW_TITLE_SET = 'app/main-window-title-set';
const APP_MACHINE_THEME_SET = 'app/machine-theme-set';
const APP_SCREEN_CAPTURE_FALLBACK_FORCED_SET = 'app/screen-capture-fallback-forced-set';

const allowedNTLMCredentialsDomains = (state = null, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      {
        const {
          allowedNTLMCredentialsDomains = state
        } = action.payload;
        return allowedNTLMCredentialsDomains;
      }
    case APP_ALLOWED_NTLM_CREDENTIALS_DOMAINS_SET:
      {
        if (action.payload === null) return null;
        return action.payload;
      }
    default:
      return state;
  }
};

const appPath = (state = null, action) => {
  switch (action.type) {
    case APP_PATH_SET:
      return action.payload;
    default:
      return state;
  }
};

const appVersion = (state = null, action) => {
  switch (action.type) {
    case APP_VERSION_SET:
      return action.payload;
    default:
      return state;
  }
};

const machineTheme = (state = 'light', action) => {
  switch (action.type) {
    case APP_MACHINE_THEME_SET:
      {
        return action.payload;
      }
    default:
      return state;
  }
};

const mainWindowTitle = (state = null, action) => {
  switch (action.type) {
    case APP_MAIN_WINDOW_TITLE_SET:
      return action.payload;
    default:
      return state;
  }
};

const screenCaptureFallbackForced = (state = false, action) => {
  switch (action.type) {
    case APP_SCREEN_CAPTURE_FALLBACK_FORCED_SET:
      return action.payload;
    default:
      return state;
  }
};

const DOWNLOAD_CREATED = 'downloads/created';
const DOWNLOAD_REMOVED = 'downloads/removed';
const DOWNLOADS_CLEARED = 'downloads/cleared';
const DOWNLOAD_UPDATED = 'downloads/updated';

const DownloadStatus = {
  ALL: 'All',
  PAUSED: 'Paused',
  CANCELLED: 'Cancelled'
};

const downloads = (state = {}, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      {
        var _action$payload$downl;
        const initDownloads = (_action$payload$downl = action.payload.downloads) !== null && _action$payload$downl !== void 0 ? _action$payload$downl : {};
        Object.values(initDownloads).forEach(value => {
          if (value.state === 'progressing' || value.state === 'paused') {
            value.state = 'cancelled';
            value.status = DownloadStatus.CANCELLED;
          }
        });
        return initDownloads !== null && initDownloads !== void 0 ? initDownloads : {};
      }
    case DOWNLOAD_CREATED:
      {
        const download = action.payload;
        return {
          ...state,
          [download.itemId]: download
        };
      }
    case DOWNLOAD_UPDATED:
      {
        const existingDownload = state[action.payload.itemId];
        if (!existingDownload) {
          return state; // Don't update if download doesn't exist
        }
        const newState = {
          ...state
        };
        newState[action.payload.itemId] = {
          ...existingDownload,
          ...action.payload
        };
        return newState;
      }
    case DOWNLOAD_REMOVED:
      {
        const newState = {
          ...state
        };
        delete newState[action.payload];
        return newState;
      }
    case DOWNLOADS_CLEARED:
      return {};
    default:
      return state;
  }
};

const JITSI_SERVER_CAPTURE_SCREEN_PERMISSION_UPDATED = 'jitsi-server-capture-screen-permission-updated';
const JITSI_SERVER_CAPTURE_SCREEN_PERMISSIONS_CLEARED = 'jitsi-server-capture-screen-permissions-cleared';

const allowedJitsiServers = (state = {}, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      {
        const {
          allowedJitsiServers = {}
        } = action.payload;
        state = allowedJitsiServers;
        return state;
      }
    case JITSI_SERVER_CAPTURE_SCREEN_PERMISSION_UPDATED:
      {
        state = {
          ...state,
          [action.payload.jitsiServer]: action.payload.allowed
        };
        return state;
      }
    case JITSI_SERVER_CAPTURE_SCREEN_PERMISSIONS_CLEARED:
      {
        state = {};
        return state;
      }
    default:
      return state;
  }
};

const CERTIFICATES_CLEARED = 'certificates/cleared';
const CERTIFICATES_LOADED = 'certificates/loaded';
const CERTIFICATES_CLIENT_CERTIFICATE_REQUESTED = 'certificates/client-certificate-requested';
const TRUSTED_CERTIFICATES_UPDATED = 'trusted-certificates/updated';
const NOT_TRUSTED_CERTIFICATES_UPDATED = 'not-trusted-certificates/updated';
const SELECT_CLIENT_CERTIFICATE_DIALOG_CERTIFICATE_SELECTED = 'select-client-certificate-dialog/certificate-selected';
const SELECT_CLIENT_CERTIFICATE_DIALOG_DISMISSED = 'select-client-certificate-dialog/dismissed';
const EXTERNAL_PROTOCOL_PERMISSION_UPDATED = 'navigation/external-protocol-permission-updated';

const clientCertificates = (state = [], action) => {
  switch (action.type) {
    case CERTIFICATES_CLIENT_CERTIFICATE_REQUESTED:
      return action.payload;
    case SELECT_CLIENT_CERTIFICATE_DIALOG_CERTIFICATE_SELECTED:
    case SELECT_CLIENT_CERTIFICATE_DIALOG_DISMISSED:
      return [];
    default:
      return state;
  }
};
const trustedCertificates = (state = {}, action) => {
  switch (action.type) {
    case CERTIFICATES_LOADED:
    case TRUSTED_CERTIFICATES_UPDATED:
      return action.payload;
    case CERTIFICATES_CLEARED:
      return {};
    case APP_SETTINGS_LOADED:
      {
        const {
          trustedCertificates = state
        } = action.payload;
        return trustedCertificates;
      }
    default:
      return state;
  }
};
const notTrustedCertificates = (state = {}, action) => {
  switch (action.type) {
    case NOT_TRUSTED_CERTIFICATES_UPDATED:
      return action.payload;
    case CERTIFICATES_CLEARED:
      return {};
    case APP_SETTINGS_LOADED:
      {
        const {
          notTrustedCertificates = state
        } = action.payload;
        return notTrustedCertificates;
      }
    default:
      return state;
  }
};
const externalProtocols = (state = {}, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      {
        const {
          externalProtocols = {}
        } = action.payload;
        state = externalProtocols;
        return state;
      }
    case EXTERNAL_PROTOCOL_PERMISSION_UPDATED:
      {
        state = {
          ...state,
          [action.payload.protocol]: action.payload.allowed
        };
        return state;
      }
    default:
      return state;
  }
};

const DEEP_LINKS_SERVER_ADDED = 'deep-links/server-added';
const DEEP_LINKS_SERVER_FOCUSED = 'deep-links/server-focused';

const OUTLOOK_CALENDAR_SET_CREDENTIALS = 'outlook-calendar/set-credentials';
const OUTLOOK_CALENDAR_ASK_CREDENTIALS = 'outlook-calendar/ask-credentials';
const OUTLOOK_CALENDAR_DIALOG_DISMISSED = 'outlook-calendar/dialog-dismissed';
const OUTLOOK_CALENDAR_SAVE_CREDENTIALS = 'outlook-calendar/save-credentials';

const ABOUT_DIALOG_DISMISSED = 'about-dialog/dismissed';
const ABOUT_DIALOG_TOGGLE_UPDATE_ON_START = 'about-dialog/toggle-update-on-start';
const ABOUT_DIALOG_UPDATE_CHANNEL_CHANGED = 'about-dialog/update-channel-changed';
const ADD_SERVER_VIEW_SERVER_ADDED = 'add-server/view-server-added';
const MENU_BAR_ABOUT_CLICKED = 'menu-bar/about-clicked';
const MENU_BAR_ADD_NEW_SERVER_CLICKED = 'menu-bar/add-new-server-clicked';
const MENU_BAR_SELECT_SERVER_CLICKED = 'menu-bar/select-server-clicked';
const MENU_BAR_TOGGLE_IS_MENU_BAR_ENABLED_CLICKED = 'menu-bar/toggle-is-menu-bar-enabled-clicked';
const MENU_BAR_TOGGLE_IS_SHOW_WINDOW_ON_UNREAD_CHANGED_ENABLED_CLICKED = 'menu-bar/toggle-is-show-window-on-unread-changed-enabled-clicked';
const MENU_BAR_TOGGLE_IS_SIDE_BAR_ENABLED_CLICKED = 'menu-bar/toggle-is-side-bar-enabled-clicked';
const MENU_BAR_TOGGLE_IS_TRAY_ICON_ENABLED_CLICKED = 'menu-bar/toggle-is-tray-icon-enabled-clicked';
const MENU_BAR_TOGGLE_IS_DEVELOPER_MODE_ENABLED_CLICKED = 'menu-bar/toggle-is-developer-mode-enabled-clicked';
const MENU_BAR_TOGGLE_IS_VIDEO_CALL_DEVTOOLS_AUTO_OPEN_ENABLED_CLICKED = 'menu-bar/toggle-is-video-call-devtools-auto-open-enabled-clicked';
const ROOT_WINDOW_ICON_CHANGED = 'root-window/icon-changed';
const ROOT_WINDOW_STATE_CHANGED = 'root-window/state-changed';
const VIDEO_CALL_WINDOW_STATE_CHANGED = 'video-call-window/state-changed';
const SIDE_BAR_ADD_NEW_SERVER_CLICKED = 'side-bar/add-new-server-clicked';
const SIDE_BAR_DOWNLOADS_BUTTON_CLICKED = 'side-bar/downloads-button-clicked';
const SIDE_BAR_SETTINGS_BUTTON_CLICKED = 'side-bar/settings-button-clicked';
const SIDE_BAR_REMOVE_SERVER_CLICKED = 'side-bar/remove-server-clicked';
const SIDE_BAR_SERVER_SELECTED = 'side-bar/server-selected';
const SIDE_BAR_SERVERS_SORTED = 'side-bar/servers-sorted';
const TOUCH_BAR_FORMAT_BUTTON_TOUCHED = 'touch-bar/format-button-touched';
const TOUCH_BAR_SELECT_SERVER_TOUCHED = 'touch-bar/select-server-touched';
const UPDATE_DIALOG_DISMISSED = 'update-dialog/dismissed';
const UPDATE_DIALOG_INSTALL_BUTTON_CLICKED = 'update-dialog/install-button-clicked';
const UPDATE_DIALOG_REMIND_UPDATE_LATER_CLICKED = 'update-dialog/remind-update-later-clicked';
const UPDATE_DIALOG_SKIP_UPDATE_CLICKED = 'update-dialog/skip-update-clicked';
const WEBVIEW_READY = 'webview/ready';
const WEBVIEW_ATTACHED = 'webview/attached';
const WEBVIEW_DID_FAIL_LOAD = 'webview/did-fail-load';
const WEBVIEW_DID_NAVIGATE = 'webview/did-navigate';
const WEBVIEW_DID_START_LOADING = 'webview/did-start-loading';
const WEBVIEW_FAVICON_CHANGED = 'webview/favicon-changed';
const WEBVIEW_FOCUS_REQUESTED = 'webview/focus-requested';
const WEBVIEW_MESSAGE_BOX_BLURRED = 'webview/message-box-blurred';
const WEBVIEW_MESSAGE_BOX_FOCUSED = 'webview/message-box-focused';
const WEBVIEW_SCREEN_SHARING_SOURCE_REQUESTED = 'webview/screen-sharing-source-requested';
const WEBVIEW_SCREEN_SHARING_SOURCE_RESPONDED = 'webview/screen-sharing-source-responded';
const WEBVIEW_SIDEBAR_STYLE_CHANGED = 'webview/sidebar-style-changed';
const WEBVIEW_SIDEBAR_CUSTOM_THEME_CHANGED = 'webview/sidebar-custom-theme-changed';
const WEBVIEW_GIT_COMMIT_HASH_CHANGED = 'webview/git-commit-hash-changed';
const WEBVIEW_GIT_COMMIT_HASH_CHECK = 'webview/git-commit-hash-check';
const WEBVIEW_TITLE_CHANGED = 'webview/title-changed';
const WEBVIEW_PAGE_TITLE_CHANGED = 'webview/page-title-changed';
const WEBVIEW_UNREAD_CHANGED = 'webview/unread-changed';
const WEBVIEW_USER_LOGGED_IN = 'webview/user-loggedin';
const WEBVIEW_USER_THEME_APPEARANCE_CHANGED = 'webview/user-theme-appearance-changed';
const WEBVIEW_ALLOWED_REDIRECTS_CHANGED = 'webview/allowed-redirects-changed';
const SETTINGS_SET_REPORT_OPT_IN_CHANGED = 'settings/set-bugsnag-opt-in-changed';
const SETTINGS_SET_FLASHFRAME_OPT_IN_CHANGED = 'settings/set-flashframe-opt-in-changed';
const SETTINGS_SET_HARDWARE_ACCELERATION_OPT_IN_CHANGED = 'settings/set-hardware-acceleration-opt-in-changed';
const SETTINGS_SET_INTERNALVIDEOCHATWINDOW_OPT_IN_CHANGED = 'settings/set-internalvideochatwindow-opt-in-changed';
const SETTINGS_SET_MINIMIZE_ON_CLOSE_OPT_IN_CHANGED = 'settings/set-minimize-on-close-opt-in-changed';
const SETTINGS_SET_IS_TRAY_ICON_ENABLED_CHANGED = 'settings/set-is-tray-icon-enabled-changed';
const SETTINGS_SET_IS_SIDE_BAR_ENABLED_CHANGED = 'settings/set-is-side-bar-enabled-changed';
const SETTINGS_SET_IS_MENU_BAR_ENABLED_CHANGED = 'settings/set-is-menu-bar-enabled-changed';
const SETTINGS_SET_IS_VIDEO_CALL_WINDOW_PERSISTENCE_ENABLED_CHANGED = 'settings/set-is-video-call-window-persistence-enabled-changed';
const SETTINGS_SET_IS_DEVELOPER_MODE_ENABLED_CHANGED = 'settings/set-is-developer-mode-enabled-changed';
const SETTINGS_SET_IS_VIDEO_CALL_DEVTOOLS_AUTO_OPEN_ENABLED_CHANGED = 'settings/set-is-video-call-devtools-auto-open-enabled-changed';
const SETTINGS_SET_IS_VIDEO_CALL_SCREEN_CAPTURE_FALLBACK_ENABLED_CHANGED = 'settings/set-is-video-call-screen-capture-fallback-enabled-changed';
const SETTINGS_NTLM_CREDENTIALS_CHANGED = 'settings/ntlm-credentials-changed';
const SETTINGS_AVAILABLE_BROWSERS_UPDATED = 'settings/available-browsers-updated';
const SETTINGS_SELECTED_BROWSER_CHANGED = 'settings/selected-browser-changed';
const SET_HAS_TRAY_MINIMIZE_NOTIFICATION_SHOWN = 'notifications/set-has-tray-minimize-notification-shown';
const DOWNLOADS_BACK_BUTTON_CLICKED = 'downloads/back-button-clicked';
const WEBVIEW_SERVER_SUPPORTED_VERSIONS_UPDATED = 'webview/server-supported-versions-updated';
const WEBVIEW_SERVER_UNIQUE_ID_UPDATED = 'webview/server-workspace-uid-updated';
const WEBVIEW_SERVER_IS_SUPPORTED_VERSION = 'webview/server-is-supported-version';
const WEBVIEW_SERVER_SUPPORTED_VERSIONS_LOADING = 'webview/server-supported-versions-loading';
const WEBVIEW_SERVER_SUPPORTED_VERSIONS_ERROR = 'webview/server-supported-versions-error';
const WEBVIEW_SERVER_VERSION_UPDATED = 'webview/version-updated';
const SUPPORTED_VERSION_DIALOG_DISMISS = 'supported-versions-dialog/dismiss';
const WEBVIEW_FORCE_RELOAD_WITH_CACHE_CLEAR = 'webview/force-reload-with-cache-clear';
const OPEN_SERVER_INFO_MODAL = 'server-info-modal/open';
const CLOSE_SERVER_INFO_MODAL = 'server-info-modal/close';

const SERVERS_LOADED = 'servers/loaded';
const SERVER_DOCUMENT_VIEWER_OPEN_URL = 'server/document-viewer/open-url';

/* eslint-disable complexity */

const ensureUrlFormat = serverUrl => {
  if (serverUrl) {
    return new URL(serverUrl).href;
  }
  throw new Error('cannot handle null server URLs');
};
const upsert = (state, server) => {
  const index = state.findIndex(({
    url
  }) => url === server.url);
  if (index === -1) {
    return [...state, server];
  }
  return state.map((_server, i) => i === index ? {
    ..._server,
    ...server
  } : _server);
};
const update = (state, server) => {
  const index = state.findIndex(({
    url
  }) => url === server.url);
  if (index === -1) {
    return state;
  }
  return state.map((_server, i) => i === index ? {
    ..._server,
    ...server
  } : _server);
};
const servers = (state = [], action) => {
  switch (action.type) {
    case ADD_SERVER_VIEW_SERVER_ADDED:
    case DEEP_LINKS_SERVER_ADDED:
      {
        const url = action.payload;
        return upsert(state, {
          url,
          title: url
        });
      }
    case SIDE_BAR_REMOVE_SERVER_CLICKED:
      {
        const _url = action.payload;
        return state.filter(({
          url
        }) => url !== _url);
      }
    case SIDE_BAR_SERVERS_SORTED:
      {
        const urls = action.payload;
        return state.sort(({
          url: a
        }, {
          url: b
        }) => urls.indexOf(a) - urls.indexOf(b));
      }
    case WEBVIEW_TITLE_CHANGED:
      {
        const {
          url,
          title = url
        } = action.payload;
        return upsert(state, {
          url,
          title
        });
      }
    case WEBVIEW_PAGE_TITLE_CHANGED:
      {
        const {
          url,
          pageTitle
        } = action.payload;
        return upsert(state, {
          url,
          pageTitle
        });
      }
    case WEBVIEW_SERVER_SUPPORTED_VERSIONS_UPDATED:
      {
        const {
          url,
          supportedVersions,
          source
        } = action.payload;
        return upsert(state, {
          url,
          supportedVersions,
          supportedVersionsSource: source,
          supportedVersionsFetchState: 'success'
        });
      }
    case WEBVIEW_SERVER_SUPPORTED_VERSIONS_LOADING:
      {
        const {
          url
        } = action.payload;
        return upsert(state, {
          url,
          supportedVersionsFetchState: 'loading'
        });
      }
    case WEBVIEW_SERVER_SUPPORTED_VERSIONS_ERROR:
      {
        const {
          url
        } = action.payload;
        return upsert(state, {
          url,
          supportedVersionsFetchState: 'error'
        });
      }
    case SUPPORTED_VERSION_DIALOG_DISMISS:
      {
        const {
          url
        } = action.payload;
        return upsert(state, {
          url,
          expirationMessageLastTimeShown: new Date()
        });
      }
    case WEBVIEW_SERVER_UNIQUE_ID_UPDATED:
      {
        const {
          url,
          uniqueID
        } = action.payload;
        return upsert(state, {
          url,
          uniqueID
        });
      }
    case WEBVIEW_USER_THEME_APPEARANCE_CHANGED:
      {
        const {
          url,
          themeAppearance
        } = action.payload;
        return upsert(state, {
          url,
          themeAppearance
        });
      }
    case WEBVIEW_SERVER_IS_SUPPORTED_VERSION:
      {
        const {
          url,
          isSupportedVersion
        } = action.payload;
        return upsert(state, {
          url,
          isSupportedVersion,
          supportedVersionsValidatedAt: new Date()
        });
      }
    case WEBVIEW_SERVER_VERSION_UPDATED:
      {
        const {
          url,
          version
        } = action.payload;
        return upsert(state, {
          url,
          version
        });
      }
    case WEBVIEW_UNREAD_CHANGED:
      {
        const {
          url,
          badge
        } = action.payload;
        return upsert(state, {
          url,
          badge
        });
      }
    case WEBVIEW_USER_LOGGED_IN:
      {
        const {
          url,
          userLoggedIn
        } = action.payload;
        return upsert(state, {
          url,
          userLoggedIn
        });
      }
    case WEBVIEW_ALLOWED_REDIRECTS_CHANGED:
      {
        const {
          url,
          allowedRedirects
        } = action.payload;
        return upsert(state, {
          url,
          allowedRedirects
        });
      }
    case WEBVIEW_SIDEBAR_STYLE_CHANGED:
      {
        const {
          url,
          style
        } = action.payload;
        return upsert(state, {
          url,
          style
        });
      }
    case WEBVIEW_SIDEBAR_CUSTOM_THEME_CHANGED:
      {
        const {
          url,
          customTheme
        } = action.payload;
        return upsert(state, {
          url,
          customTheme
        });
      }
    case WEBVIEW_GIT_COMMIT_HASH_CHANGED:
      {
        const {
          url,
          gitCommitHash
        } = action.payload;
        return upsert(state, {
          url,
          gitCommitHash
        });
      }
    case WEBVIEW_FAVICON_CHANGED:
      {
        const {
          url,
          favicon
        } = action.payload;
        return upsert(state, {
          url,
          favicon
        });
      }
    case WEBVIEW_DID_NAVIGATE:
      {
        const {
          url,
          pageUrl
        } = action.payload;
        if (pageUrl !== null && pageUrl !== void 0 && pageUrl.includes(url)) {
          return upsert(state, {
            url,
            lastPath: pageUrl
          });
        }
        return state;
      }
    case WEBVIEW_DID_START_LOADING:
      {
        const {
          url
        } = action.payload;
        return upsert(state, {
          url,
          failed: false
        });
      }
    case WEBVIEW_DID_FAIL_LOAD:
      {
        const {
          url,
          isMainFrame
        } = action.payload;
        if (isMainFrame) {
          return upsert(state, {
            url,
            failed: true
          });
        }
        return state;
      }
    case SERVERS_LOADED:
      {
        const {
          servers = state
        } = action.payload;
        return servers.map(server => ({
          ...server,
          url: ensureUrlFormat(server.url)
        }));
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          servers = state
        } = action.payload;
        return servers.map(server => ({
          ...server,
          url: ensureUrlFormat(server.url),
          documentViewerOpenUrl: ''
        }));
      }
    case WEBVIEW_READY:
      {
        const {
          url,
          webContentsId
        } = action.payload;
        return update(state, {
          url,
          webContentsId
        });
      }
    case WEBVIEW_ATTACHED:
      {
        const {
          url,
          webContentsId
        } = action.payload;
        return update(state, {
          url,
          webContentsId
        });
      }
    case OUTLOOK_CALENDAR_SAVE_CREDENTIALS:
      {
        const {
          url,
          outlookCredentials
        } = action.payload;
        return upsert(state, {
          url,
          outlookCredentials
        });
      }
    case SERVER_DOCUMENT_VIEWER_OPEN_URL:
      {
        const {
          server,
          documentUrl
        } = action.payload;
        return upsert(state, {
          url: server,
          documentViewerOpenUrl: documentUrl
        });
      }
    default:
      return state;
  }
};

const availableBrowsers = (state = [], action) => {
  switch (action.type) {
    case SETTINGS_AVAILABLE_BROWSERS_UPDATED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        return state;
      }
    default:
      return state;
  }
};

const currentView = (state = 'add-new-server', action) => {
  switch (action.type) {
    case ADD_SERVER_VIEW_SERVER_ADDED:
    case DEEP_LINKS_SERVER_ADDED:
    case DEEP_LINKS_SERVER_FOCUSED:
    case MENU_BAR_SELECT_SERVER_CLICKED:
    case TOUCH_BAR_SELECT_SERVER_TOUCHED:
    case SIDE_BAR_SERVER_SELECTED:
      {
        const url = action.payload;
        return {
          url
        };
      }
    case WEBVIEW_FOCUS_REQUESTED:
      {
        const {
          url,
          view
        } = action.payload;
        if (view === 'downloads') return 'downloads';
        return {
          url
        };
      }
    case SERVERS_LOADED:
      {
        const {
          selected
        } = action.payload;
        return selected ? {
          url: selected
        } : 'add-new-server';
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          currentView = state
        } = action.payload;
        return currentView;
      }
    case MENU_BAR_ADD_NEW_SERVER_CLICKED:
    case SIDE_BAR_ADD_NEW_SERVER_CLICKED:
      return 'add-new-server';
    case SIDE_BAR_REMOVE_SERVER_CLICKED:
      {
        if (typeof state === 'object' && state.url === action.payload) {
          return 'add-new-server';
        }
        return state;
      }
    case SIDE_BAR_DOWNLOADS_BUTTON_CLICKED:
      return 'downloads';
    case SIDE_BAR_SETTINGS_BUTTON_CLICKED:
      return 'settings';
    case DOWNLOADS_BACK_BUTTON_CLICKED:
      return {
        url: action.payload
      };
    default:
      return state;
  }
};

const initialServerInfoModalState = {
  isOpen: false,
  serverData: null
};
const serverInfoModal = (state = initialServerInfoModalState, action) => {
  switch (action.type) {
    case OPEN_SERVER_INFO_MODAL:
      return {
        isOpen: true,
        serverData: action.payload
      };
    case CLOSE_SERVER_INFO_MODAL:
      return initialServerInfoModalState;
    default:
      return state;
  }
};
const dialogs = (state = {
  serverInfoModal: initialServerInfoModalState
}, action) => {
  switch (action.type) {
    case OPEN_SERVER_INFO_MODAL:
    case CLOSE_SERVER_INFO_MODAL:
      return {
        ...state,
        serverInfoModal: serverInfoModal(state.serverInfoModal, action)
      };
    default:
      return state;
  }
};

const hasHideOnTrayNotificationShown = (state = false, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      return Boolean(action.payload.hasHideOnTrayNotificationShown);
    case SET_HAS_TRAY_MINIMIZE_NOTIFICATION_SHOWN:
      return action.payload;
    default:
      return state;
  }
};

const isAddNewServersEnabled = (state = true, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      return Boolean(action.payload.isAddNewServersEnabled);
    default:
      return state;
  }
};

const isDeveloperModeEnabled = (state = false, action) => {
  switch (action.type) {
    case SETTINGS_SET_IS_DEVELOPER_MODE_ENABLED_CHANGED:
    case MENU_BAR_TOGGLE_IS_DEVELOPER_MODE_ENABLED_CLICKED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          isDeveloperModeEnabled = state
        } = action.payload;
        return isDeveloperModeEnabled;
      }
    default:
      return state;
  }
};

const UPDATE_SKIPPED = 'update/skipped';
const UPDATES_CHECKING_FOR_UPDATE = 'updates/checking-for-update';
const UPDATES_ERROR_THROWN = 'updates/error-thrown';
const UPDATES_NEW_VERSION_AVAILABLE = 'updates/new-version-available';
const UPDATES_NEW_VERSION_NOT_AVAILABLE = 'updates/new-version-not-available';
const UPDATES_READY = 'updates/ready';
const UPDATES_CHANNEL_CHANGED = 'updates/channel-changed';

const isFlashFrameEnabled = (state = false, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      return Boolean(action.payload.isFlashFrameEnabled);
    case UPDATES_READY:
      return action.payload.isFlashFrameEnabled;
    case SETTINGS_SET_FLASHFRAME_OPT_IN_CHANGED:
      {
        return action.payload;
      }
    default:
      return state;
  }
};

const isHardwareAccelerationEnabled = (state = false, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      return Boolean(action.payload.isHardwareAccelerationEnabled);
    case UPDATES_READY:
      return action.payload.isHardwareAccelerationEnabled;
    case SETTINGS_SET_HARDWARE_ACCELERATION_OPT_IN_CHANGED:
      {
        return action.payload;
      }
    default:
      return state;
  }
};

const isInternalVideoChatWindowEnabled = (state = false, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      return Boolean(action.payload.isInternalVideoChatWindowEnabled);
    case UPDATES_READY:
      return action.payload.isInternalVideoChatWindowEnabled;
    case SETTINGS_SET_INTERNALVIDEOCHATWINDOW_OPT_IN_CHANGED:
      {
        return action.payload;
      }
    default:
      return state;
  }
};

const isMenuBarEnabled = (state = true, action) => {
  switch (action.type) {
    case SETTINGS_SET_IS_MENU_BAR_ENABLED_CHANGED:
    case MENU_BAR_TOGGLE_IS_MENU_BAR_ENABLED_CLICKED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          isMenuBarEnabled = state
        } = action.payload;
        return isMenuBarEnabled;
      }
    default:
      return state;
  }
};

const isMessageBoxFocused = (state = false, action) => {
  switch (action.type) {
    case WEBVIEW_MESSAGE_BOX_FOCUSED:
      return true;
    case WEBVIEW_DID_START_LOADING:
    case WEBVIEW_MESSAGE_BOX_BLURRED:
    case WEBVIEW_DID_FAIL_LOAD:
      return false;
    default:
      return state;
  }
};

const isMinimizeOnCloseEnabled = (state = process.platform === 'win32', action) => {
  switch (action.type) {
    case SETTINGS_SET_MINIMIZE_ON_CLOSE_OPT_IN_CHANGED:
      {
        return action.payload;
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          isMinimizeOnCloseEnabled = state
        } = action.payload;
        return isMinimizeOnCloseEnabled;
      }
    default:
      return state;
  }
};

const isNTLMCredentialsEnabled = (state = false, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      {
        const {
          isNTLMCredentialsEnabled = state
        } = action.payload;
        return isNTLMCredentialsEnabled;
      }
    case SETTINGS_NTLM_CREDENTIALS_CHANGED:
      {
        return action.payload;
      }
    default:
      return state;
  }
};

const isReportEnabled = (state = false, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      return Boolean(action.payload.isReportEnabled);
    case UPDATES_READY:
      return action.payload.isReportEnabled;
    case SETTINGS_SET_REPORT_OPT_IN_CHANGED:
      {
        return action.payload;
      }
    default:
      return state;
  }
};

const isShowWindowOnUnreadChangedEnabled = (state = false, action) => {
  switch (action.type) {
    case MENU_BAR_TOGGLE_IS_SHOW_WINDOW_ON_UNREAD_CHANGED_ENABLED_CLICKED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          isShowWindowOnUnreadChangedEnabled = state
        } = action.payload;
        return isShowWindowOnUnreadChangedEnabled;
      }
    default:
      return state;
  }
};

const isSideBarEnabled = (state = true, action) => {
  switch (action.type) {
    case SETTINGS_SET_IS_SIDE_BAR_ENABLED_CHANGED:
    case MENU_BAR_TOGGLE_IS_SIDE_BAR_ENABLED_CLICKED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          isSideBarEnabled = state
        } = action.payload;
        return isSideBarEnabled;
      }
    default:
      return state;
  }
};

const isTrayIconEnabled = (state = process.platform !== 'linux', action) => {
  switch (action.type) {
    case SETTINGS_SET_IS_TRAY_ICON_ENABLED_CHANGED:
    case MENU_BAR_TOGGLE_IS_TRAY_ICON_ENABLED_CLICKED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          isTrayIconEnabled = state
        } = action.payload;
        return isTrayIconEnabled;
      }
    default:
      return state;
  }
};

const isVideoCallDevtoolsAutoOpenEnabled = (state = false, action) => {
  switch (action.type) {
    case SETTINGS_SET_IS_VIDEO_CALL_DEVTOOLS_AUTO_OPEN_ENABLED_CHANGED:
    case MENU_BAR_TOGGLE_IS_VIDEO_CALL_DEVTOOLS_AUTO_OPEN_ENABLED_CLICKED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          isVideoCallDevtoolsAutoOpenEnabled = state
        } = action.payload;
        return isVideoCallDevtoolsAutoOpenEnabled;
      }
    default:
      return state;
  }
};

const isVideoCallScreenCaptureFallbackEnabled = (state = false, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      return Boolean(action.payload.isVideoCallScreenCaptureFallbackEnabled);
    case UPDATES_READY:
      {
        const {
          isVideoCallScreenCaptureFallbackEnabled = state
        } = action.payload;
        return isVideoCallScreenCaptureFallbackEnabled;
      }
    case SETTINGS_SET_IS_VIDEO_CALL_SCREEN_CAPTURE_FALLBACK_ENABLED_CHANGED:
      return action.payload;
    default:
      return state;
  }
};

const isVideoCallWindowPersistenceEnabled = (state = true,
// Enabled by default
action) => {
  switch (action.type) {
    case SETTINGS_SET_IS_VIDEO_CALL_WINDOW_PERSISTENCE_ENABLED_CHANGED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          isVideoCallWindowPersistenceEnabled = state
        } = action.payload;
        return isVideoCallWindowPersistenceEnabled;
      }
    default:
      return state;
  }
};

const lastSelectedServerUrl = (state = '', action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      {
        const {
          lastSelectedServerUrl = state,
          servers
        } = action.payload;
        if (state === '' && servers && servers.length > 0) {
          return servers[0].url;
        }
        return lastSelectedServerUrl;
      }
    case SIDE_BAR_SERVER_SELECTED:
      return action.payload;
    default:
      return state;
  }
};

const SCREEN_SHARING_DIALOG_DISMISSED = 'screen-sharing-dialog/dismissed';

const openDialog = (state = null, action) => {
  switch (action.type) {
    case MENU_BAR_ABOUT_CLICKED:
      return 'about';
    case WEBVIEW_SCREEN_SHARING_SOURCE_REQUESTED:
      return 'screen-sharing';
    case UPDATES_NEW_VERSION_AVAILABLE:
      return 'update';
    case CERTIFICATES_CLIENT_CERTIFICATE_REQUESTED:
      return 'select-client-certificate';
    case ABOUT_DIALOG_DISMISSED:
      if (state === 'about') {
        return null;
      }
      return state;
    case OUTLOOK_CALENDAR_ASK_CREDENTIALS:
      return 'outlook-credentials';
    case SCREEN_SHARING_DIALOG_DISMISSED:
    case WEBVIEW_SCREEN_SHARING_SOURCE_RESPONDED:
    case SELECT_CLIENT_CERTIFICATE_DIALOG_CERTIFICATE_SELECTED:
    case SELECT_CLIENT_CERTIFICATE_DIALOG_DISMISSED:
    case UPDATE_DIALOG_DISMISSED:
    case UPDATE_DIALOG_SKIP_UPDATE_CLICKED:
    case UPDATE_DIALOG_REMIND_UPDATE_LATER_CLICKED:
    case UPDATE_DIALOG_INSTALL_BUTTON_CLICKED:
    case OUTLOOK_CALENDAR_DIALOG_DISMISSED:
    case OUTLOOK_CALENDAR_SET_CREDENTIALS:
      return null;
    default:
      return state;
  }
};

const rootWindowIcon = (state = null, action) => {
  switch (action.type) {
    case ROOT_WINDOW_ICON_CHANGED:
      {
        return action.payload;
      }
    default:
      return state;
  }
};

const rootWindowState = (state = {
  focused: true,
  visible: true,
  maximized: false,
  minimized: false,
  fullscreen: false,
  normal: true,
  bounds: {
    x: undefined,
    y: undefined,
    width: 1000,
    height: 600
  }
}, action) => {
  switch (action.type) {
    case ROOT_WINDOW_STATE_CHANGED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          rootWindowState = state
        } = action.payload;
        return rootWindowState;
      }
    default:
      return state;
  }
};

const selectedBrowser = (state = null, action) => {
  switch (action.type) {
    case SETTINGS_SELECTED_BROWSER_CHANGED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          selectedBrowser = state
        } = action.payload;
        return selectedBrowser;
      }
    default:
      return state;
  }
};

const videoCallWindowState = (state = {
  focused: true,
  visible: true,
  maximized: false,
  minimized: false,
  fullscreen: false,
  normal: true,
  bounds: {
    x: undefined,
    y: undefined,
    width: 0,
    height: 0
  }
}, action) => {
  switch (action.type) {
    case VIDEO_CALL_WINDOW_STATE_CHANGED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          videoCallWindowState = state
        } = action.payload;
        return videoCallWindowState;
      }
    default:
      return state;
  }
};

const doCheckForUpdatesOnStartup = (state = true, action) => {
  switch (action.type) {
    case UPDATES_READY:
      {
        const {
          doCheckForUpdatesOnStartup
        } = action.payload;
        return doCheckForUpdatesOnStartup;
      }
    case ABOUT_DIALOG_TOGGLE_UPDATE_ON_START:
      {
        const doCheckForUpdatesOnStartup = action.payload;
        return doCheckForUpdatesOnStartup;
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          doCheckForUpdatesOnStartup = state
        } = action.payload;
        return doCheckForUpdatesOnStartup;
      }
    default:
      return state;
  }
};
const isCheckingForUpdates = (state = false, action) => {
  switch (action.type) {
    case UPDATES_CHECKING_FOR_UPDATE:
      return true;
    case UPDATES_ERROR_THROWN:
      return false;
    case UPDATES_NEW_VERSION_NOT_AVAILABLE:
      return false;
    case UPDATES_NEW_VERSION_AVAILABLE:
      return false;
    default:
      return state;
  }
};
const isEachUpdatesSettingConfigurable = (state = true, action) => {
  switch (action.type) {
    case UPDATES_READY:
      {
        const {
          isEachUpdatesSettingConfigurable
        } = action.payload;
        return isEachUpdatesSettingConfigurable;
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          isEachUpdatesSettingConfigurable = state
        } = action.payload;
        return isEachUpdatesSettingConfigurable;
      }
    default:
      return state;
  }
};
const isUpdatingAllowed = (state = true, action) => {
  switch (action.type) {
    case UPDATES_READY:
      {
        const {
          isUpdatingAllowed
        } = action.payload;
        return isUpdatingAllowed;
      }
    default:
      return state;
  }
};
const isUpdatingEnabled = (state = true, action) => {
  switch (action.type) {
    case UPDATES_READY:
      {
        const {
          isUpdatingEnabled
        } = action.payload;
        return isUpdatingEnabled;
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          isUpdatingEnabled = state
        } = action.payload;
        return isUpdatingEnabled;
      }
    default:
      return state;
  }
};
const newUpdateVersion = (state = null, action) => {
  switch (action.type) {
    case UPDATES_NEW_VERSION_AVAILABLE:
      {
        const newUpdateVersion = action.payload;
        return newUpdateVersion;
      }
    case UPDATES_NEW_VERSION_NOT_AVAILABLE:
    case UPDATE_SKIPPED:
      {
        return null;
      }
    default:
      return state;
  }
};
const skippedUpdateVersion = (state = null, action) => {
  switch (action.type) {
    case UPDATES_READY:
      {
        const {
          skippedUpdateVersion
        } = action.payload;
        return skippedUpdateVersion;
      }
    case UPDATE_SKIPPED:
      {
        const skippedUpdateVersion = action.payload;
        return skippedUpdateVersion;
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          skippedUpdateVersion = state
        } = action.payload;
        return skippedUpdateVersion;
      }
    default:
      return state;
  }
};
const updateError = (state = null, action) => {
  switch (action.type) {
    case UPDATES_CHECKING_FOR_UPDATE:
      return null;
    case UPDATES_ERROR_THROWN:
      return action.payload;
    case UPDATES_NEW_VERSION_NOT_AVAILABLE:
      return null;
    case UPDATES_NEW_VERSION_AVAILABLE:
      return null;
    default:
      return state;
  }
};
const updateChannel = (state = 'latest', action) => {
  switch (action.type) {
    case ABOUT_DIALOG_UPDATE_CHANNEL_CHANGED:
    case UPDATES_CHANNEL_CHANGED:
      {
        return action.payload;
      }
    case UPDATES_READY:
      {
        const {
          updateChannel
        } = action.payload;
        return updateChannel;
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          updateChannel = state
        } = action.payload;
        return updateChannel;
      }
    default:
      return state;
  }
};

const rootReducer = redux.combineReducers({
  allowedJitsiServers,
  appPath,
  appVersion,
  availableBrowsers,
  clientCertificates,
  currentView,
  dialogs,
  doCheckForUpdatesOnStartup,
  downloads,
  externalProtocols,
  isCheckingForUpdates,
  isEachUpdatesSettingConfigurable,
  isMenuBarEnabled,
  isMessageBoxFocused,
  isShowWindowOnUnreadChangedEnabled,
  isSideBarEnabled,
  isTrayIconEnabled,
  isMinimizeOnCloseEnabled,
  isUpdatingAllowed,
  isUpdatingEnabled,
  mainWindowTitle,
  machineTheme,
  newUpdateVersion,
  openDialog,
  rootWindowIcon,
  rootWindowState,
  selectedBrowser,
  servers,
  skippedUpdateVersion,
  trustedCertificates,
  notTrustedCertificates,
  updateError,
  isReportEnabled,
  isFlashFrameEnabled,
  isHardwareAccelerationEnabled,
  isInternalVideoChatWindowEnabled,
  isAddNewServersEnabled,
  hasHideOnTrayNotificationShown,
  lastSelectedServerUrl,
  allowedNTLMCredentialsDomains,
  isNTLMCredentialsEnabled,
  videoCallWindowState,
  isVideoCallWindowPersistenceEnabled,
  isDeveloperModeEnabled,
  updateChannel,
  screenCaptureFallbackForced,
  isVideoCallDevtoolsAutoOpenEnabled,
  isVideoCallScreenCaptureFallbackEnabled
});

let reduxStore;
let lastAction;
const catchLastAction = () => next => action => {
  lastAction = action;
  return next(action);
};
const createRendererReduxStore = async () => {
  const initialState = await getInitialState();
  const composeEnhancers = window.__REDUX_DEVTOOLS_EXTENSION_COMPOSE__ || redux.compose;
  const enhancers = composeEnhancers(redux.applyMiddleware(forwardToMain, catchLastAction));
  reduxStore = redux.createStore(rootReducer, initialState, enhancers);
  return reduxStore;
};
const dispatch = action => {
  reduxStore.dispatch(action);
};
const select = selector => selector(reduxStore.getState());
const watch = (selector, watcher) => {
  const initial = select(selector);
  watcher(initial, undefined);
  let prev = initial;
  return reduxStore.subscribe(() => {
    const curr = select(selector);
    if (Object.is(prev, curr)) {
      return;
    }
    watcher(curr, prev);
    prev = curr;
  });
};
const listen = (typeOrPredicate, listener) => {
  const effectivePredicate = typeof typeOrPredicate === 'function' ? typeOrPredicate : action => action.type === typeOrPredicate;
  return reduxStore.subscribe(() => {
    if (!effectivePredicate(lastAction)) {
      return;
    }
    listener(lastAction);
  });
};

// const isResponseTo = <Response extends RootAction>(id: unknown, type: Response['type']) =>
//   (action: RootAction): action is Response =>
//     isResponse(action) && action.type === type && action.meta.id === id;

const request = (requestAction, ...types) => new Promise((resolve, reject) => {
  const id = Math.random().toString(36).slice(2);
  const unsubscribe = listen(isResponseTo(id, ...types), action => {
    unsubscribe();
    if (isErrored(action)) {
      reject(action.payload);
      return;
    }
    if (hasPayload(action)) {
      resolve(action.payload);
    }
  });
  dispatch({
    ...requestAction,
    meta: {
      request: true,
      id
    }
  });
});

const NOTIFICATIONS_CREATE_REQUESTED = 'notifications/create-requested';
const NOTIFICATIONS_CREATE_RESPONDED = 'notifications/create-responded';
const NOTIFICATIONS_NOTIFICATION_ACTIONED = 'notifications/notification-actioned';
const NOTIFICATIONS_NOTIFICATION_CLICKED = 'notifications/notification-clicked';
const NOTIFICATIONS_NOTIFICATION_CLOSED = 'notifications/notification-closed';
const NOTIFICATIONS_NOTIFICATION_DISMISSED = 'notifications/notification-dismissed';
const NOTIFICATIONS_NOTIFICATION_REPLIED = 'notifications/notification-replied';
const NOTIFICATIONS_NOTIFICATION_SHOWN = 'notifications/notification-shown';

const normalizeIconUrl = iconUrl => {
  if (/^data:/.test(iconUrl)) {
    return iconUrl;
  }
  if (!/^https?:\/\//.test(iconUrl)) {
    return getAbsoluteUrl(iconUrl);
  }
  return iconUrl;
};
const eventHandlers = new Map();
const createNotification = async ({
  title,
  icon,
  onEvent,
  ...options
}) => {
  const id = await request({
    type: NOTIFICATIONS_CREATE_REQUESTED,
    payload: {
      title,
      ...(icon ? {
        icon: normalizeIconUrl(icon)
      } : {}),
      ...options
    }
  }, NOTIFICATIONS_CREATE_RESPONDED);
  eventHandlers.set(id, event => onEvent === null || onEvent === void 0 ? void 0 : onEvent({
    type: event.type,
    detail: event.detail
  }));
  return id;
};
const destroyNotification = id => {
  dispatch({
    type: NOTIFICATIONS_NOTIFICATION_DISMISSED,
    payload: {
      id
    }
  });
  eventHandlers.delete(id);
};
const listenToNotificationsRequests = () => {
  listen(NOTIFICATIONS_NOTIFICATION_SHOWN, action => {
    const {
      payload: {
        id
      }
    } = action;
    const eventHandler = eventHandlers.get(id);
    eventHandler === null || eventHandler === void 0 || eventHandler({
      type: 'show'
    });
  });
  listen(NOTIFICATIONS_NOTIFICATION_CLOSED, action => {
    const {
      payload: {
        id
      }
    } = action;
    const eventHandler = eventHandlers.get(id);
    eventHandler === null || eventHandler === void 0 || eventHandler({
      type: 'close'
    });
    eventHandlers.delete(id);
  });
  listen(NOTIFICATIONS_NOTIFICATION_CLICKED, action => {
    const {
      payload: {
        id,
        title
      }
    } = action;
    dispatch({
      type: WEBVIEW_FOCUS_REQUESTED,
      payload: {
        url: getServerUrl(),
        view: title === 'Downloads' ? 'downloads' : 'server'
      }
    });
    const eventHandler = eventHandlers.get(id);
    eventHandler === null || eventHandler === void 0 || eventHandler({
      type: 'click'
    });
  });
  listen(NOTIFICATIONS_NOTIFICATION_REPLIED, action => {
    const {
      payload: {
        id,
        reply
      }
    } = action;
    const eventHandler = eventHandlers.get(id);
    eventHandler === null || eventHandler === void 0 || eventHandler({
      type: 'reply',
      detail: {
        reply
      }
    });
  });
  listen(NOTIFICATIONS_NOTIFICATION_ACTIONED, action => {
    const {
      payload: {
        id,
        index
      }
    } = action;
    const eventHandler = eventHandlers.get(id);
    eventHandler === null || eventHandler === void 0 || eventHandler({
      type: 'action',
      detail: {
        index
      }
    });
  });
};

const handleGetSourceIdEvent = async () => {
  try {
    var _window$top;
    const sourceId = await request({
      type: WEBVIEW_SCREEN_SHARING_SOURCE_REQUESTED
    }, WEBVIEW_SCREEN_SHARING_SOURCE_RESPONDED);
    (_window$top = window.top) === null || _window$top === void 0 || _window$top.postMessage({
      sourceId
    }, '*');
  } catch (error) {
    var _window$top2;
    (_window$top2 = window.top) === null || _window$top2 === void 0 || _window$top2.postMessage({
      sourceId: 'PermissionDeniedError'
    }, '*');
  }
};
const listenToScreenSharingRequests = () => {
  window.addEventListener('get-sourceId', handleGetSourceIdEvent);
};

const getOutlookEvents = async date => {
  const response = await electron.ipcRenderer.invoke('outlook-calendar/get-events', date);
  return response;
};
const setOutlookExchangeUrl = (url, userId) => {
  electron.ipcRenderer.invoke('outlook-calendar/set-exchange-url', url, userId);
};
const hasOutlookCredentials = async () => electron.ipcRenderer.invoke('outlook-calendar/has-credentials');
const clearOutlookCredentials = () => {
  electron.ipcRenderer.invoke('outlook-calendar/clear-credentials');
};
const setUserToken = (token, userId) => {
  electron.ipcRenderer.invoke('outlook-calendar/set-user-token', token, userId);
};

const SYSTEM_LOCKING_SCREEN = 'system/locking-screen';
const SYSTEM_SUSPENDING = 'system/suspending';

let detachCallbacks;
const attachCallbacks = ({
  isAutoAwayEnabled,
  idleThreshold,
  setUserOnline
}) => {
  const unsubscribeFromPowerMonitorEvents = listen(action => [SYSTEM_SUSPENDING, SYSTEM_LOCKING_SCREEN].includes(action.type), () => {
    if (!isAutoAwayEnabled) {
      return;
    }
    setUserOnline(false);
  });
  let pollingTimer;
  let prevState;
  const pollSystemIdleState = async () => {
    if (!isAutoAwayEnabled || !idleThreshold) {
      return;
    }
    pollingTimer = setTimeout(pollSystemIdleState, 2000);
    const state = await invoke('power-monitor/get-system-idle-state', idleThreshold);
    if (prevState === state) {
      return;
    }
    const isOnline = state === 'active' || state === 'unknown';
    setUserOnline(isOnline);
    prevState = state;
  };
  pollSystemIdleState();
  return () => {
    unsubscribeFromPowerMonitorEvents();
    clearTimeout(pollingTimer);
  };
};
const setUserPresenceDetection = options => {
  var _detachCallbacks;
  (_detachCallbacks = detachCallbacks) === null || _detachCallbacks === void 0 || _detachCallbacks();
  detachCallbacks = attachCallbacks(options);
};

const setBadge = badge => {
  dispatch({
    type: WEBVIEW_UNREAD_CHANGED,
    payload: {
      url: getServerUrl(),
      badge
    }
  });
};

const writeTextToClipboard = text => {
  electron.clipboard.writeText(text);
};

const openDocumentViewer = (url, format, options) => {
  electron.ipcRenderer.invoke('document-viewer/open-window', url, format, options);
};

const FAVICON_SIZE = 100;
let imageElement;
const getImageElement = () => {
  if (!imageElement) {
    const canvas = document.createElement('canvas');
    canvas.width = FAVICON_SIZE;
    canvas.height = FAVICON_SIZE;
    const ctx = canvas.getContext('2d');
    if (!ctx) {
      throw new Error('failed to create canvas 2d context');
    }
    imageElement = new Image();
    const handleImageLoadEvent = () => {
      ctx.clearRect(0, 0, FAVICON_SIZE, FAVICON_SIZE);
      ctx.drawImage(imageElement, 0, 0, FAVICON_SIZE, FAVICON_SIZE);
      dispatch({
        type: WEBVIEW_FAVICON_CHANGED,
        payload: {
          url: getServerUrl(),
          favicon: canvas.toDataURL()
        }
      });
    };
    imageElement.addEventListener('load', handleImageLoadEvent, {
      passive: true
    });
  }
  return imageElement;
};
const setFavicon = faviconUrl => {
  if (typeof faviconUrl !== 'string') {
    return;
  }
  const imageElement = getImageElement();
  imageElement.src = getAbsoluteUrl(faviconUrl);
};

const setGitCommitHash = gitCommitHash => {
  dispatch({
    type: WEBVIEW_GIT_COMMIT_HASH_CHECK,
    payload: {
      url: getServerUrl(),
      gitCommitHash
    }
  });
};

const readSetting = key => {
  try {
    const filePath = path__default.default.join(electron.app.getPath('userData'), 'config.json');
    const content = fs__default.default.readFileSync(filePath, 'utf8');
    const json = JSON.parse(content);
    return json[key];
  } catch (e) {
    return null;
  }
};

// Cache browsers to avoid repeatedly fetching them
let cachedBrowsers = null;
let browserLoadPromise = null;

/**
 * Lazy load browsers asynchronously
 * This ensures we don't slow down app startup
 */
const loadBrowsersLazy = () => {
  if (cachedBrowsers) {
    return Promise.resolve(cachedBrowsers);
  }
  if (!browserLoadPromise) {
    // Start loading browsers asynchronously after a delay to not block the app startup
    browserLoadPromise = new Promise(resolve => {
      // Delay browser detection for 2 seconds after this function is first called
      // to avoid slowing down app startup and initial interactions
      setTimeout(async () => {
        try {
          const browsers = await detectBrowsers.getAvailableBrowsers();
          cachedBrowsers = browsers;
          const browserIds = browsers.map(browser => browser.browser);
          if (browserIds.length > 0) {
            dispatch({
              type: SETTINGS_AVAILABLE_BROWSERS_UPDATED,
              payload: browserIds
            });
          }
          resolve(browsers);
        } catch (error) {
          console.error('Error detecting browsers:', error);
          resolve([]);
        }
      }, 2000);
    });
  }
  return browserLoadPromise;
};

/**
 * Launches a URL in the selected browser from settings or falls back to system default
 *
 * @param url The URL to open
 * @returns Promise that resolves when the browser is launched
 */
const openExternal = async url => {
  // Get the selected browser from settings
  const selectedBrowser = readSetting('selectedBrowser');

  // If no specific browser is selected, use the system default
  if (!selectedBrowser) {
    return electron.shell.openExternal(url);
  }
  try {
    // Lazy load browsers when needed
    const browsers = await loadBrowsersLazy();

    // Find the selected browser in the available browsers
    const browser = browsers.find(browser => browser.browser === selectedBrowser);
    if (browser) {
      // Launch the selected browser with the URL
      return detectBrowsers.launchBrowser(browser, url);
    }
    // If the selected browser isn't available, fall back to system default
    console.warn(`Selected browser "${selectedBrowser}" not found, using system default.`);
    return electron.shell.openExternal(url);
  } catch (error) {
    console.error('Error launching browser:', error);
    // Fall back to shell.openExternal on error
    return electron.shell.openExternal(url);
  }
};

const getInternalVideoChatWindowEnabled = () => select(({
  isInternalVideoChatWindowEnabled
}) => ({
  isInternalVideoChatWindowEnabled
})).isInternalVideoChatWindowEnabled;
const openInternalVideoChatWindow = (url, options) => {
  const validUrl = new URL(url);
  const allowedProtocols = ['http:', 'https:'];
  if (!allowedProtocols.includes(validUrl.protocol)) {
    return;
  }
  if (!process.mas && getInternalVideoChatWindowEnabled()) {
    switch (options === null || options === void 0 ? void 0 : options.providerName) {
      case 'jitsi':
        // window.open(validUrl.href, 'Video Call', 'scrollbars=true');
        // We will open Jitsi on browser instead of opening a new window for compatibility from their side
        electron.ipcRenderer.invoke('video-call-window/open-window', validUrl.href, options);
        break;
      case 'googlemeet':
        openExternal(validUrl.href);
        break;
      default:
        electron.ipcRenderer.invoke('video-call-window/open-window', validUrl.href, options);
        break;
    }
  } else {
    openExternal(validUrl.href);
  }
};

const reloadServer = () => {
  const url = getServerUrl();

  // Dispatch action to trigger force reload with cache clear
  dispatch({
    type: WEBVIEW_FORCE_RELOAD_WITH_CACHE_CLEAR,
    payload: url
  });
};

let timer;
let prevBackground;
let prevColor;
let prevBorder;
let serverVersion;
function versionIsGreaterOrEqualsTo(version1, version2) {
  var _version1$match, _version2$match;
  const v1 = ((_version1$match = version1.match(/\d+/g)) === null || _version1$match === void 0 ? void 0 : _version1$match.map(Number)) || [];
  const v2 = ((_version2$match = version2.match(/\d+/g)) === null || _version2$match === void 0 ? void 0 : _version2$match.map(Number)) || [];
  for (let i = 0; i < 3; i++) {
    const n1 = v1[i] || 0;
    const n2 = v2[i] || 0;
    if (n1 > n2) {
      return true;
    }
    if (n1 < n2) {
      return false;
    }
  }
  return true;
}
const pollSidebarStyle = (referenceElement, emit) => {
  clearTimeout(timer);
  document.body.append(referenceElement);
  const {
    background,
    color,
    border
  } = window.getComputedStyle(referenceElement);
  referenceElement.remove();
  const newBgg = prevBackground !== background ? background : prevBackground;
  const newColor = prevColor !== color ? color : prevColor;
  const newBorder = prevBorder !== border ? border : prevBorder;
  if (prevBackground !== background || prevColor !== color || newBorder !== border) {
    emit({
      background: newBgg,
      color: newColor,
      border: newBorder
    });
    prevBackground = background;
    prevColor = color;
    prevBorder = border;
  }
  timer = setTimeout(() => pollSidebarStyle(referenceElement, emit), 5000);
};
let element;
const getElement = () => {
  if (!element) {
    element = document.createElement('div');
    element.style.backgroundColor = 'var(--sidebar-background)';
    element.style.color = 'var(--sidebar-item-text-color)';
    element.style.display = 'none';
    if (versionIsGreaterOrEqualsTo(serverVersion, '6.3.0')) {
      element.classList.add('rcx-sidebar--main');
      element.style.border = '1px solid var(--sidebar-border-color)';
    } else {
      element.classList.add('sidebar');
    }
  }
  return element;
};
const setServerVersionToSidebar = version => {
  serverVersion = version;
};
const setBackground = imageUrl => {
  const element = getElement();
  element.style.backgroundImage = imageUrl ? `url(${JSON.stringify(getAbsoluteUrl(imageUrl))})` : 'none';
  pollSidebarStyle(element, sideBarStyle => {
    dispatch({
      type: WEBVIEW_SIDEBAR_STYLE_CHANGED,
      payload: {
        url: getServerUrl(),
        style: sideBarStyle
      }
    });
  });
};
const setSidebarCustomTheme = customTheme => {
  dispatch({
    type: WEBVIEW_SIDEBAR_CUSTOM_THEME_CHANGED,
    payload: {
      url: getServerUrl(),
      customTheme
    }
  });
};

const setUserThemeAppearance = themeAppearance => {
  dispatch({
    type: WEBVIEW_USER_THEME_APPEARANCE_CHANGED,
    payload: {
      url: getServerUrl(),
      themeAppearance
    }
  });
};

const setTitle = title => {
  if (typeof title !== 'string') {
    return;
  }
  const url = getServerUrl();
  if (title === 'Rocket.Chat' && new URL(url).host !== 'open.rocket.chat') {
    dispatch({
      type: WEBVIEW_TITLE_CHANGED,
      payload: {
        url,
        title: `${title} - ${url}`
      }
    });
    return;
  }
  dispatch({
    type: WEBVIEW_TITLE_CHANGED,
    payload: {
      url,
      title
    }
  });
};

const setUserLoggedIn = userLoggedIn => {
  dispatch({
    type: WEBVIEW_USER_LOGGED_IN,
    payload: {
      url: getServerUrl(),
      userLoggedIn
    }
  });
};

let serverInfo;
let cb = _serverInfo => undefined;
const RocketChatDesktop = {
  onReady: c => {
    if (serverInfo) {
      c(serverInfo);
    }
    cb = c;
  },
  setServerInfo: _serverInfo => {
    serverInfo = _serverInfo;
    cb(_serverInfo);
    setServerVersionToSidebar(_serverInfo.version);
  },
  setUrlResolver,
  setBadge,
  setFavicon,
  setBackground,
  setTitle,
  setUserPresenceDetection,
  setUserLoggedIn,
  setUserThemeAppearance,
  createNotification,
  destroyNotification,
  getInternalVideoChatWindowEnabled,
  openInternalVideoChatWindow,
  setGitCommitHash,
  writeTextToClipboard,
  getOutlookEvents,
  setOutlookExchangeUrl,
  hasOutlookCredentials,
  clearOutlookCredentials,
  setUserToken,
  setSidebarCustomTheme,
  openDocumentViewer,
  reloadServer
};

function debounce(cb, wait = 20) {
  let h;
  const callable = (...args) => {
    h && clearTimeout(h);
    h = setTimeout(() => cb(...args), wait);
  };
  return callable;
}

let focusedMessageBoxInput = null;
const handleFocusEvent = event => {
  if (!(event.target instanceof Element)) {
    return;
  }
  if (!event.target.classList.contains('js-input-message')) {
    return;
  }
  focusedMessageBoxInput = event.target;
  dispatch({
    type: WEBVIEW_MESSAGE_BOX_FOCUSED
  });
};
const handleBlurEvent = event => {
  if (!(event.target instanceof Element)) {
    return;
  }
  if (!event.target.classList.contains('js-input-message')) {
    return;
  }
  focusedMessageBoxInput = null;
  dispatch({
    type: WEBVIEW_MESSAGE_BOX_BLURRED
  });
};
const listenToMessageBoxEvents = () => {
  listen(TOUCH_BAR_FORMAT_BUTTON_TOUCHED, action => {
    if (!focusedMessageBoxInput) {
      return;
    }
    const {
      payload: buttonId
    } = action;
    const ancestor = focusedMessageBoxInput.closest('.rc-message-box');
    const button = ancestor === null || ancestor === void 0 ? void 0 : ancestor.querySelector(`[data-id='${buttonId}']`);
    button === null || button === void 0 || button.click();
  });
  document.addEventListener('focus', handleFocusEvent, true);
  document.addEventListener('blur', handleBlurEvent, true);
};

const selectIsSideBarVisible = ({
  servers,
  isSideBarEnabled
}) => servers.length > 0 && isSideBarEnabled;
const handleTrafficLightsSpacing = () => {
  if (process.platform !== 'darwin') {
    return;
  }
  const style = document.getElementById('sidebar-padding') || document.createElement('style');
  style.id = 'sidebar-padding';
  document.head.append(style);
  watch(selectIsSideBarVisible, isSideBarVisible => {
    style.innerHTML = `
      .sidebar {
        padding-top: ${isSideBarVisible ? 0 : '10px'} !important;
        transition: padding-top 230ms ease-in-out !important;
      }
    `;
  });
};

const whenReady = () => new Promise(resolve => {
  if (document.readyState === 'complete') {
    resolve();
    return;
  }
  const handleReadyStateChange = () => {
    if (document.readyState !== 'complete') {
      return;
    }
    document.removeEventListener('readystatechange', handleReadyStateChange);
    resolve();
  };
  document.addEventListener('readystatechange', handleReadyStateChange);
});

console.log('[Rocket.Chat Desktop] Preload.ts');
electron.contextBridge.exposeInMainWorld('JitsiMeetElectron', JitsiMeetElectron);
electron.contextBridge.exposeInMainWorld('RocketChatDesktop', RocketChatDesktop);
let retryCount = 0;
const start = async () => {
  console.log('[Rocket.Chat Desktop] Preload.ts start fired');
  const serverUrl = await invoke('server-view/get-url');
  if (retryCount > 5) return;
  if (!serverUrl) {
    console.log('[Rocket.Chat Desktop] serverUrl is not defined');
    console.log('[Rocket.Chat Desktop] Preload start - retrying in 1 seconds');
    setTimeout(start, 1000);
    retryCount += 1;
    return;
  }
  window.removeEventListener('load', start);
  setServerUrl(serverUrl);
  await whenReady();
  await createRendererReduxStore();
  await invoke('server-view/ready');
  console.log('[Rocket.Chat Desktop] waiting for RocketChatDesktop.onReady');
  RocketChatDesktop.onReady(() => {
    console.log('[Rocket.Chat Desktop] RocketChatDesktop.onReady fired');
    listen(WEBVIEW_DID_NAVIGATE, debounce(() => {
      const resources = electron.webFrame.getResourceUsage();
      // TODO: make this configurable
      if (resources.images.size > 50 * 1024 * 1024) {
        electron.webFrame.clearCache();
      }
    }, 1000 * 30));
    listenToNotificationsRequests();
    listenToScreenSharingRequests();
    listenToMessageBoxEvents();
    handleTrafficLightsSpacing();
  });
};
console.log('[Rocket.Chat Desktop] waiting for window load');
window.addEventListener('load', start);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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
