"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ViewBase = void 0;
const PropertySet_1 = require("../Core/PropertySet");
const XmlAttributeNames_1 = require("../Core/XmlAttributeNames");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
/**
 * Represents the base view class for search operations.
 */
class ViewBase {
    /**
     * @internal Initializes a new instance of the **ViewBase** class.
     */
    constructor() {
        this.propertySet = null;
    }
    /**
     * Gets or sets the property set. PropertySet determines which properties will be loaded on found items. If PropertySet is null, all first class properties are loaded on found items.
     */
    get PropertySet() {
        return this.propertySet;
    }
    set PropertySet(value) {
        this.propertySet = value;
    }
    /**
     * @internal Gets the property set or the default.
     *
     * @return  {PropertySet}      PropertySet
     */
    GetPropertySetOrDefault() { return this.PropertySet || PropertySet_1.PropertySet.FirstClassProperties; }
    /**
     * @internal Gets the type of service object this view applies to.
     *
     * @return  {ServiceObjectType}      A ServiceObjectType value.
     */
    GetServiceObjectType() { throw new Error("abstract - ViewBase.ts - GetServiceObjectType : Not implemented."); }
    /**
     * @internal Validates this view.
     *
     * @param   {ServiceRequestBase}   request   The request using this view.
     */
    InternalValidate(request) {
        if (this.PropertySet !== null && typeof this.PropertySet !== 'undefined') {
            this.PropertySet.InternalValidate();
            this.PropertySet.ValidateForRequest(request, true /*summaryPropertiesOnly*/);
        }
    }
    /**
     * @internal Writes this view to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    InternalWriteViewToXml(writer) {
        var maxEntriesReturned = this.GetMaxEntriesReturned();
        if (!isNaN(maxEntriesReturned)) {
            writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.MaxEntriesReturned, maxEntriesReturned);
        }
    }
    /**
     * @internal Writes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer    The writer.
     * @param   {Grouping}              groupBy   The group by clause.
     */
    WriteToXml(writer, groupBy) {
        this.GetPropertySetOrDefault().WriteToXml(writer, this.GetServiceObjectType());
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Messages, this.GetViewXmlElementName());
        this.InternalWriteViewToXml(writer);
        writer.WriteEndElement(); // this.GetViewXmlElementName()
        this.InternalWriteSearchSettingsToXml(writer, groupBy);
    }
}
exports.ViewBase = ViewBase;
//# sourceMappingURL=ViewBase.js.map