"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SeekToConditionItemView = void 0;
const ArgumentException_1 = require("../Exceptions/ArgumentException");
const ItemTraversal_1 = require("../Enumerations/ItemTraversal");
const OffsetBasePoint_1 = require("../Enumerations/OffsetBasePoint");
const OrderByCollection_1 = require("./OrderByCollection");
const ServiceObjectType_1 = require("../Enumerations/ServiceObjectType");
const Strings_1 = require("../Strings");
const XmlAttributeNames_1 = require("../Core/XmlAttributeNames");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
const ViewBase_1 = require("./ViewBase");
/**
 * Represents the view settings in a folder search operation.
 *
 * @sealed
 */
class SeekToConditionItemView extends ViewBase_1.ViewBase {
    constructor(condition, pageSize, offsetBasePoint = OffsetBasePoint_1.OffsetBasePoint.Beginning) {
        super();
        this.pageSize = null;
        this.traversal = ItemTraversal_1.ItemTraversal.Shallow;
        this.condition = null;
        this.offsetBasePoint = OffsetBasePoint_1.OffsetBasePoint.Beginning;
        this.orderBy = new OrderByCollection_1.OrderByCollection();
        this.serviceObjType = ServiceObjectType_1.ServiceObjectType.Folder;
        this.condition = condition;
        this.pageSize = pageSize;
        this.serviceObjType = ServiceObjectType_1.ServiceObjectType.Item;
        this.offsetBasePoint = offsetBasePoint;
    }
    /**
     * The maximum number of items or folders the search operation should return.
     */
    get PageSize() {
        return this.pageSize;
    }
    set PageSize(value) {
        if (value <= 0) {
            throw new ArgumentException_1.ArgumentException(Strings_1.Strings.ValueMustBeGreaterThanZero);
        }
        this.pageSize = value;
    }
    /**
     * Gets or sets the base point of the offset.
     */
    get OffsetBasePoint() {
        return this.offsetBasePoint;
    }
    set OffsetBasePoint(value) {
        this.offsetBasePoint = value;
    }
    /**
     * Gets or sets the condition for seek.
     * Available search filter classes include SearchFilter.IsEqualTo, SearchFilter.ContainsSubstring and SearchFilter.SearchFilterCollection.
     * If SearchFilter is null, no search filters are applied.
     */
    get Condition() {
        return this.condition;
    }
    set Condition(value) {
        if (value === null) {
            throw new ArgumentException_1.ArgumentNullException("Condition");
        }
        this.condition = value;
    }
    /**
     * Gets or sets the search traversal mode. Defaults to ItemTraversal.Shallow.
     */
    get Traversal() {
        return this.traversal;
    }
    set Traversal(value) {
        this.traversal = value;
    }
    /**
     * Gets the properties against which the returned items should be ordered.
     */
    get OrderBy() {
        return this.orderBy;
    }
    /**
     * @internal Gets the maximum number of items or folders the search operation should return.
     *
     * @return  {number?}      The maximum number of items or folders that should be returned by the search operation.
     */
    GetMaxEntriesReturned() {
        return this.PageSize;
    }
    /**
     * @internal Gets the type of service object this view applies to.
     *
     * @return  {ServiceObjectType}      A ServiceObjectType value.
     */
    GetServiceObjectType() {
        return this.serviceObjType;
    }
    /**
     * @internal Gets the name of the view XML element.
     *
     * @return  {string}      XML element name.
     */
    GetViewXmlElementName() {
        return XmlElementNames_1.XmlElementNames.SeekToConditionPageItemView;
    }
    /**
     * @internal Validates this view.
     *
     * @param   {ServiceRequestBase}   request   The request using this view.
     */
    InternalValidate(request) {
        super.InternalValidate(request);
    }
    /**
     * @internal Writes the search settings to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer    The writer.
     * @param   {Grouping}              groupBy   The group by clause.
     */
    InternalWriteSearchSettingsToXml(writer, groupBy) {
        if (groupBy != null) {
            groupBy.WriteToXml(writer);
        }
    }
    /**
     * @internal Writes this view to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    InternalWriteViewToXml(writer) {
        super.InternalWriteViewToXml(writer);
        writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.BasePoint, this.OffsetBasePoint);
        if (this.Condition != null) {
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Condition);
            this.Condition.WriteToXml(writer);
            writer.WriteEndElement(); // Restriction
        }
    }
    /**
     * @internal Sets the type of service object this view applies to.
     *
     * @return  {ServiceObjectType}      Service object type
     */
    SetServiceObjectType(objType) {
        this.serviceObjType = objType;
    }
    /**
     * @internal Writes the attributes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteAttributesToXml(writer) {
        if (this.serviceObjType == ServiceObjectType_1.ServiceObjectType.Item) {
            writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.Traversal, this.Traversal);
        }
    }
    /**
     * @internal Writes OrderBy property to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteOrderByToXml(writer) {
        this.orderBy.WriteToXml(writer, XmlElementNames_1.XmlElementNames.SortOrder);
    }
    /**
     * @internal Writes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer    The writer.
     * @param   {Grouping}              groupBy   The group by clause.
     */
    WriteToXml(writer, groupBy) {
        if (this.serviceObjType == ServiceObjectType_1.ServiceObjectType.Item) {
            this.GetPropertySetOrDefault().WriteToXml(writer, this.GetServiceObjectType());
        }
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Messages, this.GetViewXmlElementName());
        this.InternalWriteViewToXml(writer);
        writer.WriteEndElement(); // this.GetViewXmlElementName()
    }
}
exports.SeekToConditionItemView = SeekToConditionItemView;
//# sourceMappingURL=SeekToConditionItemView.js.map