"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OrderByCollection = void 0;
const ServiceLocalException_1 = require("../Exceptions/ServiceLocalException");
const SortDirection_1 = require("../Enumerations/SortDirection");
const ExtensionMethods_1 = require("../ExtensionMethods");
const Strings_1 = require("../Strings");
const XmlAttributeNames_1 = require("../Core/XmlAttributeNames");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
/**
 * Represents an ordered collection of property definitions qualified with a sort direction.
 *
 * @sealed
 */
class OrderByCollection {
    /**
     * @internal Initializes a new instance of the **OrderByCollection** class.
     */
    constructor() {
        this.propDefSortOrderPairList = [];
    }
    /**
     * Gets the number of elements contained in the collection.
     */
    get Count() {
        return this.propDefSortOrderPairList.length;
    }
    /**
     * Gets the element at the specified index from the collection.
     *
     * @param   {number}   index   Index.
     */
    _getItem(index) {
        return this.propDefSortOrderPairList[index];
    }
    /**
     * Adds the specified property definition / sort direction pair to the collection.
     *
     * @param   {PropertyDefinitionBase}    propertyDefinition   The property definition.
     * @param   {SortDirection}             sortDirection        The sort direction.
     */
    Add(propertyDefinition, sortDirection) {
        if (this.Contains(propertyDefinition)) {
            throw new ServiceLocalException_1.ServiceLocalException(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.PropertyAlreadyExistsInOrderByCollection, propertyDefinition.GetPrintableName()));
        }
        this.propDefSortOrderPairList.push({ key: propertyDefinition, value: sortDirection }); //new PropertyDefinitionSortDirectionPair() not seamless in javascript
    }
    /**
     * Removes all elements from the collection.
     */
    Clear() {
        this.propDefSortOrderPairList.splice(0);
    }
    /**
     * @internal Determines whether the collection contains the specified property definition.
     *
     * @param   {PropertyDefinitionBase}   propertyDefinition   The property definition.
     * @return  {boolean}   True if the collection contains the specified property definition; otherwise, false.
     */
    Contains(propertyDefinition) {
        this.propDefSortOrderPairList.forEach((pair, index) => {
            //debugger;// check if equality works or need to use any property
            if (pair.key === propertyDefinition)
                return true;
        });
        return false;
    }
    /**
     *  Returns an enumerator that iterates through the collection. this case this.propDefSortOrderPairList
     */
    GetEnumerator() {
        return this.propDefSortOrderPairList;
    }
    /**
     * @internal Removes the specified property definition from the collection.
     *
     * @param   {PropertyDefinitionBase}   propertyDefinition   The property definition.
     * @return  {boolean}   True if the property definition is successfully removed; otherwise, false
     */
    Remove(propertyDefinition) {
        var oldCount = this.Count;
        this.propDefSortOrderPairList = this.propDefSortOrderPairList.filter((value) => { return value.key !== propertyDefinition; });
        return oldCount > this.Count;
    }
    /**
     * @internal Removes the element at the specified index from the collection.
     *
     * @param   {number}   index   The index.
     */
    RemoveAt(index) {
        this.propDefSortOrderPairList.splice(index, 1);
    }
    /**
     * Tries to get the value for a property definition in the collection.
     *
     * @param   {PropertyDefinitionBase}    propertyDefinition   The property definition.
     * @param   {IOutParam<SortDirection>}  sortDirection        The sort direction.
     * @return  {boolean}                   True if collection contains property definition, otherwise false.
     */
    TryGetValue(propertyDefinition, sortDirection) {
        for (var pair of this.propDefSortOrderPairList) {
            if (pair.key == propertyDefinition) { //possible bug - log at Github
                sortDirection.outValue = pair.value;
                return true;
            }
        }
        sortDirection.outValue = SortDirection_1.SortDirection.Ascending; // out parameter has to be set to some value.
        return false;
    }
    /**
     * @internal Writes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     * @param   {string}                xmlElementName   Name of the XML element.
     */
    WriteToXml(writer, xmlElementName) {
        if (this.Count > 0) {
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Messages, xmlElementName);
            for (var keyValuePair of this.propDefSortOrderPairList) {
                writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.FieldOrder);
                writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.Order, SortDirection_1.SortDirection[keyValuePair.value]);
                keyValuePair.key.WriteToXml(writer);
                writer.WriteEndElement(); // FieldOrder
            }
            writer.WriteEndElement();
        }
    }
}
exports.OrderByCollection = OrderByCollection;
//# sourceMappingURL=OrderByCollection.js.map