"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NameResolutionCollection = void 0;
const ArgumentException_1 = require("../Exceptions/ArgumentException");
const ExtensionMethods_1 = require("../ExtensionMethods");
const EwsLogging_1 = require("../Core/EwsLogging");
const EwsServiceJsonReader_1 = require("../Core/EwsServiceJsonReader");
const NameResolution_1 = require("./NameResolution");
const Strings_1 = require("../Strings");
const XmlAttributeNames_1 = require("../Core/XmlAttributeNames");
const XmlElementNames_1 = require("../Core/XmlElementNames");
/**
 * Represents a list of suggested name resolutions.
 *
 * @sealed
 */
class NameResolutionCollection {
    /**
     * @internal Initializes a new instance of the **NameResolutionCollection** class.
     *
     * @param   {ExchangeService}   service   The service.
     */
    constructor(service) {
        this.service = null;
        this.includesAllResolutions = false;
        this.items = [];
        EwsLogging_1.EwsLogging.Assert(service !== null, "NameResolutionSet.ctor", "service is null.");
        this.service = service;
    }
    /**
     * @internal Gets the session.
     *
     * @value   The Session.
     */
    get Session() {
        return this.service;
    }
    /**
     * Gets the total number of elements in the list.
     */
    get Count() {
        return this.items.length;
    }
    /**
     * Gets a value indicating whether more suggested resolutions are available. ResolveName only returns a maximum of 100 name resolutions.
     * When IncludesAllResolutions is false, there were more than 100 matching names on the server. To narrow the search, provide a more precise name to ResolveName.
     */
    get IncludesAllResolutions() {
        return this.includesAllResolutions;
    }
    /**
     * Gets the name resolution at the specified index.
     *
     * @param   {number}   index    The index of the name resolution to get.
     * @return  {NameResolution}    The name resolution at the speicfied index.
     */
    _getItem(index) {
        if (index < 0 || index >= this.Count) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("index", Strings_1.Strings.IndexIsOutOfRange);
        }
        return this.items[index];
    }
    /**
     *  Returns an enumerator that iterates through the collection. this case this.items
     */
    GetEnumerator() {
        return this.items;
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}				jsObject	Json Object converted from XML.
     * @param   {ExchangeService}	service	The service.
     */
    LoadFromXmlJsObject(jsonProperty, service) {
        var totalItemsInView;
        var resolutions;
        for (var key in jsonProperty) {
            switch (key) {
                case XmlAttributeNames_1.XmlAttributeNames.TotalItemsInView:
                    totalItemsInView = ExtensionMethods_1.Convert.toNumber(jsonProperty[key]);
                    break;
                case XmlAttributeNames_1.XmlAttributeNames.IncludesLastItemInRange:
                    this.includesAllResolutions = ExtensionMethods_1.Convert.toBool(jsonProperty[key]);
                    break;
                // This label only exists for Json objects.  The XML doesn't have a "Resolutions"
                // element.  
                // This was necessary becaue of the lack of attributes in JSON
                //
                case XmlElementNames_1.XmlElementNames.Resolution:
                    resolutions = EwsServiceJsonReader_1.EwsServiceJsonReader.ReadAsArray(jsonProperty, key);
                    for (var resolution of resolutions) {
                        var nameResolution = new NameResolution_1.NameResolution(this);
                        nameResolution.LoadFromXmlJsObject(resolution, service);
                        this.items.push(nameResolution);
                    }
                    break;
                default:
                    break;
            }
        }
    }
}
exports.NameResolutionCollection = NameResolutionCollection;
//# sourceMappingURL=NameResolutionCollection.js.map