"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResponseObject = void 0;
const EwsLogging_1 = require("../../EwsLogging");
const FolderId_1 = require("../../../ComplexProperties/FolderId");
const MessageDisposition_1 = require("../../../Enumerations/MessageDisposition");
const NotSupportedException_1 = require("../../../Exceptions/NotSupportedException");
const Schemas_1 = require("../Schemas/Schemas");
const ServiceObject_1 = require("../ServiceObject");
/**
 * Represents the base class for all responses that can be sent.
 *
 * @typeparam   {TMessage}     Type of message.
 */
class ResponseObject extends ServiceObject_1.ServiceObject {
    /**
     * @internal Initializes a new instance of the **ResponseObject** class.
     *
     * @param   {type}   referenceItem   The reference item.
     */
    constructor(referenceItem) {
        super(referenceItem.Service);
        this.referenceItem = null;
        EwsLogging_1.EwsLogging.Assert(referenceItem !== null, "ResponseObject.ctor", "referenceItem is null");
        referenceItem.ThrowIfThisIsNew();
        this.referenceItem = referenceItem;
    }
    /**
     * Gets or sets a value indicating whether read receipts will be requested from recipients of this response.
     */
    get IsReadReceiptRequested() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.IsReadReceiptRequested);
    }
    set IsReadReceiptRequested(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.EmailMessageSchema.IsReadReceiptRequested, value);
    }
    /**
     * Gets or sets a value indicating whether delivery receipts should be sent to the sender.
     */
    get IsDeliveryReceiptRequested() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.IsDeliveryReceiptRequested);
    }
    set IsDeliveryReceiptRequested(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.EmailMessageSchema.IsDeliveryReceiptRequested, value);
    }
    /**
     * @internal Internal method to return the schema associated with this type of object.
     *
     * @return  {ServiceObjectSchema}      The schema associated with this type of object.
     */
    GetSchema() { return Schemas_1.Schemas.ResponseObjectSchema.Instance; }
    /**
    * @internal Create the response object.
    *
    * @param   {FolderId}             destinationFolderId   The destination folder id.
    * @param   {MessageDisposition}   messageDisposition    The message disposition.
    * @return  {Promise<Item[]>}               The list of items returned by EWS.
    */
    InternalCreate(destinationFolderId, messageDisposition) {
        this.PropertyBag._getItem(Schemas_1.Schemas.ResponseObjectSchema.ReferenceItemId).Assign(this.referenceItem.Id);
        return this.Service.InternalCreateResponseObject(this, destinationFolderId, messageDisposition);
    }
    /**
     * @internal Deletes the object.
     *
     * @param   {DeleteMode}                  deleteMode                The deletion mode.
     * @param   {SendCancellationsMode}       sendCancellationsMode     Indicates whether meeting cancellation messages should be sent.
     * @param   {affectedTaskOccurrences}     affectedTaskOccurrences   Indicate which occurrence of a recurring task should be deleted.
     */
    InternalDelete(deleteMode, sendCancellationsMode, affectedTaskOccurrences) {
        throw new NotSupportedException_1.NotSupportedException();
    }
    /**
     * @internal Loads the specified set of properties on the object.
     *
     * @param   {PropertySet}   propertySet   The properties to load.
     */
    InternalLoad(propertySet) {
        throw new NotSupportedException_1.NotSupportedException();
    }
    Save(destinationFolderIdOrName) {
        var destinationFolderId = null;
        if (arguments.length === 1) {
            if (typeof destinationFolderIdOrName === "number") {
                destinationFolderId = new FolderId_1.FolderId(destinationFolderIdOrName);
            }
            else {
                //EwsUtilities.ValidateParam(destinationFolderIdOrName, "destinationFolderId");
                destinationFolderId = destinationFolderIdOrName;
            }
        }
        return this.InternalCreate(destinationFolderId, MessageDisposition_1.MessageDisposition.SaveOnly).then((result) => {
            return result[0];
        });
    }
    /**
     * Sends this response without saving a copy. Calling this method results in a call to EWS.
     */
    Send() {
        return this.InternalCreate(null, MessageDisposition_1.MessageDisposition.SendOnly);
    }
    SendAndSaveCopy(destinationFolderIdOrName) {
        var destinationFolderId = null;
        if (arguments.length === 1) {
            if (typeof destinationFolderIdOrName === "number") {
                destinationFolderId = new FolderId_1.FolderId(destinationFolderIdOrName);
            }
            else {
                //EwsUtilities.ValidateParam(destinationFolderIdOrName, "destinationFolderId");
                destinationFolderId = destinationFolderIdOrName;
            }
        }
        return this.InternalCreate(destinationFolderId, MessageDisposition_1.MessageDisposition.SendAndSaveCopy);
    }
}
exports.ResponseObject = ResponseObject;
//# sourceMappingURL=ResponseObject.js.map