"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PostItem = void 0;
const ExtensionMethods_1 = require("../../../ExtensionMethods");
const ExchangeVersion_1 = require("../../../Enumerations/ExchangeVersion");
const PostReply_1 = require("../ResponseObjects/PostReply");
const PropertySet_1 = require("../../PropertySet");
const ResponseMessage_1 = require("../ResponseObjects/ResponseMessage");
const ResponseMessageType_1 = require("../../../Enumerations/ResponseMessageType");
const Schemas_1 = require("../Schemas/Schemas");
const XmlElementNames_1 = require("../../XmlElementNames");
const Item_1 = require("./Item");
/**
 * Represents a post item. Properties available on post items are defined in the PostItemSchema class.
 *
 * @sealed
 */
class PostItem extends Item_1.Item {
    /** required to check [Attachable] attribute, AttachmentCollection.AddItemAttachment<TItem>() checks for non inherited [Attachable] attribute. */
    static get Attachable() { return this.name === "PostItem"; }
    ;
    /**
     * Gets the conversation index of the post item.
     */
    get ConversationIndex() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.ConversationIndex);
    }
    /**
     * Gets the conversation topic of the post item.
     */
    get ConversationTopic() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.ConversationTopic);
    }
    /**
     * Gets or sets the "on behalf" poster of the post item.
     */
    get From() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.From);
    }
    set From(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.EmailMessageSchema.From, value);
    }
    /**
     * Gets the Internet message Id of the post item.
     */
    get InternetMessageId() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.InternetMessageId);
    }
    /**
     * Gets or sets a value indicating whether the post item is read.
     */
    get IsRead() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.IsRead);
    }
    set IsRead(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.EmailMessageSchema.IsRead, value);
    }
    /**
     * Gets the the date and time when the post item was posted.
     */
    get PostedTime() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.PostItemSchema.PostedTime);
    }
    /**
     * Gets or sets the references of the post item.
     */
    get References() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.References);
    }
    set References(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.EmailMessageSchema.References, value);
    }
    /**
     * Gets or sets the sender (poster) of the post item.
     */
    get Sender() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.EmailMessageSchema.Sender);
    }
    set Sender(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.EmailMessageSchema.Sender, value);
    }
    constructor(serviceOrParentAttachment) {
        super(serviceOrParentAttachment);
    }
    static Bind(service, id, propertySet = PropertySet_1.PropertySet.FirstClassProperties) {
        return service.BindToItem(id, propertySet, PostItem);
    }
    /**
     * Creates a forward response to the post item.
     *
     * @return  {ResponseMessage}      A ResponseMessage representing the forward response that can subsequently be modified and sent.
     */
    CreateForward() {
        this.ThrowIfThisIsNew();
        return new ResponseMessage_1.ResponseMessage(this, ResponseMessageType_1.ResponseMessageType.Forward);
    }
    /**
     * Creates a post reply to this post item.
     *
     * @return  {PostReply}      A PostReply that can be modified and saved.
     */
    CreatePostReply() {
        this.ThrowIfThisIsNew();
        return new PostReply_1.PostReply(this);
    }
    /**
     * Creates a e-mail reply response to the post item.
     *
     * @param   {boolean}   replyAll   Indicates whether the reply should go to everyone involved in the thread.
     * @return  {ResponseMessage}      A ResponseMessage representing the e-mail reply response that can subsequently be modified and sent.
     */
    CreateReply(replyAll) {
        this.ThrowIfThisIsNew();
        return new ResponseMessage_1.ResponseMessage(this, replyAll ? ResponseMessageType_1.ResponseMessageType.ReplyAll : ResponseMessageType_1.ResponseMessageType.Reply);
    }
    Forward(bodyPrefix, _toRecipients) {
        let toRecipients = [];
        if (arguments.length <= 2) {
            if (ExtensionMethods_1.ArrayHelper.isArray(_toRecipients)) {
                toRecipients = _toRecipients;
            }
            else {
                toRecipients.push(arguments[1]);
            }
        }
        else {
            for (var _i = 1; _i < arguments.length; _i++) {
                toRecipients[_i - 1] = arguments[_i];
            }
        }
        let responseMessage = this.CreateForward();
        responseMessage.BodyPrefix = bodyPrefix;
        responseMessage.ToRecipients.AddRange(toRecipients);
        return responseMessage.SendAndSaveCopy();
    }
    /**
     * @internal Gets the minimum required server version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this service object type is supported.
     */
    GetMinimumRequiredServerVersion() {
        return ExchangeVersion_1.ExchangeVersion.Exchange2007_SP1;
    }
    /**
     * @internal Internal method to return the schema associated with this type of object.
     *
     * @return  {ServiceObjectSchema}      The schema associated with this type of object.
     */
    GetSchema() {
        return Schemas_1.Schemas.PostItemSchema.Instance;
    }
    /**
     * @internal Gets the element name of item in XML
     *
     * @return  {string} name of elelment
     */
    GetXmlElementName() {
        return XmlElementNames_1.XmlElementNames.PostItem;
    }
    /**
     * Posts a reply to this post item. Calling this method results in a call to EWS.
     *
     * @param   {MessageBody}   bodyPrefix   Body prefix.
     * @return  {Promise<void>}    :Promise.
     */
    PostReply(bodyPrefix) {
        let postReply = this.CreatePostReply();
        postReply.BodyPrefix = bodyPrefix;
        return postReply.Save();
    }
    /**
     * Replies to the post item. Calling this method results in a call to EWS.
     *
     * @param   {MessageBody}   bodyPrefix   The prefix to prepend to the original body of the post item.
     * @param   {boolean}       replyAll     Indicates whether the reply should be sent to everyone involved in the thread.
     * @return  {Promise<void>}    :Promise.
     */
    Reply(bodyPrefix, replyAll) {
        let responseMessage = this.CreateReply(replyAll);
        responseMessage.BodyPrefix = bodyPrefix;
        return responseMessage.SendAndSaveCopy();
    }
}
exports.PostItem = PostItem;
//# sourceMappingURL=PostItem.js.map