"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Conversation = void 0;
const ExchangeVersion_1 = require("../../../Enumerations/ExchangeVersion");
const Flag_1 = require("../../../ComplexProperties/Flag");
const ItemFlagStatus_1 = require("../../../Enumerations/ItemFlagStatus");
const NotSupportedException_1 = require("../../../Exceptions/NotSupportedException");
const Schemas_1 = require("../Schemas/Schemas");
const ExtensionMethods_1 = require("../../../ExtensionMethods");
const XmlElementNames_1 = require("../../XmlElementNames");
const ServiceObject_1 = require("../ServiceObject");
/**
 * Represents a collection of Conversation related properties.
 * Properties available on this object are defined in the ConversationSchema class.
 */
class Conversation extends ServiceObject_1.ServiceObject {
    /**
     * Gets the Id of this Conversation.
     */
    get Id() {
        return this.PropertyBag._getItem(this.GetIdPropertyDefinition());
    }
    /**
     * Gets the topic of this Conversation.
     */
    get Topic() {
        var returnValue = { outValue: ExtensionMethods_1.StringHelper.Empty };
        // This property need not be present hence the property bag may not contain it.
        // Check for the presence of this property before accessing it.
        if (this.PropertyBag.Contains(Schemas_1.Schemas.ConversationSchema.Topic)) {
            this.PropertyBag.TryGetPropertyAs(Schemas_1.Schemas.ConversationSchema.Topic, returnValue);
        }
        return returnValue.outValue;
    }
    /**
     * Gets a list of all the people who have received messages in this conversation in the current folder only.
     */
    get UniqueRecipients() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.UniqueRecipients);
    }
    /**
     * Gets a list of all the people who have received messages in this conversation across all folders in the mailbox.
     */
    get GlobalUniqueRecipients() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.GlobalUniqueRecipients);
    }
    /**
     * Gets a list of all the people who have sent messages that are currently unread in this conversation in the current folder only.
     */
    get UniqueUnreadSenders() {
        var unreadSenders = { outValue: null };
        // This property need not be present hence the property bag may not contain it.
        // Check for the presence of this property before accessing it.
        if (this.PropertyBag.Contains(Schemas_1.Schemas.ConversationSchema.UniqueUnreadSenders)) {
            this.PropertyBag.TryGetPropertyAs(Schemas_1.Schemas.ConversationSchema.UniqueUnreadSenders, unreadSenders);
        }
        return unreadSenders.outValue;
    }
    /**
     * Gets a list of all the people who have sent messages that are currently unread in this conversation across all folders in the mailbox.
     */
    get GlobalUniqueUnreadSenders() {
        var unreadSenders = { outValue: null };
        // This property need not be present hence the property bag may not contain it.
        // Check for the presence of this property before accessing it.
        if (this.PropertyBag.Contains(Schemas_1.Schemas.ConversationSchema.GlobalUniqueUnreadSenders)) {
            this.PropertyBag.TryGetPropertyAs(Schemas_1.Schemas.ConversationSchema.GlobalUniqueUnreadSenders, unreadSenders);
        }
        return unreadSenders.outValue;
    }
    /**
     * Gets a list of all the people who have sent messages in this conversation in the current folder only.
     */
    get UniqueSenders() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.UniqueSenders);
    }
    /**
     * Gets a list of all the people who have sent messages in this conversation across all folders in the mailbox.
     */
    get GlobalUniqueSenders() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.GlobalUniqueSenders);
    }
    /**
     * Gets the delivery time of the message that was last received in this conversation in the current folder only.
     */
    get LastDeliveryTime() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.LastDeliveryTime);
    }
    /**
     * Gets the delivery time of the message that was last received in this conversation across all folders in the mailbox.
     */
    get GlobalLastDeliveryTime() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.GlobalLastDeliveryTime);
    }
    /**
     * Gets a list summarizing the categories stamped on messages in this conversation, in the current folder only.
     */
    get Categories() {
        var returnValue = { outValue: null };
        // This property need not be present hence the property bag may not contain it.
        // Check for the presence of this property before accessing it.
        if (this.PropertyBag.Contains(Schemas_1.Schemas.ConversationSchema.Categories)) {
            this.PropertyBag.TryGetPropertyAs(Schemas_1.Schemas.ConversationSchema.Categories, returnValue);
        }
        return returnValue.outValue;
    }
    /**
     * Gets a list summarizing the categories stamped on messages in this conversation, across all folders in the mailbox.
     */
    get GlobalCategories() {
        var returnValue = { outValue: null };
        // This property need not be present hence the property bag may not contain it.
        // Check for the presence of this property before accessing it.
        if (this.PropertyBag.Contains(Schemas_1.Schemas.ConversationSchema.GlobalCategories)) {
            this.PropertyBag.TryGetPropertyAs(Schemas_1.Schemas.ConversationSchema.GlobalCategories, returnValue);
        }
        return returnValue.outValue;
    }
    /**
     * Gets the flag status for this conversation, calculated by aggregating individual messages flag status in the current folder.
     */
    get FlagStatus() {
        var returnValue = { outValue: null };
        // This property need not be present hence the property bag may not contain it.
        // Check for the presence of this property before accessing it.
        if (this.PropertyBag.Contains(Schemas_1.Schemas.ConversationSchema.FlagStatus)) {
            this.PropertyBag.TryGetPropertyAs(Schemas_1.Schemas.ConversationSchema.FlagStatus, returnValue);
        }
        return returnValue.outValue;
    }
    /**
     * Gets the flag status for this conversation, calculated by aggregating individual messages flag status across all folders in the mailbox.
     */
    get GlobalFlagStatus() {
        var returnValue = { outValue: null };
        // This property need not be present hence the property bag may not contain it.
        // Check for the presence of this property before accessing it.
        if (this.PropertyBag.Contains(Schemas_1.Schemas.ConversationSchema.GlobalFlagStatus)) {
            this.PropertyBag.TryGetPropertyAs(Schemas_1.Schemas.ConversationSchema.GlobalFlagStatus, returnValue);
        }
        return returnValue.outValue;
    }
    /**
     * Gets a value indicating if at least one message in this conversation, in the current folder only, has an attachment.
     */
    get HasAttachments() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.HasAttachments);
    }
    /**
     * Gets a value indicating if at least one message in this conversation, across all folders in the mailbox, has an attachment.
     */
    get GlobalHasAttachments() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.GlobalHasAttachments);
    }
    /**
     * Gets the total number of messages in this conversation in the current folder only.
     */
    get MessageCount() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.MessageCount);
    }
    /**
     * Gets the total number of messages in this conversation across all folders in the mailbox.
     */
    get GlobalMessageCount() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.GlobalMessageCount);
    }
    /**
     * Gets the total number of unread messages in this conversation in the current folder only.
     */
    get UnreadCount() {
        var returnValue = { outValue: 0 };
        // This property need not be present hence the property bag may not contain it.
        // Check for the presence of this property before accessing it.
        if (this.PropertyBag.Contains(Schemas_1.Schemas.ConversationSchema.UnreadCount)) {
            this.PropertyBag.TryGetPropertyAs(Schemas_1.Schemas.ConversationSchema.UnreadCount, returnValue);
        }
        return returnValue.outValue;
    }
    /**
     * Gets the total number of unread messages in this conversation across all folders in the mailbox.
     */
    get GlobalUnreadCount() {
        var returnValue = { outValue: 0 };
        // This property need not be present hence the property bag may not contain it.
        // Check for the presence of this property before accessing it.
        if (this.PropertyBag.Contains(Schemas_1.Schemas.ConversationSchema.GlobalUnreadCount)) {
            this.PropertyBag.TryGetPropertyAs(Schemas_1.Schemas.ConversationSchema.GlobalUnreadCount, returnValue);
        }
        return returnValue.outValue;
    }
    /**
     * Gets the size of this conversation, calculated by adding the sizes of all messages in the conversation in the current folder only.
     */
    get Size() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.Size);
    }
    /**
     * Gets the size of this conversation, calculated by adding the sizes of all messages in the conversation across all folders in the mailbox.
     */
    get GlobalSize() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.GlobalSize);
    }
    /**
     * Gets a list summarizing the classes of the items in this conversation, in the current folder only.
     */
    get ItemClasses() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.ItemClasses);
    }
    /**
     * Gets a list summarizing the classes of the items in this conversation, across all folders in the mailbox.
     */
    get GlobalItemClasses() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.GlobalItemClasses);
    }
    /**
     * Gets the importance of this conversation, calculated by aggregating individual messages importance in the current folder only.
     */
    get Importance() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.Importance);
    }
    /**
     * Gets the importance of this conversation, calculated by aggregating individual messages importance across all folders in the mailbox.
     */
    get GlobalImportance() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.GlobalImportance);
    }
    /**
     * Gets the Ids of the messages in this conversation, in the current folder only.
     */
    get ItemIds() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.ItemIds);
    }
    /**
     * Gets the Ids of the messages in this conversation, across all folders in the mailbox.
     */
    get GlobalItemIds() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.GlobalItemIds);
    }
    /**
     * Gets the date and time this conversation was last modified.
     */
    get LastModifiedTime() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.LastModifiedTime);
    }
    /**
     * Gets the conversation instance key.
     */
    get InstanceKey() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.InstanceKey);
    }
    /**
     * Gets the conversation Preview.
     */
    get Preview() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.Preview);
    }
    /**
     * Gets the conversation IconIndex.
     */
    get IconIndex() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.IconIndex);
    }
    /**
     * Gets the conversation global IconIndex.
     */
    get GlobalIconIndex() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.GlobalIconIndex);
    }
    /**
     * Gets the draft item ids.
     */
    get DraftItemIds() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.DraftItemIds);
    }
    /**
     * Gets a value indicating if at least one message in this conversation, in the current folder only, is an IRM.
     */
    get HasIrm() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.HasIrm);
    }
    /**
     * Gets a value indicating if at least one message in this conversation, across all folders in the mailbox, is an IRM.
     */
    get GlobalHasIrm() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.ConversationSchema.GlobalHasIrm);
    }
    /**
     * @internal Initializes an unsaved local instance of **Conversation** class.
     *
     * @param   {ExchangeService}   service   The ExchangeService object to which the item will be bound.
     */
    constructor(service) {
        super(service);
    }
    /**
     * Clear flags for conversation items. Calling this method results in a call to EWS.
     *
     * @param   {FolderId}   contextFolderId   The Id of the folder items must belong to in order to be unflagged. If contextFolderId is null, flags for items in conversation across the entire mailbox are cleared.
     * @return  {Promise<void>}    Promise
     */
    ClearItemFlags(contextFolderId) {
        let flag = new Flag_1.Flag();
        flag.FlagStatus = ItemFlagStatus_1.ItemFlagStatus.NotFlagged;
        return this.Service.SetFlagStatusForItemsInConversations([{ key: this.Id, value: this.GlobalLastDeliveryTime }], contextFolderId, flag).then((responses) => {
            responses.__thisIndexer(0).ThrowIfNecessary();
        });
    }
    /**
     * Copies items in the specified conversation to a specific folder. Calling this method results in a call to EWS.
     *
     * @param   {FolderId}   contextFolderId       The Id of the folder items must belong to in order to be copied. If contextFolderId is null, items across the entire mailbox are copied.
     * @param   {FolderId}   destinationFolderId   The Id of the destination folder.
     * @return  {Promise<void>}    Promise
     */
    CopyItemsInConversation(contextFolderId, destinationFolderId) {
        return this.Service.CopyItemsInConversations([{ key: this.Id, value: this.GlobalLastDeliveryTime }], contextFolderId, destinationFolderId).then((responses) => {
            responses.__thisIndexer(0).ThrowIfNecessary();
        });
    }
    /**
     * Deletes items in the specified conversation.
     * Calling this method results in a call to EWS.
     *
     * @param   {FolderId}      contextFolderId   The Id of the folder items must belong to in order to be deleted. If contextFolderId is null, items across the entire mailbox are deleted.
     * @param   {DeleteMode}    deleteMode        The deletion mode.
     * @return  {Promise<void>}    Promise
     */
    DeleteItems(contextFolderId, deleteMode) {
        return this.Service.DeleteItemsInConversations([{ key: this.Id, value: this.GlobalLastDeliveryTime }], contextFolderId, deleteMode).then((responses) => {
            responses.__thisIndexer(0).ThrowIfNecessary();
        });
    }
    /**
     * Sets up a conversation so that any item received within that conversation is no longer categorized.
     * Calling this method results in a call to EWS.
     *
     * @param   {boolean}   processSynchronously   **<not used>**Indicates whether the method should return only once disabling this rule and removing the categories from existing items in the conversation is completely done. If processSynchronously is false, the method returns immediately.
     * @return  {Promise<void>}    Promise
     */
    DisableAlwaysCategorizeItems(processSynchronously) {
        return this.Service.DisableAlwaysCategorizeItemsInConversations([this.Id], processSynchronously).then((responses) => {
            responses.__thisIndexer(0).ThrowIfNecessary();
        });
    }
    /**
     * Sets up a conversation so that any item received within that conversation is no longer moved to Deleted Items folder.
     * Calling this method results in a call to EWS.
     *
     * @param   {boolean}   processSynchronously   Indicates whether the method should return only once disabling this rule and restoring the items in the conversation is completely done. If processSynchronously is false, the method returns immediately.
     * @return  {Promise<void>}    Promise
     */
    DisableAlwaysDeleteItems(processSynchronously) {
        return this.Service.DisableAlwaysDeleteItemsInConversations([this.Id], processSynchronously).then((responses) => {
            responses.__thisIndexer(0).ThrowIfNecessary();
        });
    }
    /**
     * Sets up a conversation so that any item received within that conversation is no longer moved to a specific folder.
     * Calling this method results in a call to EWS.
     *
     * @param   {boolean}   processSynchronously   Indicates whether the method should return only once disabling this rule is completely done. If processSynchronously is false, the method returns immediately.
     * @return  {Promise<void>}    Promise
     */
    DisableAlwaysMoveItemsInConversation(processSynchronously) {
        return this.Service.DisableAlwaysMoveItemsInConversations([this.Id], processSynchronously).then((responses) => {
            responses.__thisIndexer(0).ThrowIfNecessary();
        });
    }
    /**
     * Sets up a conversation so that any item received within that conversation is always categorized.
     * Calling this method results in a call to EWS.
     *
     * @param   {string[]}  categories             The categories that should be stamped on items in the conversation.
     * @param   {boolean}   processSynchronously   Indicates whether the method should return only once enabling this rule and stamping existing items in the conversation is completely done. If processSynchronously is false, the method returns immediately.
     * @return  {Promise<void>}    Promise
     */
    EnableAlwaysCategorizeItems(categories, processSynchronously) {
        return this.Service.EnableAlwaysCategorizeItemsInConversations([this.Id], categories, processSynchronously).then((responses) => {
            responses.__thisIndexer(0).ThrowIfNecessary();
        });
    }
    /**
     * Sets up a conversation so that any item received within that conversation is always moved to Deleted Items folder.
     * Calling this method results in a call to EWS.
     *
     * @param   {boolean}   processSynchronously   Indicates whether the method should return only once enabling this rule and deleting existing items in the conversation is completely done. If processSynchronously is false, the method returns immediately.
     * @return  {Promise<void>}    Promise
     */
    EnableAlwaysDeleteItems(processSynchronously) {
        return this.Service.EnableAlwaysDeleteItemsInConversations([this.Id], processSynchronously).then((responses) => {
            responses.__thisIndexer(0).ThrowIfNecessary();
        });
    }
    /**
     * Sets up a conversation so that any item received within that conversation is always moved to a specific folder.
     * Calling this method results in a call to EWS.
     *
     * @param   {FolderId}  destinationFolderId    The Id of the folder to which conversation items should be moved.
     * @param   {boolean}   processSynchronously   Indicates whether the method should return only once enabling this rule and moving existing items in the conversation is completely done. If processSynchronously is false, the method returns immediately.
     * @return  {Promise<void>}    Promise
     */
    EnableAlwaysMoveItems(destinationFolderId, processSynchronously) {
        return this.Service.EnableAlwaysMoveItemsInConversations([this.Id], destinationFolderId, processSynchronously).then((responses) => {
            responses.__thisIndexer(0).ThrowIfNecessary();
        });
    }
    /**
     * Flags conversation items. Calling this method results in a call to EWS.
     *
     * @param   {FolderId}   contextFolderId   The Id of the folder items must belong to in order to be flagged. If contextFolderId is null, items in conversation across the entire mailbox are flagged.
     * @param   {DateTime}   startDate         The start date (can be null).
     * @param   {DateTime}   dueDate           The due date (can be null).
     * @return  {Promise<void>}    Promise
     */
    FlagItems(contextFolderId, startDate, dueDate) {
        let flag = new Flag_1.Flag();
        flag.FlagStatus = ItemFlagStatus_1.ItemFlagStatus.Flagged;
        if (startDate) {
            flag.StartDate = startDate;
        }
        if (dueDate) {
            flag.DueDate = dueDate;
        }
        return this.Service.SetFlagStatusForItemsInConversations([{ key: this.Id, value: this.GlobalLastDeliveryTime }], contextFolderId, flag).then((responses) => {
            responses.__thisIndexer(0).ThrowIfNecessary();
        });
    }
    /**
     * Flag conversation items as complete. Calling this method results in a call to EWS.
     *
     * @param   {FolderId}   contextFolderId   The Id of the folder items must belong to in order to be flagged as complete. If contextFolderId is null, items in conversation across the entire mailbox are marked as complete.
     * @param   {DateTime}   completeDate      The complete date (can be null).
     * @return  {Promise<void>}    Promise
     */
    FlagItemsComplete(contextFolderId, completeDate) {
        let flag = new Flag_1.Flag();
        flag.FlagStatus = ItemFlagStatus_1.ItemFlagStatus.Complete;
        if (completeDate) {
            flag.CompleteDate = completeDate;
        }
        return this.Service.SetFlagStatusForItemsInConversations([{ key: this.Id, value: this.GlobalLastDeliveryTime }], contextFolderId, flag).then((responses) => {
            responses.__thisIndexer(0).ThrowIfNecessary();
        });
    }
    /**
     * @internal This method is not supported in this object.
     * Gets the name of the change XML element.
     *
     * @return  {string}      XML element name,
     */
    GetChangeXmlElementName() {
        throw new NotSupportedException_1.NotSupportedException();
    }
    /**
     * @internal This method is not supported in this object.
     * Gets the name of the delete field XML element.
     *
     * @return  {string}      XML element name,
     */
    GetDeleteFieldXmlElementName() {
        throw new NotSupportedException_1.NotSupportedException();
    }
    /**
     * The property definition for the Id of this object.
     *
     * @return  {PropertyDefinition}      A PropertyDefinition instance.
     */
    GetIdPropertyDefinition() {
        return Schemas_1.Schemas.ConversationSchema.Id;
    }
    /**
     * @internal This method is not supported in this object.
     * Gets a value indicating whether a time zone SOAP header should be emitted in a CreateItem or UpdateItem request so this item can be property saved or updated.
     *
     * @param   {boolean}   isUpdateOperation   Indicates whether the operation being petrformed is an update operation.
     * @return  {boolean}                       true if a time zone SOAP header should be emitted; otherwise, false.
     */
    GetIsTimeZoneHeaderRequired(isUpdateOperation) {
        throw new NotSupportedException_1.NotSupportedException();
    }
    /**
     * @internal Gets the minimum required server version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this service object type is supported.
     */
    GetMinimumRequiredServerVersion() {
        return ExchangeVersion_1.ExchangeVersion.Exchange2010_SP1;
    }
    /**
     * @internal Internal method to return the schema associated with this type of object.
     *
     * @return  {ServiceObjectSchema}      The schema associated with this type of object.
     */
    GetSchema() {
        return Schemas_1.Schemas.ConversationSchema.Instance;
    }
    /**
     * @internal This method is not supported in this object.
     * Gets the name of the set field XML element.
     *
     * @return  {string}      XML element name,
     */
    GetSetFieldXmlElementName() {
        throw new NotSupportedException_1.NotSupportedException();
    }
    /**
     * @internal Gets the element name of item in XML
     *
     * @return  {string} name of elelment
     */
    GetXmlElementName() {
        return XmlElementNames_1.XmlElementNames.Conversation;
    }
    /**
     * @internal This is not supported in this object.
     * Deletes the object.
     *
     * @param   {DeleteMode}                deleteMode                The deletion mode.
     * @param   {SendCancellationsMode}     sendCancellationsMode     Indicates whether meeting cancellation messages should be sent.
     * @param   {AffectedTaskOccurrence}    affectedTaskOccurrences   Indicate which occurrence of a recurring task should be deleted.
     */
    InternalDelete(deleteMode, sendCancellationsMode, affectedTaskOccurrences) {
        throw new NotSupportedException_1.NotSupportedException();
    }
    /**
     * @internal This method is not supported in this object.
     * Loads the specified set of properties on the object.
     *
     * @param   {PropertySet}   propertySet   The properties to load.
     */
    InternalLoad(propertySet) {
        throw new NotSupportedException_1.NotSupportedException();
    }
    /**
     * Moves items in the specified conversation to a specific folder.
     * Calling this method results in a call to EWS.
     *
     * @param   {FolderId}   contextFolderId       The Id of the folder items must belong to in order to be moved. If contextFolderId is null, items across the entire mailbox are moved.
     * @param   {FolderId}   destinationFolderId   The Id of the destination folder.
     * @return  {Promise<void>}    Promise
     */
    MoveItemsInConversation(contextFolderId, destinationFolderId) {
        return this.Service.MoveItemsInConversations([{ key: this.Id, value: this.GlobalLastDeliveryTime }], contextFolderId, destinationFolderId).then((responses) => {
            responses.__thisIndexer(0).ThrowIfNecessary();
        });
    }
    SetReadStateForItemsInConversation(contextFolderId, isRead, suppressReadReceipts = null) {
        return this.Service.SetReadStateForItemsInConversations([{ key: this.Id, value: this.GlobalLastDeliveryTime }], contextFolderId, isRead, suppressReadReceipts).then((responses) => {
            responses.__thisIndexer(0).ThrowIfNecessary();
        });
    }
    /**
     * Sets the retention policy of items in the specified conversation. Calling this method results in a call to EWS.
     *
     * @param   {FolderId}          contextFolderId        The Id of the folder items must belong to in order for their retention policy to be set. If contextFolderId is null, the retention policy of items across the entire mailbox are set.
     * @param   {RetentionType}     retentionPolicyType    Retention policy type.
     * @param   {Guid}              retentionPolicyTagId   Retention policy tag id.  Null will clear the policy.
     * @return  {Promise<void>}    Promise
     */
    SetRetentionPolicyForItemsInConversation(contextFolderId, retentionPolicyType, retentionPolicyTagId) {
        return this.Service.SetRetentionPolicyForItemsInConversations([{ key: this.Id, value: this.GlobalLastDeliveryTime }], contextFolderId, retentionPolicyType, retentionPolicyTagId).then((responses) => {
            responses.__thisIndexer(0).ThrowIfNecessary();
        });
    }
}
exports.Conversation = Conversation;
//# sourceMappingURL=Conversation.js.map