"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Appointment = void 0;
const AcceptMeetingInvitationMessage_1 = require("../ResponseObjects/AcceptMeetingInvitationMessage");
const AppointmentOccurrenceId_1 = require("../../../ComplexProperties/AppointmentOccurrenceId");
const ExtensionMethods_1 = require("../../../ExtensionMethods");
const CancelMeetingMessage_1 = require("../ResponseObjects/CancelMeetingMessage");
const DeclineMeetingInvitationMessage_1 = require("../ResponseObjects/DeclineMeetingInvitationMessage");
const ExchangeVersion_1 = require("../../../Enumerations/ExchangeVersion");
const FolderId_1 = require("../../../ComplexProperties/FolderId");
const ItemAttachment_1 = require("../../../ComplexProperties/ItemAttachment");
const MessageBody_1 = require("../../../ComplexProperties/MessageBody");
const PropertySet_1 = require("../../PropertySet");
const RecurringAppointmentMasterId_1 = require("../../../ComplexProperties/RecurringAppointmentMasterId");
const ResponseMessage_1 = require("../ResponseObjects/ResponseMessage");
const ResponseMessageType_1 = require("../../../Enumerations/ResponseMessageType");
const Schemas_1 = require("../Schemas/Schemas");
const SendCancellationsMode_1 = require("../../../Enumerations/SendCancellationsMode");
const SendInvitationsMode_1 = require("../../../Enumerations/SendInvitationsMode");
const SendInvitationsOrCancellationsMode_1 = require("../../../Enumerations/SendInvitationsOrCancellationsMode");
const ServiceLocalException_1 = require("../../../Exceptions/ServiceLocalException");
const Strings_1 = require("../../../Strings");
const XmlElementNames_1 = require("../../XmlElementNames");
const Item_1 = require("./Item");
/**
 * Represents an **appointment or a meeting**. Properties available on appointments are defined in the *AppointmentSchema* class.
 */
class Appointment extends Item_1.Item {
    /** required to check [Attachable] attribute, AttachmentCollection.AddItemAttachment<TItem>() checks for non inherited [Attachable] attribute.*/
    static get Attachable() { return this.name === "Appointment"; }
    ;
    /**
     * @internal Gets the default setting for sending cancellations on Delete.
     *
     * @return  {SendCancellationsMode}      If Delete() is called on Appointment, we want to send cancellations and save a copy.
     */
    get DefaultSendCancellationsMode() {
        return SendCancellationsMode_1.SendCancellationsMode.SendToAllAndSaveCopy;
    }
    /**
     * @internal Gets the default settings for sending invitations on Save.
     */
    get DefaultSendInvitationsMode() {
        return SendInvitationsMode_1.SendInvitationsMode.SendToAllAndSaveCopy;
    }
    /**
     * @internal Gets the default settings for sending invitations or cancellations on Update.
     */
    get DefaultSendInvitationsOrCancellationsMode() {
        return SendInvitationsOrCancellationsMode_1.SendInvitationsOrCancellationsMode.SendToAllAndSaveCopy;
    }
    /**
     * Gets or sets the start time of the appointment.
     */
    get Start() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.Start);
    }
    set Start(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.AppointmentSchema.Start, value);
    }
    /**
     * Gets or sets the end time of the appointment.
     */
    get End() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.End);
    }
    set End(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.AppointmentSchema.End, value);
    }
    /**
     * Gets the original start time of this appointment.
     */
    get OriginalStart() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.OriginalStart);
    }
    /**
     * Gets or sets a value indicating whether this appointment is an all day event.
     */
    get IsAllDayEvent() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.IsAllDayEvent);
    }
    set IsAllDayEvent(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.AppointmentSchema.IsAllDayEvent, value);
    }
    /**
     * Gets or sets a value indicating the free/busy status of the owner of this appointment.
     */
    get LegacyFreeBusyStatus() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.LegacyFreeBusyStatus);
    }
    set LegacyFreeBusyStatus(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.AppointmentSchema.LegacyFreeBusyStatus, value);
    }
    /**
     * Gets or sets the location of this appointment.
     */
    get Location() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.Location);
    }
    set Location(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.AppointmentSchema.Location, value);
    }
    /**
     * Gets a text indicating when this appointment occurs.
     * The text returned by When is localized using the Exchange Server culture or using the culture specified in the PreferredCulture property of the ExchangeService object this appointment is bound to.
     */
    get When() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.When);
    }
    /**
     * Gets a value indicating whether the appointment is a meeting.
     */
    get IsMeeting() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.IsMeeting);
    }
    /**
     * Gets a value indicating whether the appointment has been cancelled.
     */
    get IsCancelled() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.IsCancelled);
    }
    /**
     * Gets a value indicating whether the appointment is recurring.
     */
    get IsRecurring() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.IsRecurring);
    }
    /**
     * Gets a value indicating whether the meeting request has already been sent.
     */
    get MeetingRequestWasSent() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.MeetingRequestWasSent);
    }
    set IsResponseRequested(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.AppointmentSchema.IsResponseRequested, value);
    }
    /**
     * Gets or sets a value indicating whether responses are requested when invitations are sent for this meeting.
     */
    get IsResponseRequested() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.IsResponseRequested);
    }
    /**
     * Gets a value indicating the type of this appointment.
     */
    get AppointmentType() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.AppointmentType);
    }
    /**
     * Gets a value indicating what was the last response of the user that loaded this meeting.
     */
    get MyResponseType() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.MyResponseType);
    }
    /**
     * Gets the organizer of this meeting. The Organizer property is read-only and is only relevant for attendees.
     * The organizer of a meeting is automatically set to the user that created the meeting.
     */
    get Organizer() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.Organizer);
    }
    /**
     * Gets a list of required attendees for this meeting.
     */
    get RequiredAttendees() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.RequiredAttendees);
    }
    /**
     * Gets a list of optional attendeed for this meeting.
     */
    get OptionalAttendees() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.OptionalAttendees);
    }
    /**
     * Gets a list of resources for this meeting.
     */
    get Resources() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.Resources);
    }
    /**
     * Gets the number of calendar entries that conflict with this appointment in the authenticated user's calendar.
     */
    get ConflictingMeetingCount() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.ConflictingMeetingCount);
    }
    /**
     * Gets the number of calendar entries that are adjacent to this appointment in the authenticated user's calendar.
     */
    get AdjacentMeetingCount() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.AdjacentMeetingCount);
    }
    /**
     * Gets a list of meetings that conflict with this appointment in the authenticated user's calendar.
     */
    get ConflictingMeetings() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.ConflictingMeetings);
    }
    /**
     * Gets a list of meetings that is adjacent to this appointment in the authenticated user's calendar.
     */
    get AdjacentMeetings() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.AdjacentMeetings);
    }
    /**
     * Gets the duration of this appointment.
     */
    get Duration() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.Duration);
    }
    /**
     * Gets the name of the time zone this appointment is defined in.
     */
    get TimeZone() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.TimeZone);
    }
    /**
     * Gets the time when the attendee replied to the meeting request.
     */
    get AppointmentReplyTime() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.AppointmentReplyTime);
    }
    /**
     * Gets the sequence number of this appointment.
     */
    get AppointmentSequenceNumber() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.AppointmentSequenceNumber);
    }
    /**
     * Gets the state of this appointment.
     */
    get AppointmentState() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.AppointmentState);
    }
    /**
     * Gets or sets the recurrence pattern for this appointment. Available recurrence pattern classes include
     * Recurrence.DailyPattern, Recurrence.MonthlyPattern and Recurrence.YearlyPattern.
     */
    get Recurrence() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.Recurrence);
    }
    set Recurrence(value) {
        if (value !== null && value.IsRegenerationPattern) {
            throw new ServiceLocalException_1.ServiceLocalException(Strings_1.Strings.RegenerationPatternsOnlyValidForTasks);
        }
        this.PropertyBag._setItem(Schemas_1.Schemas.AppointmentSchema.Recurrence, value);
    }
    /**
     * Gets an OccurrenceInfo identifying the first occurrence of this meeting.
     */
    get FirstOccurrence() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.FirstOccurrence);
    }
    /**
     * Gets an OccurrenceInfo identifying the last occurrence of this meeting.
     */
    get LastOccurrence() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.LastOccurrence);
    }
    /**
     * Gets a list of modified occurrences for this meeting.
     */
    get ModifiedOccurrences() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.ModifiedOccurrences);
    }
    /**
     * Gets a list of deleted occurrences for this meeting.
     */
    get DeletedOccurrences() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.DeletedOccurrences);
    }
    /**
     * Gets or sets time zone of the start property of this appointment.
     */
    get StartTimeZone() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.StartTimeZone);
    }
    set StartTimeZone(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.AppointmentSchema.StartTimeZone, value);
    }
    /**
     * Gets or sets time zone of the end property of this appointment.
     */
    get EndTimeZone() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.EndTimeZone);
    }
    set EndTimeZone(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.AppointmentSchema.EndTimeZone, value);
    }
    /**
     * Gets or sets the type of conferencing that will be used during the meeting.
     */
    get ConferenceType() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.ConferenceType);
    }
    set ConferenceType(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.AppointmentSchema.ConferenceType, value);
    }
    /**
     * Gets or sets a value indicating whether new time proposals are allowed for attendees of this meeting.
     */
    get AllowNewTimeProposal() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.AllowNewTimeProposal);
    }
    set AllowNewTimeProposal(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.AppointmentSchema.AllowNewTimeProposal, value);
    }
    /**
     * Gets or sets a value indicating whether this is an online meeting.
     */
    get IsOnlineMeeting() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.IsOnlineMeeting);
    }
    set IsOnlineMeeting(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.AppointmentSchema.IsOnlineMeeting, value);
    }
    /**
     * Gets or sets the URL of the meeting workspace. A meeting workspace is a shared Web site for planning meetings and tracking results.
     */
    get MeetingWorkspaceUrl() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.MeetingWorkspaceUrl);
    }
    set MeetingWorkspaceUrl(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.AppointmentSchema.MeetingWorkspaceUrl, value);
    }
    /**
     * Gets or sets the URL of the Microsoft NetShow online meeting.
     */
    get NetShowUrl() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.NetShowUrl);
    }
    set NetShowUrl(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.AppointmentSchema.NetShowUrl, value);
    }
    /**
     * Gets or sets the ICalendar Uid.
     */
    get ICalUid() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.ICalUid);
    }
    set ICalUid(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.AppointmentSchema.ICalUid, value);
    }
    /**
     * Gets the ICalendar RecurrenceId.
     */
    get ICalRecurrenceId() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.ICalRecurrenceId);
    }
    /**
     * Gets the ICalendar DateTimeStamp.
     */
    get ICalDateTimeStamp() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.ICalDateTimeStamp);
    }
    /**
     * Gets or sets the Enhanced location object.
     */
    get EnhancedLocation() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.EnhancedLocation);
    }
    set EnhancedLocation(value) {
        this.PropertyBag._setItem(Schemas_1.Schemas.AppointmentSchema.EnhancedLocation, value);
    }
    /**
     * Gets the Url for joining an online meeting
     */
    get JoinOnlineMeetingUrl() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.JoinOnlineMeetingUrl);
    }
    /**
     * Gets the Online Meeting Settings
     */
    get OnlineMeetingSettings() {
        return this.PropertyBag._getItem(Schemas_1.Schemas.AppointmentSchema.OnlineMeetingSettings);
    }
    constructor(svcOrAttachment, isNew = false) {
        super(svcOrAttachment);
        if (svcOrAttachment instanceof ItemAttachment_1.ItemAttachment) { //todo:fix -can not user instanceof with exchangeservice, creates circular loop with ewsutility
            let parentAttachment = svcOrAttachment;
            // If we're running against Exchange 2007, we need to explicitly preset
            // the StartTimeZone property since Exchange 2007 will otherwise scope
            // start and end to UTC.
            if (parentAttachment.Service.RequestedServerVersion == ExchangeVersion_1.ExchangeVersion.Exchange2007_SP1) {
                if (isNew) {
                    this.StartTimeZone = parentAttachment.Service.TimeZone;
                }
            }
        }
    }
    /**
     * Accepts the meeting. Calling this method results in a call to EWS.
     *
     * @param   {boolean}   sendResponse   Indicates whether to send a response to the organizer.
     * @return  {Promise<CalendarActionResults>}   A CalendarActionResults object containing the various items that were created or modified as a results of this operation :Promise.
     */
    Accept(sendResponse) {
        return this.InternalAccept(false, sendResponse);
    }
    /**
     * Tentatively accepts the meeting. Calling this method results in a call to EWS.
     *
     * @param   {boolean}   sendResponse   Indicates whether to send a response to the organizer.
     * @return  {Promise<CalendarActionResults>}   A CalendarActionResults object containing the various items that were created or modified as a results of this operation :Promise.
     */
    AcceptTentatively(sendResponse) {
        return this.InternalAccept(true, sendResponse);
    }
    static Bind(service, id, propertySet = PropertySet_1.PropertySet.FirstClassProperties) {
        return service.BindToItem(id, propertySet, Appointment);
    }
    static BindToOccurrence(service, recurringMasterId, occurenceIndex, propertySet = PropertySet_1.PropertySet.FirstClassProperties) {
        let occurenceId = new AppointmentOccurrenceId_1.AppointmentOccurrenceId(recurringMasterId.UniqueId, occurenceIndex);
        return Appointment.Bind(service, occurenceId, propertySet);
    }
    static BindToRecurringMaster(service, occurrenceId, propertySet = PropertySet_1.PropertySet.FirstClassProperties) {
        let recurringMasterId = new RecurringAppointmentMasterId_1.RecurringAppointmentMasterId(occurrenceId.UniqueId);
        return Appointment.Bind(service, recurringMasterId, propertySet);
    }
    CancelMeeting(cancellationMessageText) {
        if (arguments.length === 0) {
            return this.CreateCancelMeetingMessage().SendAndSaveCopy();
        }
        let cancelMsg = this.CreateCancelMeetingMessage();
        cancelMsg.Body = new MessageBody_1.MessageBody(cancellationMessageText); //todo:fix - cant use implicit operator of c#, using explicit. Assumed HTML body used in c# implicit conversion
        return cancelMsg.SendAndSaveCopy();
    }
    /**
     * Creates a local meeting acceptance message that can be customized and sent.
     *
     * @param   {boolean}   tentative   Specifies whether the meeting will be tentatively accepted.
     * @return  {AcceptMeetingInvitationMessage}  An AcceptMeetingInvitationMessage representing the meeting acceptance message.
     */
    CreateAcceptMessage(tentative) {
        return new AcceptMeetingInvitationMessage_1.AcceptMeetingInvitationMessage(this, tentative);
    }
    /**
     * Creates a local meeting cancellation message that can be customized and sent.
     *
     * @return  {CancelMeetingMessage}    A CancelMeetingMessage representing the meeting cancellation message.
     */
    CreateCancelMeetingMessage() {
        return new CancelMeetingMessage_1.CancelMeetingMessage(this);
    }
    /**
     * Creates a local meeting declination message that can be customized and sent.
     *
     * @return  {DeclineMeetingInvitationMessage}      A DeclineMeetingInvitation representing the meeting declination message.
     */
    CreateDeclineMessage() {
        return new DeclineMeetingInvitationMessage_1.DeclineMeetingInvitationMessage(this);
    }
    /**
     * Creates a forward message from this appointment.
     *
     * @return  {ResponseMessage} A ResponseMessage representing the forward response that can subsequently be modified and sent.
     */
    CreateForward() {
        this.ThrowIfThisIsNew();
        return new ResponseMessage_1.ResponseMessage(this, ResponseMessageType_1.ResponseMessageType.Forward);
    }
    /**
     * Creates a reply response to the organizer and/or attendees of the meeting.
     *
     * @param   {boolean}   replyAll   Indicates whether the reply should go to the organizer only or to all the attendees.
     * @return  {ResponseMessage} A ResponseMessage representing the reply response that can subsequently be modified and sent.
     */
    CreateReply(replyAll) {
        this.ThrowIfThisIsNew();
        return new ResponseMessage_1.ResponseMessage(this, replyAll ? ResponseMessageType_1.ResponseMessageType.ReplyAll : ResponseMessageType_1.ResponseMessageType.Reply);
    }
    /**
     * Declines the meeting invitation. Calling this method results in a call to EWS.
     *
     * @param   {boolean}   sendResponse   Indicates whether to send a response to the organizer.
     * @return  {Promise<CalendarActionResults>}   A CalendarActionResults object containing the various items that were created or modified as aresults of this operation :Promise.
     */
    Decline(sendResponse) {
        let decline = this.CreateDeclineMessage();
        if (sendResponse) {
            return decline.SendAndSaveCopy();
        }
        else {
            return decline.Save();
        }
    }
    Delete(deleteMode, sendCancellationsMode) {
        return this.InternalDelete(deleteMode, sendCancellationsMode, null);
    }
    Forward(bodyPrefix, _toRecipients) {
        let responseMessage = this.CreateForward();
        let toRecipients = [];
        responseMessage.BodyPrefix = bodyPrefix;
        if (ExtensionMethods_1.ArrayHelper.isArray(_toRecipients)) {
            toRecipients = _toRecipients;
        }
        else {
            for (var _i = 1; _i < arguments.length; _i++) {
                toRecipients[_i - 1] = arguments[_i];
            }
        }
        responseMessage.ToRecipients.AddRange(toRecipients);
        return responseMessage.SendAndSaveCopy();
    }
    /**
     * @internal Determines whether properties defined with ScopedDateTimePropertyDefinition require custom time zone scoping.
     *
     * @return  {boolean}      true if this item type requires custom scoping for scoped date/time properties; otherwise, false.
     */
    GetIsCustomDateTimeScopingRequired() {
        return true;
    }
    /**
     * @internal Gets a value indicating whether a time zone SOAP header should be emitted in a CreateItem or UpdateItem request so this item can be property saved or updated.
     *
     * @param   {boolean}   isUpdateOperation   Indicates whether the operation being petrformed is an update operation.
     * @return  {boolean}                       true if a time zone SOAP header should be emitted; otherwise, false.
     */
    GetIsTimeZoneHeaderRequired(isUpdateOperation) {
        if (isUpdateOperation) {
            return false;
        }
        else {
            let isStartTimeZoneSetOrUpdated = this.PropertyBag.IsPropertyUpdated(Schemas_1.Schemas.AppointmentSchema.StartTimeZone);
            let isEndTimeZoneSetOrUpdated = this.PropertyBag.IsPropertyUpdated(Schemas_1.Schemas.AppointmentSchema.EndTimeZone);
            if (isStartTimeZoneSetOrUpdated && isEndTimeZoneSetOrUpdated) {
                // If both StartTimeZone and EndTimeZone have been set or updated and are the same as the service's
                // time zone, we emit the time zone header and StartTimeZone and EndTimeZone are not emitted.
                let startTimeZone = { outValue: null };
                let endTimeZone = { outValue: null };
                ;
                this.PropertyBag.TryGetPropertyAs(Schemas_1.Schemas.AppointmentSchema.StartTimeZone, startTimeZone);
                this.PropertyBag.TryGetPropertyAs(Schemas_1.Schemas.AppointmentSchema.EndTimeZone, endTimeZone);
                return startTimeZone.outValue == this.Service.TimeZone || endTimeZone.outValue == this.Service.TimeZone;
            }
            else {
                return true;
            }
        }
    }
    /**
     * @internal Gets the minimum required server version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this service object type is supported.
     */
    GetMinimumRequiredServerVersion() {
        return ExchangeVersion_1.ExchangeVersion.Exchange2007_SP1;
    }
    /**
     * @internal Internal method to return the schema associated with this type of object.
     *
     * @return  {ServiceObjectSchema}      The schema associated with this type of object.
     */
    GetSchema() {
        return Schemas_1.Schemas.AppointmentSchema.Instance;
    }
    /**
     * @internal Gets the element name of item in XML
     *
     * @return  {string} name of elelment
     */
    GetXmlElementName() {
        return XmlElementNames_1.XmlElementNames.CalendarItem;
    }
    /**
     * @internal Accepts the meeting.
     *
     * @param   {boolean}   tentative      True if tentative accept.
     * @param   {boolean}   sendResponse   Indicates whether to send a response to the organizer.
     * @return  {Promise<CalendarActionResults>}    A CalendarActionResults object containing the various items that were created or modified as aresults of this operation :Promise.
     */
    InternalAccept(tentative, sendResponse) {
        let accept = this.CreateAcceptMessage(tentative);
        if (sendResponse) {
            return accept.SendAndSaveCopy();
        }
        else {
            return accept.Save();
        }
    }
    /**
     * Replies to the organizer and/or the attendees of the meeting. Calling this method results in a call to EWS.
     *
     * @param   {MessageBody}     bodyPrefix   The prefix to prepend to the body of the meeting.
     * @param   {boolean}         replyAll     Indicates whether the reply should go to the organizer only or to all the attendees.
     */
    Reply(bodyPrefix, replyAll) {
        let responseMessage = this.CreateReply(replyAll);
        responseMessage.BodyPrefix = bodyPrefix;
        return responseMessage.SendAndSaveCopy();
    }
    Save(destinationFolderNameOrIdOrSendInvitationMode, sendInvitationsMode) {
        let argsLength = arguments.length;
        if (argsLength < 1 || argsLength > 2) {
            throw new Error("Appointment.ts - Save : Invalid number of arguments");
        }
        let simode = sendInvitationsMode;
        if (argsLength === 1) {
            simode = destinationFolderNameOrIdOrSendInvitationMode;
            return this.InternalCreate(null, null, simode);
        }
        let destinationFolderId = destinationFolderNameOrIdOrSendInvitationMode;
        if (argsLength === 2) {
            if (typeof destinationFolderNameOrIdOrSendInvitationMode === "number") {
                destinationFolderId = new FolderId_1.FolderId(destinationFolderNameOrIdOrSendInvitationMode);
            }
            return this.InternalCreate(destinationFolderId, null, sendInvitationsMode);
        }
    }
    Update(conflictResolutionMode, sendInvitationsOrCancellationsMode) {
        return this.InternalUpdate(null, conflictResolutionMode, null, sendInvitationsOrCancellationsMode);
    }
    /**
     * @internal Validates this instance.
     *
     */
    Validate() {
        super.Validate();
        //  Make sure that if we're on the Exchange2007_SP1 schema version, if any of the following
        //  properties are set or updated:
        //      o   Start
        //      o   End
        //      o   IsAllDayEvent
        //      o   Recurrence
        //  ... then, we must send the MeetingTimeZone element (which is generated from StartTimeZone for
        //  Exchange2007_SP1 requests (see StartTimeZonePropertyDefinition.cs).  If the StartTimeZone isn't
        //  in the property bag, then throw, because clients must supply the proper time zone - either by
        //  loading it from a currently-existing appointment, or by setting it directly.  Otherwise, to dirty
        //  the StartTimeZone property, we just set it to its current value.
        if ((this.Service.RequestedServerVersion == ExchangeVersion_1.ExchangeVersion.Exchange2007_SP1) &&
            !this.Service.Exchange2007CompatibilityMode) {
            if (this.PropertyBag.IsPropertyUpdated(Schemas_1.Schemas.AppointmentSchema.Start) ||
                this.PropertyBag.IsPropertyUpdated(Schemas_1.Schemas.AppointmentSchema.End) ||
                this.PropertyBag.IsPropertyUpdated(Schemas_1.Schemas.AppointmentSchema.IsAllDayEvent) ||
                this.PropertyBag.IsPropertyUpdated(Schemas_1.Schemas.AppointmentSchema.Recurrence)) {
                //  If the property isn't in the property bag, throw....
                if (!this.PropertyBag.Contains(Schemas_1.Schemas.AppointmentSchema.StartTimeZone)) {
                    throw new ServiceLocalException_1.ServiceLocalException(Strings_1.Strings.StartTimeZoneRequired);
                }
                //  Otherwise, set the time zone to its current value to force it to be sent with the request.
                this.StartTimeZone = this.StartTimeZone;
            }
        }
    }
}
exports.Appointment = Appointment;
//# sourceMappingURL=Appointment.js.map