"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceResponseCollection = void 0;
const ArgumentException_1 = require("../../Exceptions/ArgumentException");
const EwsLogging_1 = require("../EwsLogging");
const ServiceResult_1 = require("../../Enumerations/ServiceResult");
const Strings_1 = require("../../Strings");
/**
 * Represents a strogly typed list of service responses.
 * @sealed
 * @typeparam   {TResponse}     The type of response stored in the list.
 */
class ServiceResponseCollection {
    /**
     * @internal Initializes a new instance of the **ServiceResponseCollection<TResponse>** class.
     */
    constructor() {
        this.responses = [];
        this.overallResult = ServiceResult_1.ServiceResult.Success;
    }
    /**
     * Gets the total number of responses in the list.
     */
    get Count() {
        return this.responses.length;
    }
    get Responses() { return this.responses; }
    /**
     * Gets a value indicating the overall result of the request that generated this response collection.
     * If all of the responses have their Result property set to Success, OverallResult returns Success.
     * If at least one response has its Result property set to Warning and all other responses have their Result property set to Success, OverallResult returns Warning.
     * If at least one response has a its Result set to Error, OverallResult returns Error.
     */
    get OverallResult() { return this.overallResult; }
    /**
     * @internal Adds specified response.
     *
     * @param   {TResponse}   response   The response.
     */
    Add(response) {
        EwsLogging_1.EwsLogging.Assert(response != null, "EwsResponseList.Add", "response is null");
        if (response.Result > this.overallResult) {
            this.overallResult = response.Result;
        }
        this.responses.push(response);
    }
    /**
     * Gets an enumerator that iterates through the elements of the collection.
     *
     * @return  {TResponse[]}      An IEnumerator for the collection.
     */
    GetEnumerator() {
        return this.responses;
    }
    /**
     * Gets the response at the specified index.
     *
     * @param   {number}        index   The zero-based index of the response to get.
     * @return  {TResponse}     The response at the specified index.
     */
    __thisIndexer(index) {
        if (index < 0 || index >= this.Count) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("index", Strings_1.Strings.IndexIsOutOfRange);
        }
        return this.responses[index];
    }
}
exports.ServiceResponseCollection = ServiceResponseCollection;
//# sourceMappingURL=ServiceResponseCollection.js.map