"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SyncFolderItemsRequest = void 0;
const ArgumentException_1 = require("../../Exceptions/ArgumentException");
const EwsUtilities_1 = require("../EwsUtilities");
const ExchangeVersion_1 = require("../../Enumerations/ExchangeVersion");
const ItemIdWrapperList_1 = require("../../Misc/ItemIdWrapperList");
const ServiceErrorHandling_1 = require("../../Enumerations/ServiceErrorHandling");
const ServiceObjectType_1 = require("../../Enumerations/ServiceObjectType");
const ServiceVersionException_1 = require("../../Exceptions/ServiceVersionException");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const Strings_1 = require("../../Strings");
const SyncFolderItemsScope_1 = require("../../Enumerations/SyncFolderItemsScope");
const XmlElementNames_1 = require("../XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const MultiResponseServiceRequest_1 = require("./MultiResponseServiceRequest");
const SyncFolderItemsResponse_1 = require("../Responses/SyncFolderItemsResponse");
/**
 * @internal Represents a SyncFolderItems request.
 */
class SyncFolderItemsRequest extends MultiResponseServiceRequest_1.MultiResponseServiceRequest {
    /**
     * @internal Initializes a new instance of the **SyncFolderItemsRequest** class.
     *
     * @param   {ExchangeService}   	service             The service.
     */
    constructor(service) {
        super(service, ServiceErrorHandling_1.ServiceErrorHandling.ThrowOnError);
        this.propertySet = null;
        this.syncFolderId = null;
        this.syncScope = SyncFolderItemsScope_1.SyncFolderItemsScope.NormalItems;
        this.syncState = null;
        this.ignoredItemIds = new ItemIdWrapperList_1.ItemIdWrapperList();
        this.maxChangesReturned = 100;
        this.numberOfDays = 0;
    }
    /**
     * Gets or sets the property set.
     *
     * @value	The property set
     */
    get PropertySet() {
        return this.propertySet;
    }
    set PropertySet(value) {
        this.propertySet = value;
    }
    /**
     * Gets or sets the sync folder id.
     *
     * @value	The sync folder id.
     */
    get SyncFolderId() {
        return this.syncFolderId;
    }
    set SyncFolderId(value) {
        this.syncFolderId = value;
    }
    /**
     * Gets or sets the scope of the sync.
     *
     * @value	The scope of the sync.
     */
    get SyncScope() {
        return this.syncScope;
    }
    set SyncScope(value) {
        this.syncScope = value;
    }
    /**
     * Gets or sets the state of the sync.
     *
     * @value	The state of the sync.
     */
    get SyncState() {
        return this.syncState;
    }
    set SyncState(value) {
        this.syncState = value;
    }
    /**
     * Gets the list of ignored item ids.
     *
     * @value	The ignored item ids.
     */
    get IgnoredItemIds() {
        return this.ignoredItemIds;
    }
    /**
     * Gets or sets the maximum number of changes returned by SyncFolderItems.
     * Values must be between 1 and 512.
     * Default is 100.
     */
    get MaxChangesReturned() {
        return this.maxChangesReturned;
    }
    set MaxChangesReturned(value) {
        if (value >= 1 && value <= 512) {
            this.maxChangesReturned = value;
        }
        else {
            throw new ArgumentException_1.ArgumentException(Strings_1.Strings.MaxChangesMustBeBetween1And512);
        }
    }
    /**
     * Gets or sets the number of days of content returned by SyncFolderItems.
     * Zero means return all content.
     * Default is zero.
     */
    get NumberOfDays() {
        return this.numberOfDays;
    }
    set NumberOfDays(value) {
        if (value >= 0) {
            this.numberOfDays = value;
        }
        else {
            throw new ArgumentException_1.ArgumentException(Strings_1.Strings.NumberOfDaysMustBePositive);
        }
    }
    /**
     * @internal Creates the service response.
     *
     * @param   {ExchangeService}   service         The service.
     * @param   {number}   			responseIndex   Index of the response.
     * @return  {SyncFolderItemsResponse}		Response object.
     */
    CreateServiceResponse(service, responseIndex) {
        return new SyncFolderItemsResponse_1.SyncFolderItemsResponse(this.PropertySet);
    }
    /**
     * @internal Gets the expected response message count.
     *
     * @return  {number}      Number of items in response.
     */
    GetExpectedResponseMessageCount() {
        return 1;
    }
    /**
     * @internal Gets the request version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this request is supported.
     */
    GetMinimumRequiredServerVersion() {
        return ExchangeVersion_1.ExchangeVersion.Exchange2007_SP1;
    }
    /**
     * @internal Gets the name of the response message XML element.
     *
     * @return  {string}      XML element name.
     */
    GetResponseMessageXmlElementName() {
        return XmlElementNames_1.XmlElementNames.SyncFolderItemsResponseMessage;
    }
    /**
     * @internal Gets the name of the response XML element.
     *
     * @return  {string}      XML element name.
     */
    GetResponseXmlElementName() {
        return XmlElementNames_1.XmlElementNames.SyncFolderItemsResponse;
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @return  {string}      XML element name.
     */
    GetXmlElementName() {
        return XmlElementNames_1.XmlElementNames.SyncFolderItems;
    }
    /**
     * @internal Validate request.
     */
    Validate() {
        super.Validate();
        EwsUtilities_1.EwsUtilities.ValidateParam(this.PropertySet, "PropertySet");
        EwsUtilities_1.EwsUtilities.ValidateParam(this.SyncFolderId, "SyncFolderId");
        this.SyncFolderId.Validate(this.Service.RequestedServerVersion);
        // SyncFolderItemsScope enum was introduced with Exchange2010.  Only
        // value NormalItems is valid with previous server versions.
        if (this.Service.RequestedServerVersion < ExchangeVersion_1.ExchangeVersion.Exchange2010 &&
            this.syncScope != SyncFolderItemsScope_1.SyncFolderItemsScope.NormalItems) {
            throw new ServiceVersionException_1.ServiceVersionException(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.EnumValueIncompatibleWithRequestVersion, SyncFolderItemsScope_1.SyncFolderItemsScope[this.syncScope], "SyncFolderItemsScope", ExchangeVersion_1.ExchangeVersion[ExchangeVersion_1.ExchangeVersion.Exchange2010]));
        }
        // NumberOfDays was introduced with Exchange 2013.
        if (this.Service.RequestedServerVersion < ExchangeVersion_1.ExchangeVersion.Exchange2013 &&
            this.NumberOfDays != 0) {
            throw new ServiceVersionException_1.ServiceVersionException(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.ParameterIncompatibleWithRequestVersion, "numberOfDays", ExchangeVersion_1.ExchangeVersion[ExchangeVersion_1.ExchangeVersion.Exchange2013]));
        }
        // SyncFolderItems can only handle summary properties
        this.PropertySet.ValidateForRequest(this, true /*summaryPropertiesOnly*/);
    }
    /**
     * @internal Writes the elements to XML writer.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        this.PropertySet.WriteToXml(writer, ServiceObjectType_1.ServiceObjectType.Item);
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.SyncFolderId);
        this.SyncFolderId.WriteToXml(writer);
        writer.WriteEndElement();
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.SyncState, this.SyncState);
        this.IgnoredItemIds.WriteToXml(writer, XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.Ignore);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.MaxChangesReturned, this.MaxChangesReturned);
        if (this.Service.RequestedServerVersion >= ExchangeVersion_1.ExchangeVersion.Exchange2010) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.SyncScope, SyncFolderItemsScope_1.SyncFolderItemsScope[this.syncScope]);
        }
        if (this.NumberOfDays != 0) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.NumberOfDays, this.numberOfDays);
        }
    }
}
exports.SyncFolderItemsRequest = SyncFolderItemsRequest;
//# sourceMappingURL=SyncFolderItemsRequest.js.map