"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubscribeToPullNotificationsRequest = void 0;
const ArgumentException_1 = require("../../Exceptions/ArgumentException");
const ExchangeVersion_1 = require("../../Enumerations/ExchangeVersion");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const Strings_1 = require("../../Strings");
const SubscribeResponse_1 = require("../Responses/SubscribeResponse");
const XmlElementNames_1 = require("../XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const PullSubscription_1 = require("../../Notifications/PullSubscription");
const SubscribeRequest_1 = require("./SubscribeRequest");
/**
 * @internal Represents a "pull" Subscribe request.
 */
class SubscribeToPullNotificationsRequest extends SubscribeRequest_1.SubscribeRequest {
    /**
     * @internal Initializes a new instance of the **SubscribeToPullNotificationsRequest** class.
     *
     * @param   {ExchangeService}   service   The service.
     */
    constructor(service) {
        super(service);
        this.timeout = 30;
    }
    /**
     * Gets or sets the timeout.
     *
     * @value	The timeout.
     */
    get Timeout() {
        return this.timeout;
    }
    set Timeout(value) {
        this.timeout = value;
    }
    /**
     * @internal Creates the service response.
     *
     * @param   {ExchangeService}   service         The service.
     * @param   {number}   			responseIndex   Index of the response.
     * @return  {SubscribeResponse<PullSubscription>}		Service response.
     */
    CreateServiceResponse(service, responseIndex) {
        return new SubscribeResponse_1.SubscribeResponse(new PullSubscription_1.PullSubscription(service));
    }
    /**
     * @internal Gets the request version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this request is supported.
     */
    GetMinimumRequiredServerVersion() {
        return ExchangeVersion_1.ExchangeVersion.Exchange2007_SP1;
    }
    /**
     * @internal Gets the name of the subscription XML element.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this request is supported.
     */
    GetSubscriptionXmlElementName() {
        return XmlElementNames_1.XmlElementNames.PullSubscriptionRequest;
    }
    /**
     * @internal Internal method to write XML elements.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    InternalWriteElementsToXml(writer) {
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Timeout, this.Timeout);
    }
    /**
     * @internal Validate request.
     */
    Validate() {
        super.Validate();
        if ((this.Timeout < 1) || (this.Timeout > 1440)) {
            throw new ArgumentException_1.ArgumentException(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.InvalidTimeoutValue, this.Timeout));
        }
    }
}
exports.SubscribeToPullNotificationsRequest = SubscribeToPullNotificationsRequest;
//# sourceMappingURL=SubscribeToPullNotificationsRequest.js.map