"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FindConversationRequest = void 0;
const ExchangeVersion_1 = require("../../Enumerations/ExchangeVersion");
const FindConversationResponse_1 = require("../Responses/FindConversationResponse");
const MailboxSearchLocation_1 = require("../../Enumerations/MailboxSearchLocation");
const SeekToConditionItemView_1 = require("../../Search/SeekToConditionItemView");
const ServiceObjectType_1 = require("../../Enumerations/ServiceObjectType");
const ServiceVersionException_1 = require("../../Exceptions/ServiceVersionException");
const ExtensionMethods_1 = require("../../ExtensionMethods");
const Strings_1 = require("../../Strings");
const XmlAttributeNames_1 = require("../XmlAttributeNames");
const XmlElementNames_1 = require("../XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const SimpleServiceRequestBase_1 = require("./SimpleServiceRequestBase");
/**
 * @internal Represents a request to a Find Conversation operation
 *
 * @sealed
 */
class FindConversationRequest extends SimpleServiceRequestBase_1.SimpleServiceRequestBase {
    /**
     * @internal Initializes a new instance of the **FindConversationRequest** class.
     *
     * @param   {service}   service   The service.
     */
    constructor(service) {
        super(service);
        this.view = null;
        this.folderId = null;
        this.queryString = null;
        this.returnHighlightTerms = false;
        this.mailboxScope = null;
    }
    /**
     * Gets or sets the view controlling the number of conversations returned.
     */
    get View() {
        return this.view;
    }
    set View(value) {
        this.view = value;
        if (this.view instanceof SeekToConditionItemView_1.SeekToConditionItemView) {
            this.view.SetServiceObjectType(ServiceObjectType_1.ServiceObjectType.Conversation);
        }
    }
    /**
     * @internal Gets or sets folder id
     */
    get FolderId() {
        return this.folderId;
    }
    set FolderId(value) {
        this.folderId = value;
    }
    /**
     * @internal Gets or sets the query string for search value.
     */
    get QueryString() {
        return this.queryString;
    }
    set QueryString(value) {
        this.queryString = value;
    }
    /**
     * @internal Gets or sets the query string highlight terms.
     */
    get ReturnHighlightTerms() {
        return this.returnHighlightTerms;
    }
    set ReturnHighlightTerms(value) {
        this.returnHighlightTerms = value;
    }
    /**
     * @internal Gets or sets the mailbox search location to include in the search.
     */
    get MailboxScope() {
        return this.mailboxScope;
    }
    set MailboxScope(value) {
        this.mailboxScope = value;
    }
    /**
     * @internal Executes this request.
     *
     * @return  {Promise<FindConversationResponse>}      Service response  :Promise.
     */
    Execute() {
        return this.InternalExecute().then((serviceResponse) => {
            serviceResponse.ThrowIfNecessary();
            return serviceResponse;
        });
    }
    /**
     * @internal Gets the request version.
     *
     * @return  {ExchangeVersion}      Earliest Exchange version in which this request is supported.
     */
    GetMinimumRequiredServerVersion() {
        return ExchangeVersion_1.ExchangeVersion.Exchange2010_SP1;
    }
    /**
     * @internal Gets the name of the response XML element.
     *
     * @return  {string}      XML element name.
     */
    GetResponseXmlElementName() {
        return XmlElementNames_1.XmlElementNames.FindConversationResponse;
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @return  {string}      XML element name.
     */
    GetXmlElementName() {
        return XmlElementNames_1.XmlElementNames.FindConversation;
    }
    /**
     * @internal Parses the response.
     *
     * @param   {any}   jsonBody   The js object response body.
     * @return  {any}              Response object.
     */
    ParseResponse(jsonBody) {
        let response = new FindConversationResponse_1.FindConversationResponse();
        response.LoadFromXmlJsObject(jsonBody, this.Service);
        return response;
    }
    /**
     * @internal Validate request.
     */
    Validate() {
        super.Validate();
        this.view.InternalValidate(this);
        // query string parameter is only valid for Exchange2013 or higher
        //
        if (!ExtensionMethods_1.StringHelper.IsNullOrEmpty(this.queryString) &&
            this.Service.RequestedServerVersion < ExchangeVersion_1.ExchangeVersion.Exchange2013) {
            throw new ServiceVersionException_1.ServiceVersionException(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.ParameterIncompatibleWithRequestVersion, "queryString", ExchangeVersion_1.ExchangeVersion[ExchangeVersion_1.ExchangeVersion.Exchange2013]));
        }
        // ReturnHighlightTerms parameter is only valid for Exchange2013 or higher
        //
        if (this.ReturnHighlightTerms &&
            this.Service.RequestedServerVersion < ExchangeVersion_1.ExchangeVersion.Exchange2013) {
            throw new ServiceVersionException_1.ServiceVersionException(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.ParameterIncompatibleWithRequestVersion, "returnHighlightTerms", ExchangeVersion_1.ExchangeVersion[ExchangeVersion_1.ExchangeVersion.Exchange2013]));
        }
        // SeekToConditionItemView is only valid for Exchange2013 or higher
        //
        if ((this.View instanceof SeekToConditionItemView_1.SeekToConditionItemView) && //todo: better detection of types with "is"
            this.Service.RequestedServerVersion < ExchangeVersion_1.ExchangeVersion.Exchange2013) {
            throw new ServiceVersionException_1.ServiceVersionException(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.ParameterIncompatibleWithRequestVersion, "SeekToConditionItemView", ExchangeVersion_1.ExchangeVersion[ExchangeVersion_1.ExchangeVersion.Exchange2013]));
        }
        // MailboxScope is only valid for Exchange2013 or higher
        //
        if (this.MailboxScope &&
            this.Service.RequestedServerVersion < ExchangeVersion_1.ExchangeVersion.Exchange2013) {
            throw new ServiceVersionException_1.ServiceVersionException(ExtensionMethods_1.StringHelper.Format(Strings_1.Strings.ParameterIncompatibleWithRequestVersion, "MailboxScope", ExchangeVersion_1.ExchangeVersion[ExchangeVersion_1.ExchangeVersion.Exchange2013]));
        }
    }
    /**
     * @internal Writes XML attributes.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteAttributesToXml(writer) {
        this.View.WriteAttributesToXml(writer);
    }
    /**
     * @internal Writes the elements to XML writer.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        // Emit the view element
        //
        this.View.WriteToXml(writer, null);
        // Emit the Sort Order
        //
        this.View.WriteOrderByToXml(writer);
        // Emit the Parent Folder Id
        //
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.ParentFolderId);
        this.FolderId.WriteToXml(writer);
        writer.WriteEndElement();
        // Emit the MailboxScope flag
        // 
        if ((0, ExtensionMethods_1.hasValue)(this.MailboxScope)) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.MailboxScope, MailboxSearchLocation_1.MailboxSearchLocation[this.MailboxScope]);
        }
        if (!ExtensionMethods_1.StringHelper.IsNullOrEmpty(this.queryString)) {
            // Emit the QueryString
            //
            writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Messages, XmlElementNames_1.XmlElementNames.QueryString);
            if (this.ReturnHighlightTerms) {
                writer.WriteAttributeString(XmlAttributeNames_1.XmlAttributeNames.ReturnHighlightTerms, this.ReturnHighlightTerms.toString().toLowerCase()); //todo: better tolower() .ToString().ToLowerInvariant());
            }
            writer.WriteValue(this.queryString, XmlElementNames_1.XmlElementNames.QueryString);
            writer.WriteEndElement();
        }
        if (this.Service.RequestedServerVersion >= ExchangeVersion_1.ExchangeVersion.Exchange2013) {
            if (this.View.PropertySet != null) {
                this.View.PropertySet.WriteToXml(writer, ServiceObjectType_1.ServiceObjectType.Conversation);
            }
        }
    }
}
exports.FindConversationRequest = FindConversationRequest;
//# sourceMappingURL=FindConversationRequest.js.map