"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RelativeDayOfMonthTransition = void 0;
const ExtensionMethods_1 = require("../../ExtensionMethods");
const DateTime_1 = require("../../DateTime");
const DayOfTheWeek_1 = require("../../Enumerations/DayOfTheWeek");
const EwsUtilities_1 = require("../../Core/EwsUtilities");
const TimeZoneInfo_1 = require("../../TimeZoneInfo");
const XmlElementNames_1 = require("../../Core/XmlElementNames");
const XmlNamespace_1 = require("../../Enumerations/XmlNamespace");
const AbsoluteMonthTransition_1 = require("./AbsoluteMonthTransition");
/**
 * @internal Represents a time zone period transition that occurs on a relative day of a specific month.
 */
class RelativeDayOfMonthTransition extends AbsoluteMonthTransition_1.AbsoluteMonthTransition {
    constructor(timeZoneDefinition, targetPeriod) {
        super(timeZoneDefinition, targetPeriod);
        this.dayOfTheWeek = DayOfTheWeek_1.DayOfTheWeek.Sunday;
        this.weekIndex = null;
    }
    /**
     * @internal Gets the day of the week when the transition occurs.
     */
    get DayOfTheWeek() {
        return this.dayOfTheWeek;
    }
    /**
     * @internal Gets the index of the week in the month when the transition occurs.
     */
    get WeekIndex() {
        return this.weekIndex;
    }
    /**
     * @internal Creates a time zone transition time.
     * @virtual
     *
     * @return  {TimeZoneInfo.TransitionTime}      A TimeZoneInfo.TransitionTime.
     */
    CreateTransitionTime() {
        return TimeZoneInfo_1.TimeZoneInfo.TransitionTime.CreateFloatingDateRule(new DateTime_1.DateTime(this.TimeOffset.Milliseconds - DateTime_1.msToEpoch), this.Month, this.WeekIndex == -1 ? 5 : this.WeekIndex, EwsUtilities_1.EwsUtilities.EwsToSystemDayOfWeek(this.DayOfTheWeek));
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @return  {string}      XML element name.
     */
    GetXmlElementName() {
        return XmlElementNames_1.XmlElementNames.RecurringDayTransition;
    }
    /**
     * @internal Initializes this transition based on the specified transition time.
     * @virtual
     *
     * @param   {TimeZoneInfo.TransitionTime}   transitionTime   The transition time to initialize from.
     */
    InitializeFromTransitionTime(transitionTime) {
        super.InitializeFromTransitionTime(transitionTime);
        this.dayOfTheWeek = EwsUtilities_1.EwsUtilities.SystemToEwsDayOfTheWeek(transitionTime.DayOfWeek);
        // TimeZoneInfo uses week indices from 1 to 5, 5 being the last week of the month.
        // EWS uses -1 to denote the last week of the month.
        this.weekIndex = transitionTime.Week == 5 ? -1 : transitionTime.Week;
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}				jsObject	Json Object converted from XML.
     * @param   {ExchangeService}	service	The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        super.LoadFromXmlJsObject(jsObject, service);
        if (jsObject[XmlElementNames_1.XmlElementNames.DayOfWeek]) {
            this.dayOfTheWeek = DayOfTheWeek_1.DayOfTheWeek[jsObject[XmlElementNames_1.XmlElementNames.DayOfWeek]];
        }
        if (jsObject[XmlElementNames_1.XmlElementNames.Occurrence]) {
            this.weekIndex = ExtensionMethods_1.Convert.toInt(jsObject[XmlElementNames_1.XmlElementNames.Occurrence]);
        }
    }
    /**
     * @internal Writes elements to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        super.WriteElementsToXml(writer);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.DayOfWeek, DayOfTheWeek_1.DayOfTheWeek[this.dayOfTheWeek]);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Occurrence, this.weekIndex);
    }
}
exports.RelativeDayOfMonthTransition = RelativeDayOfMonthTransition;
//# sourceMappingURL=RelativeDayOfMonthTransition.js.map