"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceId = void 0;
const EwsUtilities_1 = require("../Core/EwsUtilities");
const ExtensionMethods_1 = require("../ExtensionMethods");
const XmlAttributeNames_1 = require("../Core/XmlAttributeNames");
const ComplexProperty_1 = require("./ComplexProperty");
/**
 * Represents the Id of an Exchange object.
 *
 * @abstract
 * @class ServiceId
 * @extends {ComplexProperty}
 */
class ServiceId extends ComplexProperty_1.ComplexProperty {
    constructor(uniqueId) {
        super();
        if (!ExtensionMethods_1.StringHelper.IsNullOrEmpty(uniqueId)) {
            EwsUtilities_1.EwsUtilities.ValidateParam(uniqueId, "uniqueId");
            this.UniqueId = uniqueId;
        }
    }
    /**
     * True if this instance is valid, false otherthise.
     *
     * @value   *true* if this instance is valid; otherwise, *false*.
     */
    get IsValid() {
        return this.IsValidProxy();
    }
    /** @internal IsValid proxy to be able to call super. from inherited child */
    IsValidProxy() {
        return !ExtensionMethods_1.StringHelper.IsNullOrEmpty(this.UniqueId);
    }
    /**
     * @internal Assigns from existing id.
     *
     * @param   {ServiceId}   source   The source.
     */
    Assign(source) {
        this.UniqueId = source.UniqueId;
        this.ChangeKey = source.ChangeKey;
    }
    /**
     * Determines whether the specified *ServiceId* is equal to the current *ServiceId*.
     * We do not consider the ChangeKey for ServiceId.Equals.
     *
     * @param   {any}       obj   The  to compare with the current .
     * @return  {boolean}   true if the specified  is equal to the current ; otherwise, false.
     */
    Equals(obj) {
        if (this === obj) { //object.ReferenceEquals(this, obj)) {
            return true;
        }
        else {
            var other = obj;
            if (!(other instanceof ServiceId)) { // == null) {
                return false;
            }
            else if (!(this.IsValid && other.IsValid)) {
                return false;
            }
            else {
                return this.UniqueId === other.UniqueId; //.Equals(other.UniqueId);
            }
        }
    }
    //GetHashCode(): number { return this.IsValid ? this.UniqueId.GetHashCode() : super.GetHashCode();}
    /**
     * @internal Gets the name of the XML element.
     *
     * @return  {string}      XML element name.
     */
    GetXmlElementName() { throw new Error("abstract method must implement."); }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}				jsObject	Json Object converted from XML.
     * @param   {ExchangeService}	service	The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        for (var key in jsObject) {
            switch (key) {
                case XmlAttributeNames_1.XmlAttributeNames.Id:
                    this.UniqueId = jsObject[key];
                    break;
                case XmlAttributeNames_1.XmlAttributeNames.ChangeKey:
                    this.ChangeKey = jsObject[key];
                    break;
                default:
                    break;
            }
        }
    }
    /**
     * Determines whether two ServiceId instances are equal (including ChangeKeys)
     *
     * @param   {ServiceId}   other   The ServiceId to compare with the current ServiceId.
     */
    SameIdAndChangeKey(other) {
        if (this.Equals(other)) {
            return ((this.ChangeKey == null) && (other.ChangeKey == null)) ||
                this.ChangeKey === other.ChangeKey;
        }
        else {
            return false;
        }
    }
    /**
     * Returns a *String* that represents the current *ServiceId*.
     *
     * @return  {string}      A *String* that represents the current *ServiceId*.
     */
    ToString() {
        return (this.UniqueId == null) ? "" : this.UniqueId;
    }
    /**
     * @internal Writes attributes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteAttributesToXml(writer) {
        writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.Id, this.UniqueId);
        writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.ChangeKey, this.ChangeKey);
    }
    /**
     * @internal Writes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer           The writer.
     * @param   {string}                xmlElementName   Name of the XML element.
     * @param   {XmlNamespace}          xmlNamespace     The XML namespace.
     */
    WriteToXml(writer, xmlElementName, xmlNamespace) {
        if (arguments.length > 2) {
            super.WriteToXml(writer, xmlElementName, xmlNamespace);
        }
        else if (arguments.length > 1) {
            super.WriteToXml(writer, xmlElementName);
        }
        else {
            super.WriteToXml(writer, this.GetXmlElementName());
        }
    }
}
exports.ServiceId = ServiceId;
//# sourceMappingURL=ServiceId.js.map