"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RulePredicateDateRange = void 0;
const ServiceValidationException_1 = require("../Exceptions/ServiceValidationException");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
const ComplexProperty_1 = require("./ComplexProperty");
/**
 * Represents the date and time range within which messages have been received.
 *
 * @sealed
 */
class RulePredicateDateRange extends ComplexProperty_1.ComplexProperty {
    /**
     * @internal Initializes a new instance of the **RulePredicateDateRange** class.
     */
    constructor() {
        super();
        /**
         * The start DateTime.
         */
        this.start = null; //Nullable
        /**
         * The end DateTime.
         */
        this.end = null; //Nullable
    }
    /**
     * @Nullable Gets or sets the range start date and time.
     * If Start is set to null, no start date applies.
     */
    get Start() {
        return this.start;
    }
    set Start(value) {
        this.SetFieldValue({ getValue: () => this.start, setValue: (updateValue) => { this.start = updateValue; } }, value);
    }
    /**
     * @Nullable Gets or sets the range end date and time.
     * If End is set to null, no end date applies.
     */
    get End() {
        return this.end;
    }
    set End(value) {
        this.SetFieldValue({ getValue: () => this.end, setValue: (updateValue) => { this.end = updateValue; } }, value);
    }
    /**
     * @internal Validates this instance.
     */
    InternalValidate() {
        super.InternalValidate();
        if (this.start &&
            this.end &&
            this.start.TotalMilliSeconds > this.end.TotalMilliSeconds) {
            throw new ServiceValidationException_1.ServiceValidationException("Start date time cannot be bigger than end date time.");
        }
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}                 jsObject                Json Object converted from XML.
     * @param   {ExchangeService}     service                 The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        for (let key in jsObject) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.StartDateTime:
                    this.start = service.ConvertUniversalDateTimeStringToLocalDateTime(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.EndDateTime:
                    this.end = service.ConvertUniversalDateTimeStringToLocalDateTime(jsObject[key]);
                    break;
                default:
                    break;
            }
        }
    }
    /**
     * @internal Writes the elements to XML writer.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        if (this.Start) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.StartDateTime, this.Start);
        }
        if (this.End) {
            writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.EndDateTime, this.End);
        }
    }
}
exports.RulePredicateDateRange = RulePredicateDateRange;
//# sourceMappingURL=RulePredicateDateRange.js.map