"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.YearlyPattern = void 0;
const ArgumentException_1 = require("../../../Exceptions/ArgumentException");
const ExtensionMethods_1 = require("../../../ExtensionMethods");
const Month_1 = require("../../../Enumerations/Month");
const ServiceValidationException_1 = require("../../../Exceptions/ServiceValidationException");
const Strings_1 = require("../../../Strings");
const XmlElementNames_1 = require("../../../Core/XmlElementNames");
const XmlNamespace_1 = require("../../../Enumerations/XmlNamespace");
const Recurrence_1 = require("./Recurrence");
/**
 * Represents a recurrence pattern where each occurrence happens on a specific day every year.
 */
class YearlyPattern extends Recurrence_1.Recurrence {
    constructor(startDate, month = null, dayOfMonth = null) {
        if (arguments.length === 0) {
            super();
            this.month = null;
            this.dayOfMonth = null;
            return;
        }
        super(startDate);
        this.Month = month;
        this.DayOfMonth = dayOfMonth;
    }
    /**
     * @internal Gets the name of the XML element.
     *
     * @value   The name of the XML element.
     */
    get XmlElementName() {
        return XmlElementNames_1.XmlElementNames.AbsoluteYearlyRecurrence;
    }
    /**
     * Gets or sets the month of the year when each occurrence happens.
     */
    get Month() {
        return super.GetFieldValueOrThrowIfNull(this.month, "Month");
    }
    set Month(value) {
        this.SetFieldValue({ getValue: () => this.month, setValue: (updateValue) => { this.month = updateValue; } }, value);
    }
    /**
     * Gets or sets the day of the month when each occurrence happens. DayOfMonth must be between 1 and 31.
     */
    get DayOfMonth() {
        return super.GetFieldValueOrThrowIfNull(this.dayOfMonth, "DayOfMonth");
    }
    set DayOfMonth(value) {
        if (value < 1 || value > 31) {
            throw new ArgumentException_1.ArgumentOutOfRangeException("DayOfMonth", Strings_1.Strings.DayOfMonthMustBeBetween1And31);
        }
        this.SetFieldValue({ getValue: () => this.dayOfMonth, setValue: (updateValue) => { this.dayOfMonth = updateValue; } }, value);
    }
    /**
     * @internal Validates this instance.
     */
    InternalValidate() {
        super.InternalValidate();
        if (this.month === null) {
            throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.MonthMustBeSpecifiedForRecurrencePattern);
        }
        if (this.dayOfMonth === null) {
            throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.DayOfMonthMustBeSpecifiedForRecurrencePattern);
        }
    }
    /**
     * @internal Write properties to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    InternalWritePropertiesToXml(writer) {
        super.InternalWritePropertiesToXml(writer);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.DayOfMonth, this.DayOfMonth);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.Month, Month_1.Month[this.Month]);
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}                 jsObject                Json Object converted from XML.
     * @param   {ExchangeService}     service                 The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        super.LoadFromXmlJsObject(jsObject, service);
        for (let key in jsObject) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.DayOfMonth:
                    this.dayOfMonth = ExtensionMethods_1.Convert.toNumber(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.Month:
                    this.month = Month_1.Month[jsObject[key]];
                    break;
                default:
                    break;
            }
        }
    }
}
exports.YearlyPattern = YearlyPattern;
//# sourceMappingURL=Recurrence.YearlyPattern.js.map