"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FolderPermissionCollection = void 0;
const ExchangeWebService_1 = require("../ExchangeWebService");
const FolderPermission_1 = require("./FolderPermission");
const TypeContainer_1 = require("../TypeContainer");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
const ComplexPropertyCollection_1 = require("./ComplexPropertyCollection");
/**
 * Represents a collection of folder permissions.
 *
 * @sealed
 */
class FolderPermissionCollection extends ComplexPropertyCollection_1.ComplexPropertyCollection {
    /**
     * Initializes a new instance of the **FolderPermissionCollection** class.
     *
     * @param   {Folder}   owner   The folder owner.
     */
    constructor(owner) {
        super();
        this.unknownEntries = [];
        this.isCalendarFolder = owner instanceof TypeContainer_1.TypeContainer.CalendarFolder; // owner instanceof CalendarFolder;
    }
    /**
     * Gets the name of the inner collection XML element.
     *
     * @value   XML element name.
     */
    get InnerCollectionXmlElementName() {
        return this.isCalendarFolder ? XmlElementNames_1.XmlElementNames.CalendarPermissions : XmlElementNames_1.XmlElementNames.Permissions;
    }
    /**
     * Gets the name of the collection item XML element.
     *
     * @value   XML element name.
     */
    get CollectionItemXmlElementName() {
        return this.isCalendarFolder ? XmlElementNames_1.XmlElementNames.CalendarPermission : XmlElementNames_1.XmlElementNames.Permission;
    }
    /**
     * Gets a list of unknown user Ids in the collection.
     */
    get UnknownEntries() {
        return this.unknownEntries;
    }
    /**
     * Adds a permission to the collection.
     *
     * @param   {FolderPermission}   permission   The permission to add.
     */
    Add(permission) {
        this.InternalAdd(permission);
    }
    /**
     * Adds the specified permissions to the collection.
     *
     * @param   {FolderPermission[]}   permissions   The permissions to add.
     */
    AddRange(permissions) {
        ExchangeWebService_1.EwsUtilities.ValidateParam(permissions, "permissions");
        for (let permission of permissions) {
            this.Add(permission);
        }
    }
    /**
     * Clears this collection.
     */
    Clear() {
        this.InternalClear();
    }
    /**
     * @internal Creates the complex property.
     *
     * @param   {string}   xmlElementName   Name of the XML element.
     * @return  {FolderPermission}          FolderPermission instance.
     */
    CreateComplexProperty(xmlElementName) {
        return new FolderPermission_1.FolderPermission();
    }
    /**
     * @internal Creates the default complex property.
     *
     * @return  {FolderPermission}  FolderPermission instance.
     */
    CreateDefaultComplexProperty() {
        return new FolderPermission_1.FolderPermission();
    }
    /**
     * @internal Gets the name of the collection item XML element.
     *
     * @param   {FolderPermission}      complexProperty   The complex property.
     * @return  {string}                XML element name.
     */
    GetCollectionItemXmlElementName(complexProperty) {
        return this.CollectionItemXmlElementName;
    }
    /**
     * @internal Loads from XMLJsObject collection to create a new collection item.
     *
     * @interface   IJsonCollectionDeserializer
     *
     * @param   {any}               jsObjectCollection   The json collection.
     * @param   {ExchangeService}   service          The service.
     */
    CreateFromXmlJsObjectCollection(jsObjectCollection, service) {
        let jsonFolderPermissions = jsObjectCollection[this.InnerCollectionXmlElementName];
        if (jsonFolderPermissions && jsonFolderPermissions[this.CollectionItemXmlElementName])
            jsonFolderPermissions = jsonFolderPermissions[this.CollectionItemXmlElementName];
        if (!Array.isArray(jsonFolderPermissions)) {
            //debugger;
            throw new Error("FolderPermissionCollection.ts - LoadFromXmlJsObject - Invalid xml parsing, jsonproperty must contain collectionxmlelementname and collectionitemelementname underneeth");
        }
        for (let jsonFolderPermission of jsonFolderPermissions) {
            let permission = new FolderPermission_1.FolderPermission();
            permission.LoadFromXmlJsObject(jsonFolderPermission, service);
            this.InternalAdd(permission);
        }
        if (jsObjectCollection[XmlElementNames_1.XmlElementNames.UnknownEntries]) {
            let jsonUnknownEntries = jsObjectCollection[XmlElementNames_1.XmlElementNames.UnknownEntries];
            if (typeof jsonUnknownEntries !== 'object' && !Array.isArray(jsonFolderPermissions)) {
                //debugger;
                throw new Error("FolderPermissionCollection.ts - LoadFromXmlJsObject - Invalid xml returned - check for consistency, UnknownEntries must be array type");
            }
            for (let jsonUnknownEntry of jsonUnknownEntries) {
                this.unknownEntries.push(jsonUnknownEntry);
            }
        }
    }
    /**
     * Removes a permission from the collection.
     *
     * @param   {FolderPermission}  permission   The permission to remove.
     * @return  {boolean}           True if the folder permission was successfully removed from the collection, false otherwise.
     */
    Remove(permission) {
        return this.InternalRemove(permission);
    }
    /**
     * Removes a permission from the collection.
     *
     * @param   {number}   index   The zero-based index of the permission to remove.
     */
    RemoveAt(index) {
        this.InternalRemoveAt(index);
    }
    /**
     * @internal Validates this instance.
     */
    Validate() {
        for (let permissionIndex = 0; permissionIndex < this.Items.length; permissionIndex++) {
            let permission = this.Items[permissionIndex];
            permission.Validate(this.isCalendarFolder, permissionIndex);
        }
    }
    /**
     * @internal Writes the elements to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        writer.WriteStartElement(XmlNamespace_1.XmlNamespace.Types, this.InnerCollectionXmlElementName);
        for (let folderPermission of this.Items) {
            folderPermission.WriteToXml(writer, this.GetCollectionItemXmlElementName(folderPermission), undefined, //XmlNamespace - incorrect inheritance error with typesctipt in folderpermission class if removed xmlnamespace parameter
            this.isCalendarFolder);
        }
        writer.WriteEndElement(); // this.InnerCollectionXmlElementName
    }
}
exports.FolderPermissionCollection = FolderPermissionCollection;
//# sourceMappingURL=FolderPermissionCollection.js.map