"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FolderId = void 0;
const EwsUtilities_1 = require("../Core/EwsUtilities");
const ExtensionMethods_1 = require("../ExtensionMethods");
const WellKnownFolderName_1 = require("../Enumerations/WellKnownFolderName");
const XmlAttributeNames_1 = require("../Core/XmlAttributeNames");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const ServiceId_1 = require("./ServiceId");
/**
 * Represents the Id of a folder.
 *
 * @class FolderId
 * @extends {ServiceId}
 */
class FolderId extends ServiceId_1.ServiceId {
    constructor(uniqueIdOrFolderName, mailbox) {
        arguments.length === 1 && typeof uniqueIdOrFolderName === 'string' ? super(uniqueIdOrFolderName) : super();
        if (arguments.length > 0 && typeof uniqueIdOrFolderName === 'number') {
            this.folderName = uniqueIdOrFolderName;
        }
        if (arguments.length > 1) {
            this.mailbox = mailbox;
        }
    }
    /**
     * Gets the name of the folder associated with the folder Id. Name and Id are mutually exclusive; if one is set, the other is null.
     *
     * @readonly
     * @type {WellKnownFolderName}
     */
    get FolderName() {
        return this.folderName;
    }
    /**
     * Gets the mailbox of the folder. Mailbox is only set when FolderName is set.
     *
     * @readonly
     * @type {Mailbox}
     */
    get Mailbox() {
        return this.mailbox;
    }
    /**
     * True if this instance is valid, false otherthise.
     *
     * @value   *true* if this instance is valid; otherwise, *false*.
     */
    get IsValid() {
        if ((0, ExtensionMethods_1.hasValue)(this.FolderName)) {
            return (this.Mailbox == null) || this.Mailbox.IsValid;
        }
        else {
            return super.IsValidProxy();
        }
    }
    /**
     * Determines whether the specified *FolderId* is equal to the current *FolderId*.
     * We do not consider the ChangeKey for FolderId.Equals.
     *
     * @param   {any}       obj   The  to compare with the current .
     * @return  {boolean}   true if the specified  is equal to the current ; otherwise, false.
     */
    Equals(obj) {
        if (this === obj) {
            return true;
        }
        else {
            var other = obj;
            if (!(other instanceof FolderId)) {
                return false;
            }
            else if (this.FolderName) {
                if (other.FolderName && this.FolderName === other.FolderName) {
                    if (this.Mailbox != null) {
                        return this.Mailbox.Equals(other.Mailbox);
                    }
                    else if (other.Mailbox == null) {
                        return true;
                    }
                }
            }
            else if (super.Equals(other)) {
                return true;
            }
            return false;
        }
    }
    //GetHashCode(): number { throw new Error("FolderId.ts - GetHashCode : Not implemented."); }
    /**
     * @internal Gets the name of the XML element.
     *
     * @return  {string}      XML element name.
     */
    GetXmlElementName() {
        return typeof this.folderName !== 'undefined' && this.FolderName >= 0 ? XmlElementNames_1.XmlElementNames.DistinguishedFolderId : XmlElementNames_1.XmlElementNames.FolderId;
    }
    /**
     * Returns a *String* that represents the current *FolderId*.
     *
     * @return  {string}      A *String* that represents the current *FolderId*.
     */
    ToString() {
        if (this.IsValid) {
            if ((0, ExtensionMethods_1.hasValue)(this.FolderName)) {
                if ((this.mailbox != null) && this.mailbox.IsValid) {
                    return ExtensionMethods_1.StringHelper.Format("{0} ({1})", WellKnownFolderName_1.WellKnownFolderName[this.folderName], this.Mailbox.ToString());
                }
                else {
                    return WellKnownFolderName_1.WellKnownFolderName[this.FolderName];
                }
            }
            else {
                return super.ToString();
            }
        }
        else {
            return "";
        }
    }
    /**
     * @internal Validates FolderId against a specified request version.
     *
     * @param   {ExchangeVersion}   version   The version.
     */
    Validate(version) {
        if ((0, ExtensionMethods_1.hasValue)(version)) {
            // The FolderName property is a WellKnownFolderName, an enumeration type. If the property
            // is set, make sure that the value is valid for the request version.
            if ((0, ExtensionMethods_1.hasValue)(this.FolderName)) {
                EwsUtilities_1.EwsUtilities.ValidateEnumVersionValue(WellKnownFolderName_1.WellKnownFolderName, this.FolderName, version, "WellKnownFolderName");
            }
        }
        else {
            super.Validate();
        }
    }
    /**
     * @internal Writes attributes to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteAttributesToXml(writer) {
        if (typeof this.folderName !== 'undefined' && this.FolderName >= 0) {
            writer.WriteAttributeValue(XmlAttributeNames_1.XmlAttributeNames.Id, WellKnownFolderName_1.WellKnownFolderName[this.FolderName].toLowerCase());
            if (this.Mailbox != null) {
                this.Mailbox.WriteToXml(writer, XmlElementNames_1.XmlElementNames.Mailbox);
            }
        }
        else {
            super.WriteAttributesToXml(writer);
        }
    }
}
exports.FolderId = FolderId;
//# sourceMappingURL=FolderId.js.map