"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DelegateUser = void 0;
const ExtensionMethods_1 = require("../ExtensionMethods");
const DelegatePermissions_1 = require("./DelegatePermissions");
const ServiceValidationException_1 = require("../Exceptions/ServiceValidationException");
const Strings_1 = require("../Strings");
const UserId_1 = require("./UserId");
const XmlElementNames_1 = require("../Core/XmlElementNames");
const XmlNamespace_1 = require("../Enumerations/XmlNamespace");
const ComplexProperty_1 = require("./ComplexProperty");
/**
 * Represents a delegate user.
 *
 * @sealed
 */
class DelegateUser extends ComplexProperty_1.ComplexProperty {
    constructor(primarySmtpAddressOrStandardUser) {
        super();
        this.userId = new UserId_1.UserId();
        this.permissions = new DelegatePermissions_1.DelegatePermissions();
        this.receiveCopiesOfMeetingMessages = false;
        this.viewPrivateItems = false;
        // Confusing error message refers to Calendar folder permissions when adding delegate access for a user
        // without including Calendar Folder permissions.
        //
        this.receiveCopiesOfMeetingMessages = false;
        this.viewPrivateItems = false;
        if (typeof primarySmtpAddressOrStandardUser === 'string') {
            this.userId.PrimarySmtpAddress = primarySmtpAddressOrStandardUser;
        }
        else {
            this.userId.StandardUser = primarySmtpAddressOrStandardUser;
        }
    }
    /**
     * Gets the user Id of the delegate user.
     */
    get UserId() {
        return this.userId;
    }
    /**
     * Gets the list of delegate user's permissions.
     */
    get Permissions() {
        return this.permissions;
    }
    /**
     * Gets or sets a value indicating if the delegate user should receive copies of meeting requests.
     */
    get ReceiveCopiesOfMeetingMessages() {
        return this.receiveCopiesOfMeetingMessages;
    }
    set ReceiveCopiesOfMeetingMessages(value) {
        this.receiveCopiesOfMeetingMessages = value;
    }
    /**
     * Gets or sets a value indicating if the delegate user should be able to view the principal's private items.
     */
    get ViewPrivateItems() {
        return this.viewPrivateItems;
    }
    set ViewPrivateItems(value) {
        this.viewPrivateItems = value;
    }
    /**
     * @internal Validates this instance.
     */
    InternalValidate() {
        if (this.UserId == null) {
            throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.UserIdForDelegateUserNotSpecified);
        }
        else if (!this.UserId.IsValid()) {
            throw new ServiceValidationException_1.ServiceValidationException(Strings_1.Strings.DelegateUserHasInvalidUserId);
        }
    }
    /**
     * @internal Loads service object from XML.
     *
     * @param   {any}                 jsObject                Json Object converted from XML.
     * @param   {ExchangeService}     service                 The service.
     */
    LoadFromXmlJsObject(jsObject, service) {
        for (let key in jsObject) {
            switch (key) {
                case XmlElementNames_1.XmlElementNames.UserId:
                    this.userId = new UserId_1.UserId();
                    this.userId.LoadFromXmlJsObject(jsObject[key], service);
                    break;
                case XmlElementNames_1.XmlElementNames.DelegatePermissions:
                    this.permissions.Reset();
                    this.permissions.LoadFromXmlJsObject(jsObject[key], service);
                    break;
                case XmlElementNames_1.XmlElementNames.ReceiveCopiesOfMeetingMessages:
                    this.receiveCopiesOfMeetingMessages = ExtensionMethods_1.Convert.toBool(jsObject[key]);
                    break;
                case XmlElementNames_1.XmlElementNames.ViewPrivateItems:
                    this.viewPrivateItems = ExtensionMethods_1.Convert.toBool(jsObject[key]);
                    break;
                default:
                    break;
            }
        }
    }
    /**
     * @internal Validates this instance for AddDelegate.
     */
    ValidateAddDelegate() {
        this.permissions.ValidateAddDelegate();
    }
    /**
     * @internal Validates this instance for UpdateDelegate.
     */
    ValidateUpdateDelegate() {
        this.permissions.ValidateUpdateDelegate();
    }
    /**
     * @internal Writes the elements to XML.
     *
     * @param   {EwsServiceXmlWriter}   writer   The writer.
     */
    WriteElementsToXml(writer) {
        this.UserId.WriteToXml(writer, XmlElementNames_1.XmlElementNames.UserId);
        this.Permissions.WriteToXml(writer, XmlElementNames_1.XmlElementNames.DelegatePermissions);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.ReceiveCopiesOfMeetingMessages, this.ReceiveCopiesOfMeetingMessages);
        writer.WriteElementValue(XmlNamespace_1.XmlNamespace.Types, XmlElementNames_1.XmlElementNames.ViewPrivateItems, this.ViewPrivateItems);
    }
}
exports.DelegateUser = DelegateUser;
//# sourceMappingURL=DelegateUser.js.map