'use strict';

var path = require('path');
var electron = require('electron');
var i18next = require('i18next');

function _interopDefault (e) { return e && e.__esModule ? e : { default: e }; }

var path__default = /*#__PURE__*/_interopDefault(path);
var i18next__default = /*#__PURE__*/_interopDefault(i18next);

const fallbackLng = 'en';
const byteUnits = ['byte', 'kilobyte', 'megabyte', 'gigabyte', 'terabyte', 'petabyte'];
const formatBytes = bytes => {
  const order = Math.min(Math.floor(Math.log(bytes) / Math.log(1024)), byteUnits.length - 1);
  const unit = byteUnits[order];
  if (!unit) {
    return '???';
  }
  const formatter = new Intl.NumberFormat(undefined, {
    notation: 'compact',
    style: 'unit',
    unit,
    maximumFractionDigits: 1
  });
  return formatter.format(bytes / Math.pow(1024, order));
};
const formatByteSpeed = bytesPerSecond => {
  const order = Math.min(Math.floor(Math.log(bytesPerSecond) / Math.log(1024)), byteUnits.length - 1);
  const unit = byteUnits[order];
  if (!unit) {
    return '???';
  }
  const formatter = new Intl.NumberFormat(undefined, {
    notation: 'compact',
    style: 'unit',
    unit: `${unit}-per-second`,
    maximumFractionDigits: 1
  });
  return formatter.format(bytesPerSecond / Math.pow(1024, order));
};
const formatPercentage = ratio => {
  const formatter = new Intl.NumberFormat(undefined, {
    style: 'percent',
    maximumFractionDigits: 0
  });
  return formatter.format(ratio);
};
const formatDuration = duration => {
  const formatter = new Intl.RelativeTimeFormat(undefined, {
    style: 'narrow',
    numeric: 'always'
  });
  duration /= 1000;
  if (duration / 60 < 1) {
    return formatter.format(duration, 'second');
  }
  duration /= 60;
  if (duration / 60 < 1) {
    return formatter.format(duration, 'minute');
  }
  duration /= 60;
  if (duration / 24 < 1) {
    return formatter.format(duration, 'hour');
  }
  duration /= 24;
  if (duration / 7 < 1) {
    return formatter.format(duration, 'day');
  }
  duration /= 7;
  if (duration / 30 < 1) {
    return formatter.format(duration, 'week');
  }
  duration /= 30;
  if (duration / 12 < 1) {
    return formatter.format(duration, 'month');
  }
  duration /= 12;
  return formatter.format(duration, 'year');
};
const interpolation = {
  format: (value, format, lng) => {
    if (value instanceof Date && !Number.isNaN(value.getTime())) {
      return new Intl.DateTimeFormat(lng).format(value);
    }
    switch (format) {
      case 'byteSize':
        return formatBytes(value);
      case 'byteSpeed':
        return formatByteSpeed(value);
      case 'percentage':
        return formatPercentage(value);
      case 'duration':
        return formatDuration(value);
    }
    return String(value);
  }
};

var resources = {
  'de-DE': () => Promise.resolve().then(function () { return require('./de-DE.i18n-Dit6C4kK.js'); }),
  'en': () => Promise.resolve().then(function () { return require('./en.i18n-yTu9CcvG.js'); }),
  'es': () => Promise.resolve().then(function () { return require('./es.i18n-ElGg1WHL.js'); }),
  'fi': () => Promise.resolve().then(function () { return require('./fi.i18n-aoBgNzCt.js'); }),
  'sv': () => Promise.resolve().then(function () { return require('./sv.i18n-hAuWAHhV.js'); }),
  'no': () => Promise.resolve().then(function () { return require('./no.i18n-d7-NyvoU.js'); }),
  'fr': () => Promise.resolve().then(function () { return require('./fr.i18n-xlxgrvr0.js'); }),
  'hu': () => Promise.resolve().then(function () { return require('./hu.i18n-68OzyLj8.js'); }),
  'it-IT': () => Promise.resolve().then(function () { return require('./it-IT.i18n-fQannT1r.js'); }),
  'ja': () => Promise.resolve().then(function () { return require('./ja.i18n-x0gh7Xie.js'); }),
  'pl': () => Promise.resolve().then(function () { return require('./pl.i18n-GpTyPFE3.js'); }),
  'pt-BR': () => Promise.resolve().then(function () { return require('./pt-BR.i18n-gDLGPKy6.js'); }),
  'ru': () => Promise.resolve().then(function () { return require('./ru.i18n-6VkRzdj5.js'); }),
  'tr-TR': () => Promise.resolve().then(function () { return require('./tr-TR.i18n-GmrX7VAX.js'); }),
  'uk-UA': () => Promise.resolve().then(function () { return require('./uk-UA.i18n-Ah5dKRFd.js'); }),
  'zh-CN': () => Promise.resolve().then(function () { return require('./zh-CN.i18n-SdfKVrgS.js'); }),
  'zh-TW': () => Promise.resolve().then(function () { return require('./zh-TW.i18n-dDGpEwdN.js'); })
};

const invokeWithRetry = (channel, retryOptions = {}, ...args) => {
  const {
    maxAttempts = 3,
    retryDelay = 1000,
    logRetries = true,
    shouldRetry = () => true
  } = retryOptions;
  const attemptInvoke = async attempt => {
    try {
      const result = await electron.ipcRenderer.invoke(channel, ...args);

      // Check if result indicates failure (for channels that return success flags)
      if (result && typeof result === 'object' && 'success' in result && result.success === false) {
        throw new Error(`IPC call failed: ${channel} returned success: false`);
      }
      return result;
    } catch (error) {
      const isLastAttempt = attempt >= maxAttempts;
      if (logRetries) {
        console.log(`IPC call failed: ${channel} (attempt ${attempt}/${maxAttempts})`, error);
      }
      if (isLastAttempt || !shouldRetry(error, attempt)) {
        if (logRetries) {
          console.error(`IPC call giving up: ${channel} after ${attempt} attempts`, error);
        }
        throw error;
      }
      if (logRetries) {
        console.log(`IPC call retrying: ${channel} in ${retryDelay}ms... (attempt ${attempt + 1}/${maxAttempts})`);
      }
      await new Promise(resolve => setTimeout(resolve, retryDelay));
      return attemptInvoke(attempt + 1);
    }
  };
  return attemptInvoke(1);
};

function _defineProperty(e, r, t) {
  return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, {
    value: t,
    enumerable: !0,
    configurable: !0,
    writable: !0
  }) : e[r] = t, e;
}
function _toPrimitive(t, r) {
  if ("object" != typeof t || !t) return t;
  var e = t[Symbol.toPrimitive];
  if (void 0 !== e) {
    var i = e.call(t, r || "default");
    if ("object" != typeof i) return i;
    throw new TypeError("@@toPrimitive must return a primitive value.");
  }
  return ("string" === r ? String : Number)(t);
}
function _toPropertyKey(t) {
  var i = _toPrimitive(t, "string");
  return "symbol" == typeof i ? i : i + "";
}

/**
 * Internal picker provider that uses the React UI and desktopCapturer cache.
 * This provider works in both main and renderer processes:
 * - handleDisplayMediaRequest: called from main process, uses ipc.ts state
 * - initialize: called from renderer process, preloads UI and cache
 *
 * Note: This provider needs access to state in ipc.ts. The factory will
 * pass a handler function that encapsulates the internal picker logic.
 */
class InternalPickerProvider {
  constructor() {
    _defineProperty(this, "type", 'internal');
    _defineProperty(this, "requiresInternalUI", true);
    _defineProperty(this, "requiresCacheWarming", true);
    _defineProperty(this, "isInitialized", false);
    _defineProperty(this, "handleRequestFn", null);
    _defineProperty(this, "initializeFn", null);
  }
  /**
   * Set the handler function from main process (ipc.ts)
   * This encapsulates all the internal picker logic with access to ipc.ts state
   */
  setHandleRequestHandler(handler) {
    this.handleRequestFn = handler;
  }

  /**
   * Set the initialize function from renderer process (video-call-window.ts)
   * This encapsulates the UI preload and cache warming logic
   */
  setInitializeHandler(handler) {
    this.initializeFn = handler;
  }
  handleDisplayMediaRequest(callback) {
    if (this.handleRequestFn) {
      this.handleRequestFn(callback);
    } else {
      console.error('InternalPickerProvider: handleRequest handler not set. This should be set by ipc.ts');
      callback({
        video: false
      });
    }
  }
  async initialize() {
    if (this.isInitialized) return;
    console.log('Screen picker [internal]: initializing cache and UI');
    if (this.initializeFn) {
      await this.initializeFn();
    } else {
      console.warn('InternalPickerProvider: initialize handler not set. Call setInitializeHandler() first.');
    }
    this.isInitialized = true;
  }
  cleanup() {
    // Clear caches, reset state
    this.isInitialized = false;
  }
}

class PortalPickerProvider {
  constructor() {
    _defineProperty(this, "type", 'portal');
    _defineProperty(this, "requiresInternalUI", false);
    _defineProperty(this, "requiresCacheWarming", false);
  }
  handleDisplayMediaRequest(callback) {
    // On Linux/Wayland, calling getSources() triggers the XDG portal picker.
    // The portal typically returns exactly one source on selection or an empty array
    // on cancellation; we defensively check for > 0 and use only the first source.
    console.log('Screen picker [portal]: triggering XDG portal picker via getSources()');
    electron.desktopCapturer.getSources({
      types: ['screen', 'window']
    }).then(sources => {
      if (sources.length > 0) {
        // User selected a source via portal picker
        callback({
          video: sources[0]
        });
      } else {
        // User cancelled or no source available
        console.warn('Screen picker [portal]: No source selected by user');
        callback({
          video: false
        });
      }
    }).catch(error => {
      console.error('Screen picker [portal]: Failed to get source from XDG portal:', error);
      callback({
        video: false
      });
    });
  }
  async initialize() {
    // Portal picker requires no initialization
    console.log('Screen picker [portal]: initialized (no-op)');
  }
  cleanup() {
    // Portal picker requires no cleanup
    console.log('Screen picker [portal]: cleanup (no-op)');
  }
}

function detectPickerType() {
  if (process.platform === 'linux') {
    const sessionType = process.env.XDG_SESSION_TYPE;
    const desktop = process.env.XDG_CURRENT_DESKTOP || '';
    const isWayland = sessionType === 'wayland';
    const hasPortal = /GNOME|KDE|XFCE|Cinnamon|MATE|Pantheon|Budgie|Unity/i.test(desktop);
    if (isWayland || hasPortal) {
      return 'portal';
    }
  }

  // Future: Add macOS native picker detection here
  // if (process.platform === 'darwin' && hasMacOSNativePicker()) {
  //   return 'native-macos';
  // }

  return 'internal';
}
let cachedProvider = null;
function createScreenPicker() {
  if (cachedProvider) return cachedProvider;
  const type = detectPickerType();
  switch (type) {
    case 'portal':
      cachedProvider = new PortalPickerProvider();
      break;
    case 'internal':
    default:
      cachedProvider = new InternalPickerProvider();
  }
  console.log(`Screen picker: using ${cachedProvider.type} provider`);
  return cachedProvider;
}

const MAX_INIT_ATTEMPTS = 10;
const MAX_RECOVERY_ATTEMPTS = 3;
const LOADING_TIMEOUT_MS = 15000;
const LOADING_SHOW_DELAY = 500;
const ERROR_SHOW_DELAY = 800;
const ERROR_SHOW_DELAY_404 = 1500;
const VIDEO_CALL_PRELOAD_RELATIVE_PATH = 'preload/preload.js';
const RECOVERY_DELAYS = {
  webviewReload: 1000,
  urlRefresh: 2000,
  fullReinitialize: 3000
};
let initAttempts = 0;
let isWindowDestroying = false;
let screenPickerModule = null;
let screenPickerModulePromise = null;
const state = {
  url: null,
  status: 'idle',
  recoveryAttempt: 0,
  webview: null,
  shouldAutoOpenDevtools: false,
  isReloading: false,
  hasInitialLoadCompleted: false,
  errorMessage: null
};
let loadingTimeout = null;
let recoveryTimeout = null;
let loadingDisplayTimeout = null;
let errorDisplayTimeout = null;
const initializeI18n = async () => {
  try {
    var _resources;
    const languageResult = await invokeWithRetry('video-call-window/get-language', {
      maxAttempts: 3,
      retryDelay: 500,
      logRetries: "production" === 'development'
    });
    const lng = (languageResult === null || languageResult === void 0 ? void 0 : languageResult.language) || fallbackLng;
    if ("production" === 'development') ;
    await i18next__default.default.init({
      lng,
      fallbackLng,
      resources: {
        ...(lng && lng !== fallbackLng && {
          [lng]: {
            translation: await ((_resources = resources[lng]) === null || _resources === void 0 ? void 0 : _resources.call(resources))
          }
        }),
        [fallbackLng]: {
          translation: await resources[fallbackLng]()
        }
      },
      interpolation,
      initImmediate: true
    });
    if ("production" === 'development') ;
  } catch (error) {
    console.error('Failed to initialize i18n for video call window:', error);
    throw error;
  }
};
const t = (key, defaultValue) => {
  return i18next__default.default.t(key, {
    defaultValue
  });
};
const clearAllTimeouts = () => {
  if (loadingTimeout) {
    clearTimeout(loadingTimeout);
    loadingTimeout = null;
  }
  if (recoveryTimeout) {
    clearTimeout(recoveryTimeout);
    recoveryTimeout = null;
  }
  if (loadingDisplayTimeout) {
    clearTimeout(loadingDisplayTimeout);
    loadingDisplayTimeout = null;
  }
  if (errorDisplayTimeout) {
    clearTimeout(errorDisplayTimeout);
    errorDisplayTimeout = null;
  }
};
const updateLoadingUI = (show, isReloading = false) => {
  const overlay = document.getElementById('loading-overlay-root');
  if (!overlay) return;
  if (show) {
    overlay.classList.add('show');
    const text = overlay.querySelector('.loading-text');
    const description = overlay.querySelector('.loading-description');
    if (text) {
      text.textContent = isReloading ? t('videoCall.loading.reloading', 'Reloading...') : t('videoCall.loading.initial', 'Loading video call...');
    }
    if (description) {
      description.textContent = t('videoCall.loading.description', 'Please wait while we connect you');
    }
    // Hide webview when showing loading
    if (state.webview) {
      state.webview.style.visibility = 'hidden';
    }
  } else {
    overlay.classList.remove('show');
  }
};
const updateErrorUI = (show, message = null) => {
  const overlay = document.getElementById('error-overlay-root');
  if (!overlay) return;
  if (show) {
    overlay.classList.add('show');
    const title = overlay.querySelector('.error-title');
    const announcement = overlay.querySelector('.error-announcement');
    const errorMsg = overlay.querySelector('.error-message');
    const reloadButton = overlay.querySelector('#error-reload-button');
    if (title) {
      title.textContent = t('videoCall.error.title', 'Connection Failed');
    }
    if (announcement) {
      announcement.textContent = t('videoCall.error.announcement', 'Unable to connect to video call');
    }
    if (errorMsg) {
      if (message) {
        errorMsg.textContent = message;
        errorMsg.classList.add('show');
      } else {
        errorMsg.classList.remove('show');
      }
    }
    if (reloadButton) {
      reloadButton.textContent = t('videoCall.error.reload', 'Reload');
    }
    // Hide webview when showing error
    if (state.webview) {
      state.webview.style.visibility = 'hidden';
    }
  } else {
    overlay.classList.remove('show');
  }
};
const showLoadingWithDelay = (isReloading = false) => {
  if (loadingDisplayTimeout) {
    clearTimeout(loadingDisplayTimeout);
  }
  loadingDisplayTimeout = setTimeout(() => {
    if (state.status === 'loading' && !state.errorMessage) {
      updateLoadingUI(true, isReloading);
    }
    loadingDisplayTimeout = null;
  }, LOADING_SHOW_DELAY);
};
const showErrorWithDelay = (message, is404Like = false) => {
  if (errorDisplayTimeout) {
    clearTimeout(errorDisplayTimeout);
  }
  const delay = is404Like ? ERROR_SHOW_DELAY_404 : ERROR_SHOW_DELAY;
  errorDisplayTimeout = setTimeout(() => {
    if (state.status === 'error') {
      updateErrorUI(true, message);
    }
    errorDisplayTimeout = null;
  }, delay);
};
const attemptAutoRecovery = () => {
  if (state.recoveryAttempt >= MAX_RECOVERY_ATTEMPTS) {
    state.status = 'error';
    state.errorMessage = t('videoCall.error.maxRetriesReached', 'Failed to load after multiple attempts');
    updateLoadingUI(false);
    updateErrorUI(true, state.errorMessage);
    return;
  }
  const currentAttempt = state.recoveryAttempt + 1;
  state.recoveryAttempt = currentAttempt;
  state.isReloading = true;
  let delay;
  switch (currentAttempt) {
    case 1:
      delay = RECOVERY_DELAYS.webviewReload;
      break;
    case 2:
      delay = RECOVERY_DELAYS.urlRefresh;
      break;
    case 3:
      delay = RECOVERY_DELAYS.fullReinitialize;
      break;
    default:
      return;
  }
  recoveryTimeout = setTimeout(() => {
    const webview = state.webview;
    switch (currentAttempt) {
      case 1:
        if (webview) {
          webview.reload();
        }
        break;
      case 2:
        if (webview && state.url) {
          try {
            const validatedUrl = validateVideoCallUrl(state.url);
            webview.src = 'about:blank';
            setTimeout(() => {
              if (webview) {
                webview.src = validatedUrl;
              }
            }, 500);
          } catch (error) {
            console.error('Video call window: URL validation failed during recovery:', error);
            console.error('Video call window: Skipping URL refresh recovery step, proceeding to next recovery attempt');
            if (recoveryTimeout) {
              clearTimeout(recoveryTimeout);
              recoveryTimeout = null;
            }
            state.recoveryAttempt = currentAttempt;
            attemptAutoRecovery();
            return;
          }
        }
        break;
      case 3:
        window.location.reload();
        break;
    }
    recoveryTimeout = null;
  }, delay);
};
const checkForClosePage = async url => {
  if (url.includes('/close.html') || url.includes('/close2.html')) {
    console.log('Video call window: Close page detected, scheduling window close:', url);
    state.status = 'closing';
    clearAllTimeouts();
    updateLoadingUI(false);
    updateErrorUI(false);
    setTimeout(async () => {
      try {
        await invokeWithRetry('video-call-window/close-requested', {
          maxAttempts: 2,
          retryDelay: 500,
          logRetries: "production" === 'development'
        });
        if ("production" === 'development') ;
      } catch (error) {
        console.error('Video call window: Failed to send close request:', error);
      }
    }, 1000);
  }
};
const setupWebviewEventHandlers = webview => {
  const webviewElement = webview;
  const handleLoadStart = () => {
    console.log('Video call window: Load started');
    if (state.status === 'closing') {
      console.log('Video call window: Skipping load start handling - window is closing');
      return;
    }
    if (state.hasInitialLoadCompleted && !state.isReloading && state.recoveryAttempt === 0) {
      console.log('Video call window: Skipping loading UI - initial load already completed');
      return;
    }
    state.status = 'loading';
    state.errorMessage = null;
    state.isReloading = false;
    clearAllTimeouts();
    showLoadingWithDelay(false);
    invokeWithRetry('video-call-window/webview-loading', {
      maxAttempts: 2,
      retryDelay: 500,
      logRetries: "production" === 'development'
    }).catch(error => {
      console.error('Video call window: Failed to send webview loading state:', error);
    });
    loadingTimeout = setTimeout(() => {
      loadingTimeout = null;
      attemptAutoRecovery();
    }, LOADING_TIMEOUT_MS);
  };
  const handleNavigate = event => {
    console.log('Video call window: Navigation event:', event.url);
    checkForClosePage(event.url);
  };
  const handleDomReady = () => {
    console.log('Video call window: Webview DOM ready');
    if (state.shouldAutoOpenDevtools) {
      console.log('Video call window: Auto-opening devtools for webview');
      invokeWithRetry('video-call-window/open-webview-dev-tools', {
        maxAttempts: 2,
        retryDelay: 500,
        logRetries: "production" === 'development'
      }).then(success => {
        if (success) {
          console.log('Video call window: Successfully auto-opened devtools');
        } else {
          console.warn('Video call window: Failed to auto-open devtools');
        }
      }).catch(error => {
        console.error('Video call window: Error auto-opening devtools:', error);
      });
    }
  };
  const handleFinishLoad = () => {
    console.log('Video call window: Webview finished loading (all resources loaded)');
    state.recoveryAttempt = 0;
    state.status = 'ready';
    state.hasInitialLoadCompleted = true;
    state.isReloading = false;
    clearAllTimeouts();
    updateLoadingUI(false);
    updateErrorUI(false);
    if (state.webview) {
      state.webview.style.visibility = 'visible';
    }
    invokeWithRetry('video-call-window/webview-ready', {
      maxAttempts: 2,
      retryDelay: 500,
      logRetries: "production" === 'development'
    }).catch(error => {
      console.error('Video call window: Failed to send webview ready state:', error);
    });

    // Initialize the screen picker (provider handles its own needs)
    const provider = createScreenPicker();

    // Set initialize handler for internal picker
    if (provider instanceof InternalPickerProvider) {
      provider.setInitializeHandler(async () => {
        await preloadScreenSharePicker();
        electron.ipcRenderer.invoke('video-call-window/prewarm-capturer-cache').catch(() => {
          // Silent failure - cache warming is optional optimization
        });
      });
    }
    provider.initialize().catch(error => {
      console.error('Screen picker initialization failed:', error);
    });
  };
  const handleDidFailLoad = event => {
    const errorInfo = {
      errorCode: event.errorCode,
      errorDescription: event.errorDescription,
      validatedURL: event.validatedURL,
      isMainFrame: event.isMainFrame
    };
    console.error('Video call window: Webview failed to load:', errorInfo);
    if (event.isMainFrame) {
      clearAllTimeouts();
      state.status = 'error';
      state.errorMessage = `${event.errorDescription} (${event.errorCode})`;
      updateLoadingUI(false);
      const is404LikeError = [-6, -105, -106].includes(event.errorCode);
      showErrorWithDelay(state.errorMessage, is404LikeError);
      electron.ipcRenderer.invoke('video-call-window/webview-failed', `${event.errorDescription} (${event.errorCode})`).catch(error => {
        console.error('Video call window: Failed to send webview failed state:', error);
      });
    }
  };
  const handleCrashed = () => {
    console.error('Video call window: Webview crashed');
    clearAllTimeouts();
    state.status = 'error';
    state.errorMessage = t('videoCall.error.crashed', 'Webview crashed');
    updateLoadingUI(false);
    updateErrorUI(true, state.errorMessage);
    invokeWithRetry('video-call-window/webview-failed', {
      maxAttempts: 2,
      retryDelay: 500,
      logRetries: "production" === 'development'
    }, 'Webview crashed').catch(error => {
      console.error('Video call window: Failed to send webview failed state:', error);
    });
  };
  const handleWebviewAttached = () => {
    console.log('Video call window: Webview attached');
    invokeWithRetry('video-call-window/webview-created', {
      maxAttempts: 2,
      retryDelay: 500,
      logRetries: "production" === 'development'
    }).catch(error => {
      console.error('Video call window: Failed to send webview created state:', error);
    });
    if (state.shouldAutoOpenDevtools) {
      setTimeout(() => {
        invokeWithRetry('video-call-window/open-webview-dev-tools', {
          maxAttempts: 2,
          retryDelay: 500,
          logRetries: "production" === 'development'
        }).then(success => {
          if (success) {
            console.log('Video call window: Successfully auto-opened devtools on attach');
          }
        }).catch(error => {
          console.error('Video call window: Error auto-opening devtools on attach:', error);
        });
      }, 100);
    }
  };
  webviewElement.addEventListener('webview-attached', handleWebviewAttached);
  webviewElement.addEventListener('did-start-loading', handleLoadStart);
  webviewElement.addEventListener('did-navigate', handleNavigate);
  webviewElement.addEventListener('dom-ready', handleDomReady);
  webviewElement.addEventListener('did-finish-load', handleFinishLoad);
  webviewElement.addEventListener('did-fail-load', handleDidFailLoad);
  webviewElement.addEventListener('crashed', handleCrashed);
};
const validateVideoCallUrl = url => {
  try {
    const parsedUrl = new URL(url);
    const allowedProtocols = ['http:', 'https:'];
    if (!allowedProtocols.includes(parsedUrl.protocol)) {
      throw new Error(`Invalid URL protocol: ${parsedUrl.protocol}. Only http: and https: are allowed.`);
    }
    return parsedUrl.href;
  } catch (error) {
    if (error instanceof TypeError) {
      throw new Error(`Invalid URL format: ${url}`);
    }
    throw error;
  }
};
const createWebview = url => {
  const container = document.getElementById('webview-container');
  if (!container) {
    throw new Error('Webview container not found');
  }
  const validatedUrl = validateVideoCallUrl(url);
  const htmlPath = window.location.pathname;
  const appDir = path__default.default.posix.dirname(htmlPath);
  const preloadRelativePath = path__default.default.posix.join(appDir, VIDEO_CALL_PRELOAD_RELATIVE_PATH);

  // Convert to file:// URL format required by Electron webview preload attribute
  // window.location.origin is 'file://' on all platforms
  // path.posix.join produces absolute paths starting with '/'
  // Combining them: 'file://' + '/path' = 'file:///path' (correct format)
  // Works on Linux: file:///home/user/app/preload/preload.js
  // Works on macOS: file:///Users/user/app/preload/preload.js
  // Works on Windows: file:///C:/Users/user/app/preload/preload.js
  const preloadPath = `${window.location.origin}${preloadRelativePath}`;
  const webview = document.createElement('webview');
  webview.setAttribute('preload', preloadPath);
  webview.setAttribute('webpreferences', 'nodeIntegration,nativeWindowOpen=true');
  webview.setAttribute('allowpopups', 'true');
  webview.setAttribute('partition', 'persist:jitsi-session');
  webview.src = validatedUrl;
  webview.style.cssText = `
    position: absolute;
    inset: 0;
    width: 100%;
    height: 100%;
    visibility: hidden;
    z-index: 0;
  `;
  container.appendChild(webview);
  state.webview = webview;
  state.url = url;
  setupWebviewEventHandlers(webview);
};
const preloadScreenSharePicker = async () => {
  if (screenPickerModule) {
    return;
  }
  try {
    if ("production" === 'development') ;
    // Track import promise to prevent concurrent imports
    if (!screenPickerModulePromise) {
      screenPickerModulePromise = Promise.resolve().then(function () { return require('./screenSharePickerMount-sEQPcA5C.js'); });
    }
    screenPickerModule = await screenPickerModulePromise;
    screenPickerModulePromise = null;
    screenPickerModule.mount(); // Mount only (stays hidden until IPC event)
    if ("production" === 'development') ;
  } catch (error) {
    console.error('Video call window: Failed to preload React:', error);
    screenPickerModulePromise = null;
  }
};
const handleReload = () => {
  console.log('Video call window: Manual reload requested');
  state.isReloading = true;
  state.status = 'loading';
  state.errorMessage = null;
  state.hasInitialLoadCompleted = false;
  state.recoveryAttempt = 0;
  clearAllTimeouts();
  updateErrorUI(false);
  updateLoadingUI(true, true);
  const webview = state.webview;
  if (webview) {
    webview.reload();
  }
};
const FALLBACK_RELOAD_KEY = 'video-call-fallback-reloads';
const MAX_FALLBACK_RELOADS = 3;
const showFallbackUI = () => {
  const reloadCount = parseInt(sessionStorage.getItem(FALLBACK_RELOAD_KEY) || '0', 10);
  const canAutoReload = reloadCount < MAX_FALLBACK_RELOADS;
  const fallbackContainer = document.createElement('div');
  fallbackContainer.style.cssText = `
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    display: flex;
    align-items: center;
    justify-content: center;
    background: #2f343d;
    color: white;
    font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Helvetica Neue', Arial, sans-serif;
    z-index: 9999;
  `;
  fallbackContainer.innerHTML = `
    <div style="text-align: center;">
      <h2 style="color: #fff; margin: 0;">Video Call Unavailable</h2>
      <p style="color: #ccc; margin: 10px 0;">Unable to initialize video call window</p>
      <p style="color: #999; margin: 10px 0; font-size: 14px;">${canAutoReload ? 'Retrying automatically in 3 seconds...' : 'Please close and reopen the video call.'}</p>
    </div>
  `;
  document.body.appendChild(fallbackContainer);
  if (canAutoReload) {
    sessionStorage.setItem(FALLBACK_RELOAD_KEY, String(reloadCount + 1));
    setTimeout(() => {
      window.location.reload();
    }, 3000);
  }
};
const start = async () => {
  if (isWindowDestroying) {
    console.log('Video call window: Skipping initialization - window is being destroyed');
    return;
  }
  initAttempts++;
  if (initAttempts > 1 || "production" === 'development') {
    console.log(`Video call window initialization attempt ${initAttempts}/${MAX_INIT_ATTEMPTS}`);
  }
  try {
    if (document.readyState === 'loading') {
      if ("production" === 'development') ;
      return new Promise(resolve => {
        document.addEventListener('DOMContentLoaded', () => {
          start().then(resolve).catch(resolve);
        });
      });
    }
    await initializeI18n();
    const params = new URLSearchParams(window.location.search);
    let url = params.get('url');
    const autoOpenDevtools = params.get('autoOpenDevtools') === 'true';
    state.shouldAutoOpenDevtools = autoOpenDevtools;
    if (!url) {
      // Try to get URL via IPC if not provided in query params
      try {
        const urlResult = await invokeWithRetry('video-call-window/request-url', {
          maxAttempts: 2,
          retryDelay: 500,
          logRetries: "production" === 'development'
        });
        if (urlResult !== null && urlResult !== void 0 && urlResult.success && urlResult !== null && urlResult !== void 0 && urlResult.url) {
          url = urlResult.url;
          if (urlResult.autoOpenDevtools !== undefined) {
            state.shouldAutoOpenDevtools = urlResult.autoOpenDevtools;
          }
        }
      } catch (error) {
        console.error('Video call window: Failed to request URL via IPC:', error);
      }
    }
    if (!url) {
      // No URL available - show error state instead of stuck loading screen
      state.status = 'error';
      state.errorMessage = t('videoCall.error.noUrl', 'No video call URL provided');
      updateLoadingUI(false);
      showErrorWithDelay(state.errorMessage, false);
      return;
    }
    createWebview(url);
    await invokeWithRetry('video-call-window/url-received', {
      maxAttempts: 2,
      retryDelay: 500,
      logRetries: "production" === 'development'
    });
    state.status = 'loading';
    // Show loading immediately for initial load
    updateLoadingUI(true, false);
    const handshakeRetryOptions = {
      maxAttempts: 3,
      retryDelay: 1000,
      logRetries: "production" === 'development'
    };
    await invokeWithRetry('video-call-window/handshake', handshakeRetryOptions);
    await invokeWithRetry('video-call-window/renderer-ready', handshakeRetryOptions);
    if ("production" === 'development') ;
  } catch (error) {
    console.error(`Video call window initialization failed (attempt ${initAttempts}):`, error);
    if (initAttempts < MAX_INIT_ATTEMPTS && !isWindowDestroying) {
      console.log('Video call window: Retrying initialization in 1 second...');
      setTimeout(() => {
        if (!isWindowDestroying) {
          start().catch(retryError => {
            console.error('Video call window retry also failed:', retryError);
          });
        }
      }, 1000);
    } else if (!isWindowDestroying) {
      console.error('Video call window: Max initialization attempts reached, showing fallback UI');
      showFallbackUI();
    }
  }
};
window.addEventListener('error', event => {
  console.error('Video call window global error:', event.error);
});
window.addEventListener('unhandledrejection', event => {
  console.error('Video call window unhandled rejection:', event.reason);
  event.preventDefault();
});

// IPC listener for screen picker (with cleanup on page unload)
const handleOpenScreenPicker = () => {
  (async () => {
    try {
      if (!screenPickerModule) {
        // Track import promise to prevent concurrent imports
        if (!screenPickerModulePromise) {
          screenPickerModulePromise = Promise.resolve().then(function () { return require('./screenSharePickerMount-sEQPcA5C.js'); });
        }
        screenPickerModule = await screenPickerModulePromise;
        screenPickerModulePromise = null;
      }
      screenPickerModule.show();
    } catch (error) {
      screenPickerModulePromise = null;
      console.error('Video call window: Failed to open screen picker:', error);
    }
  })();
};
electron.ipcRenderer.on('video-call-window/open-screen-picker', handleOpenScreenPicker);
window.addEventListener('beforeunload', () => {
  isWindowDestroying = true;
  clearAllTimeouts();
  // Clean up IPC listener to prevent memory leaks on reload
  electron.ipcRenderer.removeListener('video-call-window/open-screen-picker', handleOpenScreenPicker);
});

// Setup reload button handler when DOM is ready
const setupReloadButton = () => {
  const reloadButton = document.getElementById('error-reload-button');
  if (reloadButton) {
    reloadButton.addEventListener('click', handleReload);
  }
};
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', setupReloadButton);
} else {
  setupReloadButton();
}
start().catch(error => {
  console.error('Video call window: Fatal initialization error:', error);
  showFallbackUI();
});

exports.preloadScreenSharePicker = preloadScreenSharePicker;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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
