'use strict';

var electron = require('electron');

function _defineProperty(e, r, t) {
  return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, {
    value: t,
    enumerable: !0,
    configurable: !0,
    writable: !0
  }) : e[r] = t, e;
}
function _toPrimitive(t, r) {
  if ("object" != typeof t || !t) return t;
  var e = t[Symbol.toPrimitive];
  if (void 0 !== e) {
    var i = e.call(t, r || "default");
    if ("object" != typeof i) return i;
    throw new TypeError("@@toPrimitive must return a primitive value.");
  }
  return ("string" === r ? String : Number)(t);
}
function _toPropertyKey(t) {
  var i = _toPrimitive(t, "string");
  return "symbol" == typeof i ? i : i + "";
}

/**
 * Jitsi Meet External API Interface
 */
// eslint-disable-next-line @typescript-eslint/naming-convention

// eslint-disable-next-line @typescript-eslint/naming-convention

/**
 * Options for Jitsi Meet External API
 */
// eslint-disable-next-line @typescript-eslint/naming-convention

/**
 * Interface for the Jitsi Bridge
 */
// eslint-disable-next-line @typescript-eslint/naming-convention

/**
 * Configuration for Jitsi Bridge initialization
 */
// eslint-disable-next-line @typescript-eslint/naming-convention

/**
 * JitsiBridge - Bridge between Electron application and Jitsi Meet's External API
 * Handles initialization, event handling, and screen sharing coordination
 */
class JitsiBridgeImpl {
  constructor() {
    _defineProperty(this, "jitsiApi", null);
    _defineProperty(this, "isApiInitialized", false);
    _defineProperty(this, "domain", '');
    _defineProperty(this, "roomName", '');
    // @ts-expect-error: variable is used in the implementation
    _defineProperty(this, "displayName", '');
    // @ts-expect-error: variable is used in the implementation
    _defineProperty(this, "options", {});
    _defineProperty(this, "detectionInProgress", false);
    console.log('JitsiBridge: Initializing detection mechanisms');
    this.setupDetection();
  }

  /**
   * Initialize detection mechanisms to automatically detect Jitsi meetings
   * and set up appropriate listeners
   */
  setupDetection() {
    // Set up mutation observer to detect when Jitsi iframes are added to the DOM
    const observer = new MutationObserver(mutations => {
      for (const mutation of mutations) {
        if (mutation.type === 'childList') {
          this.detectJitsiMeeting();
        }
      }
    });

    // Start observing once DOM is loaded
    window.addEventListener('DOMContentLoaded', () => {
      observer.observe(document.body, {
        childList: true,
        subtree: true
      });
      this.detectJitsiMeeting();
    });

    // Also check on page load
    window.addEventListener('load', () => {
      this.detectJitsiMeeting();
    });
  }

  /**
   * Detect if current page is a Jitsi meeting
   */
  async detectJitsiMeeting() {
    if (this.detectionInProgress || this.isApiInitialized) return;
    this.detectionInProgress = true;
    try {
      // Check URL for Jitsi patterns
      if (this.isJitsiMeetingUrl(window.location.href)) {
        console.log('JitsiBridge: Detected Jitsi meeting URL:', window.location.href);

        // Parse domain and room name from URL
        const url = new URL(window.location.href);
        this.domain = url.hostname;
        this.roomName = this.extractRoomNameFromUrl(url);
        if (this.domain && this.roomName) {
          await this.initializeJitsiApi({
            domain: this.domain,
            roomName: this.roomName
          });
        }
      }

      // Check for Jitsi iframes
      const jitsiIframes = this.findJitsiIframes();
      if (jitsiIframes.length > 0) {
        const iframe = jitsiIframes[0];
        console.log('JitsiBridge: Detected Jitsi iframe:', iframe);
        try {
          const frameUrl = new URL(iframe.src);
          this.domain = frameUrl.hostname;
          this.roomName = this.extractRoomNameFromUrl(frameUrl);
          if (this.domain && this.roomName) {
            await this.initializeJitsiApi({
              domain: this.domain,
              roomName: this.roomName,
              options: {
                parentNode: iframe.parentElement || undefined
              }
            });
          }
        } catch (e) {
          console.error('JitsiBridge: Error parsing iframe URL:', e);
        }
      }
    } finally {
      this.detectionInProgress = false;
    }
  }

  /**
   * Find any iframes that might be Jitsi meetings
   */
  findJitsiIframes() {
    return Array.from(document.querySelectorAll('iframe')).filter(iframe => {
      if (!iframe.src) return false;
      return this.isJitsiMeetingUrl(iframe.src);
    });
  }

  /**
   * Check if a URL is likely a Jitsi meeting
   */
  isJitsiMeetingUrl(url) {
    try {
      const parsedUrl = new URL(url);

      // Check for known Jitsi hosts
      const knownJitsiHosts = ['meet.jit.si', '8x8.vc', 'jitsi.rocket.chat'];
      const isKnownHost = knownJitsiHosts.some(host => parsedUrl.hostname === host || parsedUrl.hostname.endsWith(`.${host}`));
      if (isKnownHost) return true;

      // Check URL patterns common in Jitsi deployments
      return url.includes('/meet/') || url.includes('/conference/') || url.includes('?jwt=') || !!parsedUrl.pathname.match(/\/[a-zA-Z0-9_-]{6,}$/);
    } catch (e) {
      console.error('JitsiBridge: Error parsing URL:', e);
      return false;
    }
  }

  /**
   * Extract room name from a Jitsi URL
   */
  extractRoomNameFromUrl(url) {
    // Different Jitsi deployments might have different URL patterns
    const pathParts = url.pathname.split('/').filter(Boolean);
    if (pathParts.length > 0) {
      // Most Jitsi deployments have the room name as the last path segment
      return pathParts[pathParts.length - 1];
    }
    return '';
  }

  /**
   * Initialize the Jitsi Meet External API
   */
  async initializeJitsiApi(config) {
    if (this.isApiInitialized) {
      console.log('JitsiBridge: API already initialized');
      return true;
    }
    if (!(config !== null && config !== void 0 && config.domain) || !config.roomName) {
      console.error('JitsiBridge: Invalid configuration');
      return false;
    }

    // Store configuration
    this.domain = config.domain;
    this.roomName = config.roomName;
    this.displayName = config.displayName || '';
    this.options = config.options || {};
    try {
      // Load the external API script if needed
      if (!window.JitsiMeetExternalAPI) {
        await this.loadJitsiScript(this.domain);
      }
      console.log('JitsiBridge: Creating Jitsi Meet External API instance');

      // We don't actually create a new instance with the External API
      // because we don't want to create a new iframe when one might already exist
      // Instead, we just initialize our event listeners for the existing iframe

      // Set up event listeners on the window for iframe communication
      this.setupMessageEventListener();
      this.isApiInitialized = true;
      console.log('JitsiBridge: Jitsi Meet External API initialized successfully');
      return true;
    } catch (error) {
      console.error('JitsiBridge: Error initializing Jitsi Meet External API:', error);
      return false;
    }
  }

  /**
   * Load the Jitsi Meet External API script
   */
  async loadJitsiScript(domain) {
    return new Promise((resolve, reject) => {
      const script = document.createElement('script');
      // Ensure we use https
      const protocol = window.location.protocol === 'https:' ? 'https:' : 'https:';
      script.src = `${protocol}//${domain}/external_api.js`;
      script.async = true;
      script.onload = () => {
        console.log('JitsiBridge: Jitsi Meet External API script loaded');
        resolve();
      };
      script.onerror = error => {
        console.error('JitsiBridge: Error loading Jitsi Meet External API script:', error);
        reject(error);
      };
      document.head.appendChild(script);
    });
  }

  /**
   * Set up window message event listener to communicate with the Jitsi iframe
   */
  setupMessageEventListener() {
    window.addEventListener('message', event => {
      try {
        const {
          data
        } = event;

        // Handle screen sharing requests from Jitsi
        if (data && typeof data === 'object' && data.type === 'request-desktop-picker') {
          console.log('JitsiBridge: Received screen sharing request from Jitsi');
          this.handleScreenSharingRequest();
        }
      } catch (e) {
        console.error('JitsiBridge: Error handling message event:', e);
      }
    }, false);
  }

  /**
   * Handle screen sharing requests from Jitsi
   */
  handleScreenSharingRequest() {
    // Directly invoke the screen picker
    electron.ipcRenderer.invoke('video-call-window/open-screen-picker').then(() => {
      // Listener for the selected source remains the same
      electron.ipcRenderer.once('video-call-window/screen-sharing-source-responded', (_event, sourceId) => {
        if (!sourceId) {
          console.log('JitsiBridge: Screen sharing cancelled');
          this.sendMessageToJitsiIframe({
            type: 'screen-sharing-canceled'
          });
          return;
        }
        console.log('JitsiBridge: Screen sharing source selected:', sourceId);

        // Send the selected source ID to Jitsi
        this.sendMessageToJitsiIframe({
          type: 'selected-screen-share-source',
          sourceId
        });
      });
    });
  }

  /**
   * Start screen sharing
   */
  async startScreenSharing() {
    console.log('JitsiBridge: Start screen sharing requested');
    try {
      // Direct invoke to screen picker
      await electron.ipcRenderer.invoke('video-call-window/open-screen-picker');
      return new Promise(resolve => {
        electron.ipcRenderer.once('video-call-window/screen-sharing-source-responded', (_event, sourceId) => {
          if (!sourceId) {
            console.log('JitsiBridge: Screen sharing cancelled');
            resolve(false);
            return;
          }
          console.log('JitsiBridge: Screen sharing source selected:', sourceId);

          // Send the selected source ID to Jitsi
          this.sendMessageToJitsiIframe({
            type: 'selected-screen-share-source',
            sourceId
          });
          resolve(true);
        });
      });
    } catch (error) {
      console.error('JitsiBridge: Error starting screen sharing:', error);
      return false;
    }
  }

  /**
   * Send a message to the Jitsi iframe
   */
  sendMessageToJitsiIframe(message) {
    const jitsiIframes = document.querySelectorAll('iframe');
    for (const iframe of Array.from(jitsiIframes)) {
      if (iframe.src && this.isJitsiMeetingUrl(iframe.src)) {
        var _iframe$contentWindow;
        (_iframe$contentWindow = iframe.contentWindow) === null || _iframe$contentWindow === void 0 || _iframe$contentWindow.postMessage(message, '*');
        console.log('JitsiBridge: Sent message to Jitsi iframe:', message);
        return;
      }
    }
    console.warn('JitsiBridge: No Jitsi iframe found to send message');
  }

  /**
   * End the current call
   */
  endCall() {
    console.log('JitsiBridge: End call requested');
    if (this.jitsiApi) {
      this.jitsiApi.executeCommand('hangup');
    } else {
      this.sendMessageToJitsiIframe({
        type: 'hangup'
      });
    }
    this.dispose();
  }

  /**
   * Get the Jitsi Meet version (may not be supported by all deployments)
   */
  async getJitsiVersion() {
    if (!this.isApiInitialized) {
      return 'API not initialized';
    }

    // This functionality might not be available in all Jitsi deployments
    return 'Version not available';
  }

  /**
   * Dispose of the Jitsi Meet External API instance
   */
  dispose() {
    console.log('JitsiBridge: Disposing');
    if (this.jitsiApi) {
      try {
        this.jitsiApi.dispose();
      } catch (e) {
        console.error('JitsiBridge: Error disposing Jitsi API:', e);
      }
      this.jitsiApi = null;
    }
    this.isApiInitialized = false;
  }

  /**
   * Check if the API is initialized
   */
  isInitialized() {
    return this.isApiInitialized;
  }

  /**
   * Get the Jitsi API instance
   */
  getApi() {
    return this.jitsiApi;
  }

  /**
   * Get the current domain
   */
  getCurrentDomain() {
    return this.domain;
  }

  /**
   * Get the current room name
   */
  getCurrentRoomName() {
    return this.roomName;
  }
}

// Create and expose the Jitsi Bridge
const jitsiBridge = new JitsiBridgeImpl();
window.jitsiBridge = jitsiBridge;

// Expose any necessary APIs to the webview content
electron.contextBridge.exposeInMainWorld('videoCallWindow', {
  // Add methods here if needed for communication with the main process
  requestScreenSharing: async () => {
    // Directly invoke the screen picker
    await electron.ipcRenderer.invoke('video-call-window/open-screen-picker');
    return new Promise(resolve => {
      electron.ipcRenderer.once('video-call-window/screen-sharing-source-responded', (_event, id) => {
        resolve(id);
      });
    });
  }
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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
