'use strict';

var electron = require('electron');
var redux = require('redux');
var detectBrowsers = require('detect-browsers');
var fs = require('fs');
var path = require('path');

function _interopDefault (e) { return e && e.__esModule ? e : { default: e }; }

var fs__default = /*#__PURE__*/_interopDefault(fs);
var path__default = /*#__PURE__*/_interopDefault(path);

const handle = (channel, handler) => {
  const listener = async (_, id, ...args) => {
    try {
      const resolved = await handler(...args);
      electron.ipcRenderer.send(`${channel}@${id}`, {
        resolved
      });
    } catch (error) {
      error instanceof Error && electron.ipcRenderer.send(`${channel}@${id}`, {
        rejected: {
          name: error.name,
          message: error.message,
          stack: error.stack
        }
      });
    }
  };
  electron.ipcRenderer.on(channel, listener);
  return () => {
    electron.ipcRenderer.removeListener(channel, listener);
  };
};
const invoke = (channel, ...args) => electron.ipcRenderer.invoke(channel, ...args);

const jitsiDomain = window.location.origin;
const desktopCapturer = {
  getSources: opts => electron.ipcRenderer.invoke('jitsi-desktop-capturer-get-sources', [opts, jitsiDomain])
};
const JitsiMeetElectron = {
  async obtainDesktopStreams(callback, errorCallback, options) {
    try {
      const sources = (await desktopCapturer.getSources(options)).map(source => ({
        id: source.id,
        name: source.name,
        display_id: source.display_id,
        thumbnail: {
          toDataURL: () => source.thumbnail.toDataURL()
        },
        appIcon: {
          toDataURL: () => source.appIcon.toDataURL()
        }
      }));
      callback(sources);
    } catch (error) {
      error instanceof Error && errorCallback(error);
      console.log(error);
    }
  }
};

let getAbsoluteUrl;
let serverUrl;
const setServerUrl = _serverUrl => {
  serverUrl = _serverUrl;
};
const getServerUrl = () => serverUrl;
const setUrlResolver = _getAbsoluteUrl => {
  getAbsoluteUrl = _getAbsoluteUrl;
};

const isFSA = action => typeof action === 'object' && action !== null && !Array.isArray(action) && 'type' in action && typeof action.type === 'string';
const hasMeta = action => 'meta' in action && typeof action.meta === 'object' && action.meta !== null;
const isResponse = action => hasMeta(action) && action.meta.response === true;
const isLocallyScoped = action => hasMeta(action) && action.meta.scope === 'local';
const isErrored = action => 'meta' in action && action.error === true && action.payload instanceof Error;
const hasPayload = action => 'payload' in action;
const isResponseTo = (id, ...types) => action => isResponse(action) && types.includes(action.type) && action.meta.id === id;

const getInitialState = () => invoke('redux/get-initial-state');
const forwardToMain = api => {
  handle('redux/action-dispatched', async action => {
    api.dispatch(action);
  });
  return next => action => {
    if (!isFSA(action) || isLocallyScoped(action)) {
      return next(action);
    }
    invoke('redux/action-dispatched', action);
    return action;
  };
};

const APP_PATH_SET = 'app/path-set';
const APP_VERSION_SET = 'app/version-set';
const APP_SETTINGS_LOADED = 'app/settings-loaded';
const APP_ALLOWED_NTLM_CREDENTIALS_DOMAINS_SET = 'app/allowed-ntlm-credentials-domains-set';
const APP_MAIN_WINDOW_TITLE_SET = 'app/main-window-title-set';
const APP_MACHINE_THEME_SET = 'app/machine-theme-set';
const APP_SCREEN_CAPTURE_FALLBACK_FORCED_SET = 'app/screen-capture-fallback-forced-set';

const allowedNTLMCredentialsDomains = (state = null, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      {
        const {
          allowedNTLMCredentialsDomains = state
        } = action.payload;
        return allowedNTLMCredentialsDomains;
      }
    case APP_ALLOWED_NTLM_CREDENTIALS_DOMAINS_SET:
      {
        if (action.payload === null) return null;
        return action.payload;
      }
    default:
      return state;
  }
};

const appPath = (state = null, action) => {
  switch (action.type) {
    case APP_PATH_SET:
      return action.payload;
    default:
      return state;
  }
};

const appVersion = (state = null, action) => {
  switch (action.type) {
    case APP_VERSION_SET:
      return action.payload;
    default:
      return state;
  }
};

const machineTheme = (state = 'light', action) => {
  switch (action.type) {
    case APP_MACHINE_THEME_SET:
      {
        return action.payload;
      }
    default:
      return state;
  }
};

const mainWindowTitle = (state = null, action) => {
  switch (action.type) {
    case APP_MAIN_WINDOW_TITLE_SET:
      return action.payload;
    default:
      return state;
  }
};

const screenCaptureFallbackForced = (state = false, action) => {
  switch (action.type) {
    case APP_SCREEN_CAPTURE_FALLBACK_FORCED_SET:
      return action.payload;
    default:
      return state;
  }
};

const DOWNLOAD_CREATED = 'downloads/created';
const DOWNLOAD_REMOVED = 'downloads/removed';
const DOWNLOADS_CLEARED = 'downloads/cleared';
const DOWNLOAD_UPDATED = 'downloads/updated';

const DownloadStatus = {
  ALL: 'All',
  PAUSED: 'Paused',
  CANCELLED: 'Cancelled'
};

const downloads = (state = {}, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      {
        var _action$payload$downl;
        const initDownloads = (_action$payload$downl = action.payload.downloads) !== null && _action$payload$downl !== void 0 ? _action$payload$downl : {};
        Object.values(initDownloads).forEach(value => {
          if (value.state === 'progressing' || value.state === 'paused') {
            value.state = 'cancelled';
            value.status = DownloadStatus.CANCELLED;
          }
        });
        return initDownloads !== null && initDownloads !== void 0 ? initDownloads : {};
      }
    case DOWNLOAD_CREATED:
      {
        const download = action.payload;
        return {
          ...state,
          [download.itemId]: download
        };
      }
    case DOWNLOAD_UPDATED:
      {
        const existingDownload = state[action.payload.itemId];
        if (!existingDownload) {
          return state; // Don't update if download doesn't exist
        }
        const newState = {
          ...state
        };
        newState[action.payload.itemId] = {
          ...existingDownload,
          ...action.payload
        };
        return newState;
      }
    case DOWNLOAD_REMOVED:
      {
        const newState = {
          ...state
        };
        delete newState[action.payload];
        return newState;
      }
    case DOWNLOADS_CLEARED:
      return {};
    default:
      return state;
  }
};

const JITSI_SERVER_CAPTURE_SCREEN_PERMISSION_UPDATED = 'jitsi-server-capture-screen-permission-updated';
const JITSI_SERVER_CAPTURE_SCREEN_PERMISSIONS_CLEARED = 'jitsi-server-capture-screen-permissions-cleared';

const allowedJitsiServers = (state = {}, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      {
        const {
          allowedJitsiServers = {}
        } = action.payload;
        state = allowedJitsiServers;
        return state;
      }
    case JITSI_SERVER_CAPTURE_SCREEN_PERMISSION_UPDATED:
      {
        state = {
          ...state,
          [action.payload.jitsiServer]: action.payload.allowed
        };
        return state;
      }
    case JITSI_SERVER_CAPTURE_SCREEN_PERMISSIONS_CLEARED:
      {
        state = {};
        return state;
      }
    default:
      return state;
  }
};

const CERTIFICATES_CLEARED = 'certificates/cleared';
const CERTIFICATES_LOADED = 'certificates/loaded';
const CERTIFICATES_CLIENT_CERTIFICATE_REQUESTED = 'certificates/client-certificate-requested';
const TRUSTED_CERTIFICATES_UPDATED = 'trusted-certificates/updated';
const NOT_TRUSTED_CERTIFICATES_UPDATED = 'not-trusted-certificates/updated';
const SELECT_CLIENT_CERTIFICATE_DIALOG_CERTIFICATE_SELECTED = 'select-client-certificate-dialog/certificate-selected';
const SELECT_CLIENT_CERTIFICATE_DIALOG_DISMISSED = 'select-client-certificate-dialog/dismissed';
const EXTERNAL_PROTOCOL_PERMISSION_UPDATED = 'navigation/external-protocol-permission-updated';

const clientCertificates = (state = [], action) => {
  switch (action.type) {
    case CERTIFICATES_CLIENT_CERTIFICATE_REQUESTED:
      return action.payload;
    case SELECT_CLIENT_CERTIFICATE_DIALOG_CERTIFICATE_SELECTED:
    case SELECT_CLIENT_CERTIFICATE_DIALOG_DISMISSED:
      return [];
    default:
      return state;
  }
};
const trustedCertificates = (state = {}, action) => {
  switch (action.type) {
    case CERTIFICATES_LOADED:
    case TRUSTED_CERTIFICATES_UPDATED:
      return action.payload;
    case CERTIFICATES_CLEARED:
      return {};
    case APP_SETTINGS_LOADED:
      {
        const {
          trustedCertificates = state
        } = action.payload;
        return trustedCertificates;
      }
    default:
      return state;
  }
};
const notTrustedCertificates = (state = {}, action) => {
  switch (action.type) {
    case NOT_TRUSTED_CERTIFICATES_UPDATED:
      return action.payload;
    case CERTIFICATES_CLEARED:
      return {};
    case APP_SETTINGS_LOADED:
      {
        const {
          notTrustedCertificates = state
        } = action.payload;
        return notTrustedCertificates;
      }
    default:
      return state;
  }
};
const externalProtocols = (state = {}, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      {
        const {
          externalProtocols = {}
        } = action.payload;
        state = externalProtocols;
        return state;
      }
    case EXTERNAL_PROTOCOL_PERMISSION_UPDATED:
      {
        state = {
          ...state,
          [action.payload.protocol]: action.payload.allowed
        };
        return state;
      }
    default:
      return state;
  }
};

const DEEP_LINKS_SERVER_ADDED = 'deep-links/server-added';
const DEEP_LINKS_SERVER_FOCUSED = 'deep-links/server-focused';

const OUTLOOK_CALENDAR_SET_CREDENTIALS = 'outlook-calendar/set-credentials';
const OUTLOOK_CALENDAR_ASK_CREDENTIALS = 'outlook-calendar/ask-credentials';
const OUTLOOK_CALENDAR_DIALOG_DISMISSED = 'outlook-calendar/dialog-dismissed';
const OUTLOOK_CALENDAR_SAVE_CREDENTIALS = 'outlook-calendar/save-credentials';

const ABOUT_DIALOG_DISMISSED = 'about-dialog/dismissed';
const ABOUT_DIALOG_TOGGLE_UPDATE_ON_START = 'about-dialog/toggle-update-on-start';
const ABOUT_DIALOG_UPDATE_CHANNEL_CHANGED = 'about-dialog/update-channel-changed';
const ADD_SERVER_VIEW_SERVER_ADDED = 'add-server/view-server-added';
const MENU_BAR_ABOUT_CLICKED = 'menu-bar/about-clicked';
const MENU_BAR_ADD_NEW_SERVER_CLICKED = 'menu-bar/add-new-server-clicked';
const MENU_BAR_SELECT_SERVER_CLICKED = 'menu-bar/select-server-clicked';
const MENU_BAR_TOGGLE_IS_MENU_BAR_ENABLED_CLICKED = 'menu-bar/toggle-is-menu-bar-enabled-clicked';
const MENU_BAR_TOGGLE_IS_SHOW_WINDOW_ON_UNREAD_CHANGED_ENABLED_CLICKED = 'menu-bar/toggle-is-show-window-on-unread-changed-enabled-clicked';
const MENU_BAR_TOGGLE_IS_SIDE_BAR_ENABLED_CLICKED = 'menu-bar/toggle-is-side-bar-enabled-clicked';
const MENU_BAR_TOGGLE_IS_TRAY_ICON_ENABLED_CLICKED = 'menu-bar/toggle-is-tray-icon-enabled-clicked';
const MENU_BAR_TOGGLE_IS_DEVELOPER_MODE_ENABLED_CLICKED = 'menu-bar/toggle-is-developer-mode-enabled-clicked';
const MENU_BAR_TOGGLE_IS_VIDEO_CALL_DEVTOOLS_AUTO_OPEN_ENABLED_CLICKED = 'menu-bar/toggle-is-video-call-devtools-auto-open-enabled-clicked';
const ROOT_WINDOW_ICON_CHANGED = 'root-window/icon-changed';
const ROOT_WINDOW_STATE_CHANGED = 'root-window/state-changed';
const VIDEO_CALL_WINDOW_STATE_CHANGED = 'video-call-window/state-changed';
const SIDE_BAR_ADD_NEW_SERVER_CLICKED = 'side-bar/add-new-server-clicked';
const SIDE_BAR_DOWNLOADS_BUTTON_CLICKED = 'side-bar/downloads-button-clicked';
const SIDE_BAR_SETTINGS_BUTTON_CLICKED = 'side-bar/settings-button-clicked';
const SIDE_BAR_REMOVE_SERVER_CLICKED = 'side-bar/remove-server-clicked';
const SIDE_BAR_SERVER_SELECTED = 'side-bar/server-selected';
const SIDE_BAR_SERVERS_SORTED = 'side-bar/servers-sorted';
const TOUCH_BAR_FORMAT_BUTTON_TOUCHED = 'touch-bar/format-button-touched';
const TOUCH_BAR_SELECT_SERVER_TOUCHED = 'touch-bar/select-server-touched';
const UPDATE_DIALOG_DISMISSED = 'update-dialog/dismissed';
const UPDATE_DIALOG_INSTALL_BUTTON_CLICKED = 'update-dialog/install-button-clicked';
const UPDATE_DIALOG_REMIND_UPDATE_LATER_CLICKED = 'update-dialog/remind-update-later-clicked';
const UPDATE_DIALOG_SKIP_UPDATE_CLICKED = 'update-dialog/skip-update-clicked';
const WEBVIEW_READY = 'webview/ready';
const WEBVIEW_ATTACHED = 'webview/attached';
const WEBVIEW_DID_FAIL_LOAD = 'webview/did-fail-load';
const WEBVIEW_DID_NAVIGATE = 'webview/did-navigate';
const WEBVIEW_DID_START_LOADING = 'webview/did-start-loading';
const WEBVIEW_FAVICON_CHANGED = 'webview/favicon-changed';
const WEBVIEW_FOCUS_REQUESTED = 'webview/focus-requested';
const WEBVIEW_MESSAGE_BOX_BLURRED = 'webview/message-box-blurred';
const WEBVIEW_MESSAGE_BOX_FOCUSED = 'webview/message-box-focused';
const WEBVIEW_SCREEN_SHARING_SOURCE_REQUESTED = 'webview/screen-sharing-source-requested';
const WEBVIEW_SCREEN_SHARING_SOURCE_RESPONDED = 'webview/screen-sharing-source-responded';
const WEBVIEW_SIDEBAR_STYLE_CHANGED = 'webview/sidebar-style-changed';
const WEBVIEW_SIDEBAR_CUSTOM_THEME_CHANGED = 'webview/sidebar-custom-theme-changed';
const WEBVIEW_GIT_COMMIT_HASH_CHANGED = 'webview/git-commit-hash-changed';
const WEBVIEW_GIT_COMMIT_HASH_CHECK = 'webview/git-commit-hash-check';
const WEBVIEW_TITLE_CHANGED = 'webview/title-changed';
const WEBVIEW_PAGE_TITLE_CHANGED = 'webview/page-title-changed';
const WEBVIEW_UNREAD_CHANGED = 'webview/unread-changed';
const WEBVIEW_USER_LOGGED_IN = 'webview/user-loggedin';
const WEBVIEW_USER_THEME_APPEARANCE_CHANGED = 'webview/user-theme-appearance-changed';
const WEBVIEW_ALLOWED_REDIRECTS_CHANGED = 'webview/allowed-redirects-changed';
const SETTINGS_SET_REPORT_OPT_IN_CHANGED = 'settings/set-bugsnag-opt-in-changed';
const SETTINGS_SET_FLASHFRAME_OPT_IN_CHANGED = 'settings/set-flashframe-opt-in-changed';
const SETTINGS_SET_HARDWARE_ACCELERATION_OPT_IN_CHANGED = 'settings/set-hardware-acceleration-opt-in-changed';
const SETTINGS_SET_INTERNALVIDEOCHATWINDOW_OPT_IN_CHANGED = 'settings/set-internalvideochatwindow-opt-in-changed';
const SETTINGS_SET_MINIMIZE_ON_CLOSE_OPT_IN_CHANGED = 'settings/set-minimize-on-close-opt-in-changed';
const SETTINGS_SET_IS_TRAY_ICON_ENABLED_CHANGED = 'settings/set-is-tray-icon-enabled-changed';
const SETTINGS_SET_IS_SIDE_BAR_ENABLED_CHANGED = 'settings/set-is-side-bar-enabled-changed';
const SETTINGS_SET_IS_MENU_BAR_ENABLED_CHANGED = 'settings/set-is-menu-bar-enabled-changed';
const SETTINGS_SET_IS_VIDEO_CALL_WINDOW_PERSISTENCE_ENABLED_CHANGED = 'settings/set-is-video-call-window-persistence-enabled-changed';
const SETTINGS_SET_IS_TRANSPARENT_WINDOW_ENABLED_CHANGED = 'settings/set-is-transparent-window-enabled-changed';
const SETTINGS_SET_IS_DEVELOPER_MODE_ENABLED_CHANGED = 'settings/set-is-developer-mode-enabled-changed';
const SETTINGS_SET_IS_VIDEO_CALL_DEVTOOLS_AUTO_OPEN_ENABLED_CHANGED = 'settings/set-is-video-call-devtools-auto-open-enabled-changed';
const SETTINGS_SET_IS_VIDEO_CALL_SCREEN_CAPTURE_FALLBACK_ENABLED_CHANGED = 'settings/set-is-video-call-screen-capture-fallback-enabled-changed';
const SETTINGS_NTLM_CREDENTIALS_CHANGED = 'settings/ntlm-credentials-changed';
const SETTINGS_AVAILABLE_BROWSERS_UPDATED = 'settings/available-browsers-updated';
const SETTINGS_SELECTED_BROWSER_CHANGED = 'settings/selected-browser-changed';
const SET_HAS_TRAY_MINIMIZE_NOTIFICATION_SHOWN = 'notifications/set-has-tray-minimize-notification-shown';
const DOWNLOADS_BACK_BUTTON_CLICKED = 'downloads/back-button-clicked';
const WEBVIEW_SERVER_SUPPORTED_VERSIONS_UPDATED = 'webview/server-supported-versions-updated';
const WEBVIEW_SERVER_UNIQUE_ID_UPDATED = 'webview/server-workspace-uid-updated';
const WEBVIEW_SERVER_IS_SUPPORTED_VERSION = 'webview/server-is-supported-version';
const WEBVIEW_SERVER_SUPPORTED_VERSIONS_LOADING = 'webview/server-supported-versions-loading';
const WEBVIEW_SERVER_SUPPORTED_VERSIONS_ERROR = 'webview/server-supported-versions-error';
const WEBVIEW_SERVER_VERSION_UPDATED = 'webview/version-updated';
const SUPPORTED_VERSION_DIALOG_DISMISS = 'supported-versions-dialog/dismiss';
const WEBVIEW_FORCE_RELOAD_WITH_CACHE_CLEAR = 'webview/force-reload-with-cache-clear';
const OPEN_SERVER_INFO_MODAL = 'server-info-modal/open';
const CLOSE_SERVER_INFO_MODAL = 'server-info-modal/close';

const SERVERS_LOADED = 'servers/loaded';
const SERVER_DOCUMENT_VIEWER_OPEN_URL = 'server/document-viewer/open-url';

/* eslint-disable complexity */

const ensureUrlFormat = serverUrl => {
  if (serverUrl) {
    return new URL(serverUrl).href;
  }
  throw new Error('cannot handle null server URLs');
};
const upsert = (state, server) => {
  const index = state.findIndex(({
    url
  }) => url === server.url);
  if (index === -1) {
    return [...state, server];
  }
  return state.map((_server, i) => i === index ? {
    ..._server,
    ...server
  } : _server);
};
const update = (state, server) => {
  const index = state.findIndex(({
    url
  }) => url === server.url);
  if (index === -1) {
    return state;
  }
  return state.map((_server, i) => i === index ? {
    ..._server,
    ...server
  } : _server);
};
const servers = (state = [], action) => {
  switch (action.type) {
    case ADD_SERVER_VIEW_SERVER_ADDED:
    case DEEP_LINKS_SERVER_ADDED:
      {
        const url = action.payload;
        return upsert(state, {
          url,
          title: url
        });
      }
    case SIDE_BAR_REMOVE_SERVER_CLICKED:
      {
        const _url = action.payload;
        return state.filter(({
          url
        }) => url !== _url);
      }
    case SIDE_BAR_SERVERS_SORTED:
      {
        const urls = action.payload;
        return state.sort(({
          url: a
        }, {
          url: b
        }) => urls.indexOf(a) - urls.indexOf(b));
      }
    case WEBVIEW_TITLE_CHANGED:
      {
        const {
          url,
          title = url
        } = action.payload;
        return upsert(state, {
          url,
          title
        });
      }
    case WEBVIEW_PAGE_TITLE_CHANGED:
      {
        const {
          url,
          pageTitle
        } = action.payload;
        return upsert(state, {
          url,
          pageTitle
        });
      }
    case WEBVIEW_SERVER_SUPPORTED_VERSIONS_UPDATED:
      {
        const {
          url,
          supportedVersions,
          source
        } = action.payload;
        return upsert(state, {
          url,
          supportedVersions,
          supportedVersionsSource: source,
          supportedVersionsFetchState: 'success'
        });
      }
    case WEBVIEW_SERVER_SUPPORTED_VERSIONS_LOADING:
      {
        const {
          url
        } = action.payload;
        return upsert(state, {
          url,
          supportedVersionsFetchState: 'loading'
        });
      }
    case WEBVIEW_SERVER_SUPPORTED_VERSIONS_ERROR:
      {
        const {
          url
        } = action.payload;
        return upsert(state, {
          url,
          supportedVersionsFetchState: 'error'
        });
      }
    case SUPPORTED_VERSION_DIALOG_DISMISS:
      {
        const {
          url
        } = action.payload;
        return upsert(state, {
          url,
          expirationMessageLastTimeShown: new Date()
        });
      }
    case WEBVIEW_SERVER_UNIQUE_ID_UPDATED:
      {
        const {
          url,
          uniqueID
        } = action.payload;
        return upsert(state, {
          url,
          uniqueID
        });
      }
    case WEBVIEW_USER_THEME_APPEARANCE_CHANGED:
      {
        const {
          url,
          themeAppearance
        } = action.payload;
        return upsert(state, {
          url,
          themeAppearance
        });
      }
    case WEBVIEW_SERVER_IS_SUPPORTED_VERSION:
      {
        const {
          url,
          isSupportedVersion
        } = action.payload;
        return upsert(state, {
          url,
          isSupportedVersion,
          supportedVersionsValidatedAt: new Date()
        });
      }
    case WEBVIEW_SERVER_VERSION_UPDATED:
      {
        const {
          url,
          version
        } = action.payload;
        return upsert(state, {
          url,
          version
        });
      }
    case WEBVIEW_UNREAD_CHANGED:
      {
        const {
          url,
          badge
        } = action.payload;
        return upsert(state, {
          url,
          badge
        });
      }
    case WEBVIEW_USER_LOGGED_IN:
      {
        const {
          url,
          userLoggedIn
        } = action.payload;
        return upsert(state, {
          url,
          userLoggedIn
        });
      }
    case WEBVIEW_ALLOWED_REDIRECTS_CHANGED:
      {
        const {
          url,
          allowedRedirects
        } = action.payload;
        return upsert(state, {
          url,
          allowedRedirects
        });
      }
    case WEBVIEW_SIDEBAR_STYLE_CHANGED:
      {
        const {
          url,
          style
        } = action.payload;
        return upsert(state, {
          url,
          style
        });
      }
    case WEBVIEW_SIDEBAR_CUSTOM_THEME_CHANGED:
      {
        const {
          url,
          customTheme
        } = action.payload;
        return upsert(state, {
          url,
          customTheme
        });
      }
    case WEBVIEW_GIT_COMMIT_HASH_CHANGED:
      {
        const {
          url,
          gitCommitHash
        } = action.payload;
        return upsert(state, {
          url,
          gitCommitHash
        });
      }
    case WEBVIEW_FAVICON_CHANGED:
      {
        const {
          url,
          favicon
        } = action.payload;
        return upsert(state, {
          url,
          favicon
        });
      }
    case WEBVIEW_DID_NAVIGATE:
      {
        const {
          url,
          pageUrl
        } = action.payload;
        if (pageUrl !== null && pageUrl !== void 0 && pageUrl.includes(url)) {
          return upsert(state, {
            url,
            lastPath: pageUrl
          });
        }
        return state;
      }
    case WEBVIEW_DID_START_LOADING:
      {
        const {
          url
        } = action.payload;
        return upsert(state, {
          url,
          failed: false
        });
      }
    case WEBVIEW_DID_FAIL_LOAD:
      {
        const {
          url,
          isMainFrame
        } = action.payload;
        if (isMainFrame) {
          return upsert(state, {
            url,
            failed: true
          });
        }
        return state;
      }
    case SERVERS_LOADED:
      {
        const {
          servers = state
        } = action.payload;
        return servers.map(server => ({
          ...server,
          url: ensureUrlFormat(server.url)
        }));
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          servers = state
        } = action.payload;
        return servers.map(server => ({
          ...server,
          url: ensureUrlFormat(server.url),
          documentViewerOpenUrl: ''
        }));
      }
    case WEBVIEW_READY:
      {
        const {
          url,
          webContentsId
        } = action.payload;
        return update(state, {
          url,
          webContentsId
        });
      }
    case WEBVIEW_ATTACHED:
      {
        const {
          url,
          webContentsId
        } = action.payload;
        return update(state, {
          url,
          webContentsId
        });
      }
    case OUTLOOK_CALENDAR_SAVE_CREDENTIALS:
      {
        const {
          url,
          outlookCredentials
        } = action.payload;
        return upsert(state, {
          url,
          outlookCredentials
        });
      }
    case SERVER_DOCUMENT_VIEWER_OPEN_URL:
      {
        const {
          server,
          documentUrl
        } = action.payload;
        return upsert(state, {
          url: server,
          documentViewerOpenUrl: documentUrl
        });
      }
    default:
      return state;
  }
};

const availableBrowsers = (state = [], action) => {
  switch (action.type) {
    case SETTINGS_AVAILABLE_BROWSERS_UPDATED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        return state;
      }
    default:
      return state;
  }
};

const currentView = (state = 'add-new-server', action) => {
  switch (action.type) {
    case ADD_SERVER_VIEW_SERVER_ADDED:
    case DEEP_LINKS_SERVER_ADDED:
    case DEEP_LINKS_SERVER_FOCUSED:
    case MENU_BAR_SELECT_SERVER_CLICKED:
    case TOUCH_BAR_SELECT_SERVER_TOUCHED:
    case SIDE_BAR_SERVER_SELECTED:
      {
        const url = action.payload;
        return {
          url
        };
      }
    case WEBVIEW_FOCUS_REQUESTED:
      {
        const {
          url,
          view
        } = action.payload;
        if (view === 'downloads') return 'downloads';
        return {
          url
        };
      }
    case SERVERS_LOADED:
      {
        const {
          selected
        } = action.payload;
        return selected ? {
          url: selected
        } : 'add-new-server';
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          currentView = state
        } = action.payload;
        return currentView;
      }
    case MENU_BAR_ADD_NEW_SERVER_CLICKED:
    case SIDE_BAR_ADD_NEW_SERVER_CLICKED:
      return 'add-new-server';
    case SIDE_BAR_REMOVE_SERVER_CLICKED:
      {
        if (typeof state === 'object' && state.url === action.payload) {
          return 'add-new-server';
        }
        return state;
      }
    case SIDE_BAR_DOWNLOADS_BUTTON_CLICKED:
      return 'downloads';
    case SIDE_BAR_SETTINGS_BUTTON_CLICKED:
      return 'settings';
    case DOWNLOADS_BACK_BUTTON_CLICKED:
      return {
        url: action.payload
      };
    default:
      return state;
  }
};

const initialServerInfoModalState = {
  isOpen: false,
  serverData: null
};
const serverInfoModal = (state = initialServerInfoModalState, action) => {
  switch (action.type) {
    case OPEN_SERVER_INFO_MODAL:
      return {
        isOpen: true,
        serverData: action.payload
      };
    case CLOSE_SERVER_INFO_MODAL:
      return initialServerInfoModalState;
    default:
      return state;
  }
};
const dialogs = (state = {
  serverInfoModal: initialServerInfoModalState
}, action) => {
  switch (action.type) {
    case OPEN_SERVER_INFO_MODAL:
    case CLOSE_SERVER_INFO_MODAL:
      return {
        ...state,
        serverInfoModal: serverInfoModal(state.serverInfoModal, action)
      };
    default:
      return state;
  }
};

const hasHideOnTrayNotificationShown = (state = false, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      return Boolean(action.payload.hasHideOnTrayNotificationShown);
    case SET_HAS_TRAY_MINIMIZE_NOTIFICATION_SHOWN:
      return action.payload;
    default:
      return state;
  }
};

const isAddNewServersEnabled = (state = true, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      return Boolean(action.payload.isAddNewServersEnabled);
    default:
      return state;
  }
};

const isDeveloperModeEnabled = (state = false, action) => {
  switch (action.type) {
    case SETTINGS_SET_IS_DEVELOPER_MODE_ENABLED_CHANGED:
    case MENU_BAR_TOGGLE_IS_DEVELOPER_MODE_ENABLED_CLICKED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          isDeveloperModeEnabled = state
        } = action.payload;
        return isDeveloperModeEnabled;
      }
    default:
      return state;
  }
};

const UPDATE_SKIPPED = 'update/skipped';
const UPDATES_CHECKING_FOR_UPDATE = 'updates/checking-for-update';
const UPDATES_ERROR_THROWN = 'updates/error-thrown';
const UPDATES_NEW_VERSION_AVAILABLE = 'updates/new-version-available';
const UPDATES_NEW_VERSION_NOT_AVAILABLE = 'updates/new-version-not-available';
const UPDATES_READY = 'updates/ready';
const UPDATES_CHANNEL_CHANGED = 'updates/channel-changed';

const isFlashFrameEnabled = (state = false, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      return Boolean(action.payload.isFlashFrameEnabled);
    case UPDATES_READY:
      return action.payload.isFlashFrameEnabled;
    case SETTINGS_SET_FLASHFRAME_OPT_IN_CHANGED:
      {
        return action.payload;
      }
    default:
      return state;
  }
};

const isHardwareAccelerationEnabled = (state = false, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      return Boolean(action.payload.isHardwareAccelerationEnabled);
    case UPDATES_READY:
      return action.payload.isHardwareAccelerationEnabled;
    case SETTINGS_SET_HARDWARE_ACCELERATION_OPT_IN_CHANGED:
      {
        return action.payload;
      }
    default:
      return state;
  }
};

const isInternalVideoChatWindowEnabled = (state = false, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      return Boolean(action.payload.isInternalVideoChatWindowEnabled);
    case UPDATES_READY:
      return action.payload.isInternalVideoChatWindowEnabled;
    case SETTINGS_SET_INTERNALVIDEOCHATWINDOW_OPT_IN_CHANGED:
      {
        return action.payload;
      }
    default:
      return state;
  }
};

const isMenuBarEnabled = (state = true, action) => {
  switch (action.type) {
    case SETTINGS_SET_IS_MENU_BAR_ENABLED_CHANGED:
    case MENU_BAR_TOGGLE_IS_MENU_BAR_ENABLED_CLICKED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          isMenuBarEnabled = state
        } = action.payload;
        return isMenuBarEnabled;
      }
    default:
      return state;
  }
};

const isMessageBoxFocused = (state = false, action) => {
  switch (action.type) {
    case WEBVIEW_MESSAGE_BOX_FOCUSED:
      return true;
    case WEBVIEW_DID_START_LOADING:
    case WEBVIEW_MESSAGE_BOX_BLURRED:
    case WEBVIEW_DID_FAIL_LOAD:
      return false;
    default:
      return state;
  }
};

const isMinimizeOnCloseEnabled = (state = process.platform === 'win32', action) => {
  switch (action.type) {
    case SETTINGS_SET_MINIMIZE_ON_CLOSE_OPT_IN_CHANGED:
      {
        return action.payload;
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          isMinimizeOnCloseEnabled = state
        } = action.payload;
        return isMinimizeOnCloseEnabled;
      }
    default:
      return state;
  }
};

const isNTLMCredentialsEnabled = (state = false, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      {
        const {
          isNTLMCredentialsEnabled = state
        } = action.payload;
        return isNTLMCredentialsEnabled;
      }
    case SETTINGS_NTLM_CREDENTIALS_CHANGED:
      {
        return action.payload;
      }
    default:
      return state;
  }
};

const isReportEnabled = (state = false, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      return Boolean(action.payload.isReportEnabled);
    case UPDATES_READY:
      return action.payload.isReportEnabled;
    case SETTINGS_SET_REPORT_OPT_IN_CHANGED:
      {
        return action.payload;
      }
    default:
      return state;
  }
};

const isShowWindowOnUnreadChangedEnabled = (state = false, action) => {
  switch (action.type) {
    case MENU_BAR_TOGGLE_IS_SHOW_WINDOW_ON_UNREAD_CHANGED_ENABLED_CLICKED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          isShowWindowOnUnreadChangedEnabled = state
        } = action.payload;
        return isShowWindowOnUnreadChangedEnabled;
      }
    default:
      return state;
  }
};

const isSideBarEnabled = (state = true, action) => {
  switch (action.type) {
    case SETTINGS_SET_IS_SIDE_BAR_ENABLED_CHANGED:
    case MENU_BAR_TOGGLE_IS_SIDE_BAR_ENABLED_CLICKED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          isSideBarEnabled = state
        } = action.payload;
        return isSideBarEnabled;
      }
    default:
      return state;
  }
};

const isTransparentWindowEnabled = (state = false, action) => {
  switch (action.type) {
    case SETTINGS_SET_IS_TRANSPARENT_WINDOW_ENABLED_CHANGED:
      {
        const {
          payload
        } = action;
        if (typeof payload === 'boolean') {
          return payload;
        }
        console.warn(`Invalid payload type for ${SETTINGS_SET_IS_TRANSPARENT_WINDOW_ENABLED_CHANGED}: expected boolean, got ${typeof payload}`);
        return state;
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          isTransparentWindowEnabled = state
        } = action.payload;
        return isTransparentWindowEnabled;
      }
    default:
      return state;
  }
};

const isTrayIconEnabled = (state = process.platform !== 'linux', action) => {
  switch (action.type) {
    case SETTINGS_SET_IS_TRAY_ICON_ENABLED_CHANGED:
    case MENU_BAR_TOGGLE_IS_TRAY_ICON_ENABLED_CLICKED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          isTrayIconEnabled = state
        } = action.payload;
        return isTrayIconEnabled;
      }
    default:
      return state;
  }
};

const isVideoCallDevtoolsAutoOpenEnabled = (state = false, action) => {
  switch (action.type) {
    case SETTINGS_SET_IS_VIDEO_CALL_DEVTOOLS_AUTO_OPEN_ENABLED_CHANGED:
    case MENU_BAR_TOGGLE_IS_VIDEO_CALL_DEVTOOLS_AUTO_OPEN_ENABLED_CLICKED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          isVideoCallDevtoolsAutoOpenEnabled = state
        } = action.payload;
        return isVideoCallDevtoolsAutoOpenEnabled;
      }
    default:
      return state;
  }
};

const isVideoCallScreenCaptureFallbackEnabled = (state = false, action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      return Boolean(action.payload.isVideoCallScreenCaptureFallbackEnabled);
    case UPDATES_READY:
      {
        const {
          isVideoCallScreenCaptureFallbackEnabled = state
        } = action.payload;
        return isVideoCallScreenCaptureFallbackEnabled;
      }
    case SETTINGS_SET_IS_VIDEO_CALL_SCREEN_CAPTURE_FALLBACK_ENABLED_CHANGED:
      return action.payload;
    default:
      return state;
  }
};

const isVideoCallWindowPersistenceEnabled = (state = true,
// Enabled by default
action) => {
  switch (action.type) {
    case SETTINGS_SET_IS_VIDEO_CALL_WINDOW_PERSISTENCE_ENABLED_CHANGED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          isVideoCallWindowPersistenceEnabled = state
        } = action.payload;
        return isVideoCallWindowPersistenceEnabled;
      }
    default:
      return state;
  }
};

const lastSelectedServerUrl = (state = '', action) => {
  switch (action.type) {
    case APP_SETTINGS_LOADED:
      {
        const {
          lastSelectedServerUrl = state,
          servers
        } = action.payload;
        if (state === '' && servers && servers.length > 0) {
          return servers[0].url;
        }
        return lastSelectedServerUrl;
      }
    case SIDE_BAR_SERVER_SELECTED:
      return action.payload;
    default:
      return state;
  }
};

const SCREEN_SHARING_DIALOG_DISMISSED = 'screen-sharing-dialog/dismissed';

const openDialog = (state = null, action) => {
  switch (action.type) {
    case MENU_BAR_ABOUT_CLICKED:
      return 'about';
    case WEBVIEW_SCREEN_SHARING_SOURCE_REQUESTED:
      return 'screen-sharing';
    case UPDATES_NEW_VERSION_AVAILABLE:
      return 'update';
    case CERTIFICATES_CLIENT_CERTIFICATE_REQUESTED:
      return 'select-client-certificate';
    case ABOUT_DIALOG_DISMISSED:
      if (state === 'about') {
        return null;
      }
      return state;
    case OUTLOOK_CALENDAR_ASK_CREDENTIALS:
      return 'outlook-credentials';
    case SCREEN_SHARING_DIALOG_DISMISSED:
    case WEBVIEW_SCREEN_SHARING_SOURCE_RESPONDED:
    case SELECT_CLIENT_CERTIFICATE_DIALOG_CERTIFICATE_SELECTED:
    case SELECT_CLIENT_CERTIFICATE_DIALOG_DISMISSED:
    case UPDATE_DIALOG_DISMISSED:
    case UPDATE_DIALOG_SKIP_UPDATE_CLICKED:
    case UPDATE_DIALOG_REMIND_UPDATE_LATER_CLICKED:
    case UPDATE_DIALOG_INSTALL_BUTTON_CLICKED:
    case OUTLOOK_CALENDAR_DIALOG_DISMISSED:
    case OUTLOOK_CALENDAR_SET_CREDENTIALS:
      return null;
    default:
      return state;
  }
};

const rootWindowIcon = (state = null, action) => {
  switch (action.type) {
    case ROOT_WINDOW_ICON_CHANGED:
      {
        return action.payload;
      }
    default:
      return state;
  }
};

const rootWindowState = (state = {
  focused: true,
  visible: true,
  maximized: false,
  minimized: false,
  fullscreen: false,
  normal: true,
  bounds: {
    x: undefined,
    y: undefined,
    width: 1000,
    height: 600
  }
}, action) => {
  switch (action.type) {
    case ROOT_WINDOW_STATE_CHANGED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          rootWindowState = state
        } = action.payload;
        return rootWindowState;
      }
    default:
      return state;
  }
};

const selectedBrowser = (state = null, action) => {
  switch (action.type) {
    case SETTINGS_SELECTED_BROWSER_CHANGED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          selectedBrowser = state
        } = action.payload;
        return selectedBrowser;
      }
    default:
      return state;
  }
};

const videoCallWindowState = (state = {
  focused: true,
  visible: true,
  maximized: false,
  minimized: false,
  fullscreen: false,
  normal: true,
  bounds: {
    x: undefined,
    y: undefined,
    width: 0,
    height: 0
  }
}, action) => {
  switch (action.type) {
    case VIDEO_CALL_WINDOW_STATE_CHANGED:
      return action.payload;
    case APP_SETTINGS_LOADED:
      {
        const {
          videoCallWindowState = state
        } = action.payload;
        return videoCallWindowState;
      }
    default:
      return state;
  }
};

const doCheckForUpdatesOnStartup = (state = true, action) => {
  switch (action.type) {
    case UPDATES_READY:
      {
        const {
          doCheckForUpdatesOnStartup
        } = action.payload;
        return doCheckForUpdatesOnStartup;
      }
    case ABOUT_DIALOG_TOGGLE_UPDATE_ON_START:
      {
        const doCheckForUpdatesOnStartup = action.payload;
        return doCheckForUpdatesOnStartup;
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          doCheckForUpdatesOnStartup = state
        } = action.payload;
        return doCheckForUpdatesOnStartup;
      }
    default:
      return state;
  }
};
const isCheckingForUpdates = (state = false, action) => {
  switch (action.type) {
    case UPDATES_CHECKING_FOR_UPDATE:
      return true;
    case UPDATES_ERROR_THROWN:
      return false;
    case UPDATES_NEW_VERSION_NOT_AVAILABLE:
      return false;
    case UPDATES_NEW_VERSION_AVAILABLE:
      return false;
    default:
      return state;
  }
};
const isEachUpdatesSettingConfigurable = (state = true, action) => {
  switch (action.type) {
    case UPDATES_READY:
      {
        const {
          isEachUpdatesSettingConfigurable
        } = action.payload;
        return isEachUpdatesSettingConfigurable;
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          isEachUpdatesSettingConfigurable = state
        } = action.payload;
        return isEachUpdatesSettingConfigurable;
      }
    default:
      return state;
  }
};
const isUpdatingAllowed = (state = true, action) => {
  switch (action.type) {
    case UPDATES_READY:
      {
        const {
          isUpdatingAllowed
        } = action.payload;
        return isUpdatingAllowed;
      }
    default:
      return state;
  }
};
const isUpdatingEnabled = (state = true, action) => {
  switch (action.type) {
    case UPDATES_READY:
      {
        const {
          isUpdatingEnabled
        } = action.payload;
        return isUpdatingEnabled;
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          isUpdatingEnabled = state
        } = action.payload;
        return isUpdatingEnabled;
      }
    default:
      return state;
  }
};
const newUpdateVersion = (state = null, action) => {
  switch (action.type) {
    case UPDATES_NEW_VERSION_AVAILABLE:
      {
        const newUpdateVersion = action.payload;
        return newUpdateVersion;
      }
    case UPDATES_NEW_VERSION_NOT_AVAILABLE:
    case UPDATE_SKIPPED:
      {
        return null;
      }
    default:
      return state;
  }
};
const skippedUpdateVersion = (state = null, action) => {
  switch (action.type) {
    case UPDATES_READY:
      {
        const {
          skippedUpdateVersion
        } = action.payload;
        return skippedUpdateVersion;
      }
    case UPDATE_SKIPPED:
      {
        const skippedUpdateVersion = action.payload;
        return skippedUpdateVersion;
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          skippedUpdateVersion = state
        } = action.payload;
        return skippedUpdateVersion;
      }
    default:
      return state;
  }
};
const updateError = (state = null, action) => {
  switch (action.type) {
    case UPDATES_CHECKING_FOR_UPDATE:
      return null;
    case UPDATES_ERROR_THROWN:
      return action.payload;
    case UPDATES_NEW_VERSION_NOT_AVAILABLE:
      return null;
    case UPDATES_NEW_VERSION_AVAILABLE:
      return null;
    default:
      return state;
  }
};
const updateChannel = (state = 'latest', action) => {
  switch (action.type) {
    case ABOUT_DIALOG_UPDATE_CHANNEL_CHANGED:
    case UPDATES_CHANNEL_CHANGED:
      {
        return action.payload;
      }
    case UPDATES_READY:
      {
        const {
          updateChannel
        } = action.payload;
        return updateChannel;
      }
    case APP_SETTINGS_LOADED:
      {
        const {
          updateChannel = state
        } = action.payload;
        return updateChannel;
      }
    default:
      return state;
  }
};

const rootReducer = redux.combineReducers({
  allowedJitsiServers,
  appPath,
  appVersion,
  availableBrowsers,
  clientCertificates,
  currentView,
  dialogs,
  doCheckForUpdatesOnStartup,
  downloads,
  externalProtocols,
  isCheckingForUpdates,
  isEachUpdatesSettingConfigurable,
  isMenuBarEnabled,
  isMessageBoxFocused,
  isShowWindowOnUnreadChangedEnabled,
  isSideBarEnabled,
  isTrayIconEnabled,
  isMinimizeOnCloseEnabled,
  isUpdatingAllowed,
  isUpdatingEnabled,
  mainWindowTitle,
  machineTheme,
  newUpdateVersion,
  openDialog,
  rootWindowIcon,
  rootWindowState,
  selectedBrowser,
  servers,
  skippedUpdateVersion,
  trustedCertificates,
  notTrustedCertificates,
  updateError,
  isReportEnabled,
  isFlashFrameEnabled,
  isHardwareAccelerationEnabled,
  isInternalVideoChatWindowEnabled,
  isAddNewServersEnabled,
  hasHideOnTrayNotificationShown,
  lastSelectedServerUrl,
  allowedNTLMCredentialsDomains,
  isNTLMCredentialsEnabled,
  videoCallWindowState,
  isVideoCallWindowPersistenceEnabled,
  isDeveloperModeEnabled,
  updateChannel,
  screenCaptureFallbackForced,
  isVideoCallDevtoolsAutoOpenEnabled,
  isTransparentWindowEnabled,
  isVideoCallScreenCaptureFallbackEnabled
});

let reduxStore;
let lastAction;
const catchLastAction = () => next => action => {
  lastAction = action;
  return next(action);
};
const createRendererReduxStore = async () => {
  const initialState = await getInitialState();
  const composeEnhancers = window.__REDUX_DEVTOOLS_EXTENSION_COMPOSE__ || redux.compose;
  const enhancers = composeEnhancers(redux.applyMiddleware(forwardToMain, catchLastAction));
  reduxStore = redux.createStore(rootReducer, initialState, enhancers);
  return reduxStore;
};
const dispatch = action => {
  reduxStore.dispatch(action);
};
const select = selector => selector(reduxStore.getState());
const watch = (selector, watcher) => {
  const initial = select(selector);
  watcher(initial, undefined);
  let prev = initial;
  return reduxStore.subscribe(() => {
    const curr = select(selector);
    if (Object.is(prev, curr)) {
      return;
    }
    watcher(curr, prev);
    prev = curr;
  });
};
const listen = (typeOrPredicate, listener) => {
  const effectivePredicate = typeof typeOrPredicate === 'function' ? typeOrPredicate : action => action.type === typeOrPredicate;
  return reduxStore.subscribe(() => {
    if (!effectivePredicate(lastAction)) {
      return;
    }
    listener(lastAction);
  });
};

// const isResponseTo = <Response extends RootAction>(id: unknown, type: Response['type']) =>
//   (action: RootAction): action is Response =>
//     isResponse(action) && action.type === type && action.meta.id === id;

const request = (requestAction, ...types) => new Promise((resolve, reject) => {
  const id = Math.random().toString(36).slice(2);
  const unsubscribe = listen(isResponseTo(id, ...types), action => {
    unsubscribe();
    if (isErrored(action)) {
      reject(action.payload);
      return;
    }
    if (hasPayload(action)) {
      resolve(action.payload);
    }
  });
  dispatch({
    ...requestAction,
    meta: {
      request: true,
      id
    }
  });
});

const NOTIFICATIONS_CREATE_REQUESTED = 'notifications/create-requested';
const NOTIFICATIONS_CREATE_RESPONDED = 'notifications/create-responded';
const NOTIFICATIONS_NOTIFICATION_ACTIONED = 'notifications/notification-actioned';
const NOTIFICATIONS_NOTIFICATION_CLICKED = 'notifications/notification-clicked';
const NOTIFICATIONS_NOTIFICATION_CLOSED = 'notifications/notification-closed';
const NOTIFICATIONS_NOTIFICATION_DISMISSED = 'notifications/notification-dismissed';
const NOTIFICATIONS_NOTIFICATION_REPLIED = 'notifications/notification-replied';
const NOTIFICATIONS_NOTIFICATION_SHOWN = 'notifications/notification-shown';

const normalizeIconUrl = iconUrl => {
  if (/^data:/.test(iconUrl)) {
    return iconUrl;
  }
  if (!/^https?:\/\//.test(iconUrl)) {
    return getAbsoluteUrl(iconUrl);
  }
  return iconUrl;
};
const eventHandlers = new Map();
const createNotification = async ({
  title,
  icon,
  onEvent,
  ...options
}) => {
  const id = await request({
    type: NOTIFICATIONS_CREATE_REQUESTED,
    payload: {
      title,
      ...(icon ? {
        icon: normalizeIconUrl(icon)
      } : {}),
      ...options
    }
  }, NOTIFICATIONS_CREATE_RESPONDED);
  eventHandlers.set(id, event => onEvent === null || onEvent === void 0 ? void 0 : onEvent({
    type: event.type,
    detail: event.detail
  }));
  return id;
};
const destroyNotification = id => {
  dispatch({
    type: NOTIFICATIONS_NOTIFICATION_DISMISSED,
    payload: {
      id
    }
  });
  eventHandlers.delete(id);
};
const dispatchCustomNotification = async options => {
  const {
    id,
    payload,
    type
  } = options;
  const notificationId = id || Math.random().toString(36).slice(2);
  return createNotification({
    title: payload.title,
    body: payload.body,
    icon: payload.avatar,
    tag: notificationId,
    requireInteraction: payload.requireInteraction,
    notificationType: type
  });
};
const closeCustomNotification = id => {
  destroyNotification(id);
};
const listenToNotificationsRequests = () => {
  listen(NOTIFICATIONS_NOTIFICATION_SHOWN, action => {
    const {
      payload: {
        id
      }
    } = action;
    const eventHandler = eventHandlers.get(id);
    eventHandler === null || eventHandler === void 0 || eventHandler({
      type: 'show'
    });
  });
  listen(NOTIFICATIONS_NOTIFICATION_CLOSED, action => {
    const {
      payload: {
        id
      }
    } = action;
    const eventHandler = eventHandlers.get(id);
    eventHandler === null || eventHandler === void 0 || eventHandler({
      type: 'close'
    });
    eventHandlers.delete(id);
  });
  listen(NOTIFICATIONS_NOTIFICATION_CLICKED, action => {
    const {
      payload: {
        id,
        serverUrl,
        category
      }
    } = action;
    if (category === 'DOWNLOADS') {
      dispatch({
        type: SIDE_BAR_DOWNLOADS_BUTTON_CLICKED
      });
    } else {
      dispatch({
        type: WEBVIEW_FOCUS_REQUESTED,
        payload: {
          url: serverUrl || getServerUrl(),
          view: 'server'
        }
      });
    }
    const eventHandler = eventHandlers.get(id);
    eventHandler === null || eventHandler === void 0 || eventHandler({
      type: 'click'
    });
  });
  listen(NOTIFICATIONS_NOTIFICATION_REPLIED, action => {
    const {
      payload: {
        id,
        reply
      }
    } = action;
    const eventHandler = eventHandlers.get(id);
    eventHandler === null || eventHandler === void 0 || eventHandler({
      type: 'reply',
      detail: {
        reply
      }
    });
  });
  listen(NOTIFICATIONS_NOTIFICATION_ACTIONED, action => {
    const {
      payload: {
        id,
        index
      }
    } = action;
    const eventHandler = eventHandlers.get(id);
    eventHandler === null || eventHandler === void 0 || eventHandler({
      type: 'action',
      detail: {
        index
      }
    });
  });
};

const handleGetSourceIdEvent = async () => {
  try {
    var _window$top;
    const sourceId = await request({
      type: WEBVIEW_SCREEN_SHARING_SOURCE_REQUESTED
    }, WEBVIEW_SCREEN_SHARING_SOURCE_RESPONDED);
    (_window$top = window.top) === null || _window$top === void 0 || _window$top.postMessage({
      sourceId
    }, '*');
  } catch (error) {
    var _window$top2;
    (_window$top2 = window.top) === null || _window$top2 === void 0 || _window$top2.postMessage({
      sourceId: 'PermissionDeniedError'
    }, '*');
  }
};
const listenToScreenSharingRequests = () => {
  window.addEventListener('get-sourceId', handleGetSourceIdEvent);
};

const getOutlookEvents = async date => {
  const response = await electron.ipcRenderer.invoke('outlook-calendar/get-events', date);
  return response;
};
const setOutlookExchangeUrl = (url, userId) => {
  electron.ipcRenderer.invoke('outlook-calendar/set-exchange-url', url, userId);
};
const hasOutlookCredentials = async () => electron.ipcRenderer.invoke('outlook-calendar/has-credentials');
const clearOutlookCredentials = () => {
  electron.ipcRenderer.invoke('outlook-calendar/clear-credentials');
};
const setUserToken = (token, userId) => {
  electron.ipcRenderer.invoke('outlook-calendar/set-user-token', token, userId);
};

const SYSTEM_LOCKING_SCREEN = 'system/locking-screen';
const SYSTEM_SUSPENDING = 'system/suspending';

let detachCallbacks;
const attachCallbacks = ({
  isAutoAwayEnabled,
  idleThreshold,
  setUserOnline
}) => {
  const unsubscribeFromPowerMonitorEvents = listen(action => [SYSTEM_SUSPENDING, SYSTEM_LOCKING_SCREEN].includes(action.type), () => {
    if (!isAutoAwayEnabled) {
      return;
    }
    setUserOnline(false);
  });
  let pollingTimer;
  let prevState;
  const pollSystemIdleState = async () => {
    if (!isAutoAwayEnabled || !idleThreshold) {
      return;
    }
    pollingTimer = setTimeout(pollSystemIdleState, 2000);
    const state = await invoke('power-monitor/get-system-idle-state', idleThreshold);
    if (prevState === state) {
      return;
    }
    const isOnline = state === 'active' || state === 'unknown';
    setUserOnline(isOnline);
    prevState = state;
  };
  pollSystemIdleState();
  return () => {
    unsubscribeFromPowerMonitorEvents();
    clearTimeout(pollingTimer);
  };
};
const setUserPresenceDetection = options => {
  var _detachCallbacks;
  (_detachCallbacks = detachCallbacks) === null || _detachCallbacks === void 0 || _detachCallbacks();
  detachCallbacks = attachCallbacks(options);
};

const setBadge = badge => {
  dispatch({
    type: WEBVIEW_UNREAD_CHANGED,
    payload: {
      url: getServerUrl(),
      badge
    }
  });
};

const writeTextToClipboard = text => {
  electron.clipboard.writeText(text);
};

const openDocumentViewer = (url, format, options) => {
  electron.ipcRenderer.invoke('document-viewer/open-window', url, format, options);
};

const FAVICON_SIZE = 100;
let imageElement;
const getImageElement = () => {
  if (!imageElement) {
    const canvas = document.createElement('canvas');
    canvas.width = FAVICON_SIZE;
    canvas.height = FAVICON_SIZE;
    const ctx = canvas.getContext('2d');
    if (!ctx) {
      throw new Error('failed to create canvas 2d context');
    }
    imageElement = new Image();
    const handleImageLoadEvent = () => {
      ctx.clearRect(0, 0, FAVICON_SIZE, FAVICON_SIZE);
      ctx.drawImage(imageElement, 0, 0, FAVICON_SIZE, FAVICON_SIZE);
      dispatch({
        type: WEBVIEW_FAVICON_CHANGED,
        payload: {
          url: getServerUrl(),
          favicon: canvas.toDataURL()
        }
      });
    };
    imageElement.addEventListener('load', handleImageLoadEvent, {
      passive: true
    });
  }
  return imageElement;
};
const setFavicon = faviconUrl => {
  if (typeof faviconUrl !== 'string') {
    return;
  }
  const imageElement = getImageElement();
  imageElement.src = getAbsoluteUrl(faviconUrl);
};

const setGitCommitHash = gitCommitHash => {
  dispatch({
    type: WEBVIEW_GIT_COMMIT_HASH_CHECK,
    payload: {
      url: getServerUrl(),
      gitCommitHash
    }
  });
};

const readSetting = key => {
  try {
    const filePath = path__default.default.join(electron.app.getPath('userData'), 'config.json');
    const content = fs__default.default.readFileSync(filePath, 'utf8');
    const json = JSON.parse(content);
    return json[key];
  } catch (e) {
    return null;
  }
};

// Cache browsers to avoid repeatedly fetching them
let cachedBrowsers = null;
let browserLoadPromise = null;

/**
 * Lazy load browsers asynchronously
 * This ensures we don't slow down app startup
 */
const loadBrowsersLazy = () => {
  if (cachedBrowsers) {
    return Promise.resolve(cachedBrowsers);
  }
  if (!browserLoadPromise) {
    // Start loading browsers asynchronously after a delay to not block the app startup
    browserLoadPromise = new Promise(resolve => {
      // Delay browser detection for 2 seconds after this function is first called
      // to avoid slowing down app startup and initial interactions
      setTimeout(async () => {
        try {
          const browsers = await detectBrowsers.getAvailableBrowsers();
          cachedBrowsers = browsers;
          const browserIds = browsers.map(browser => browser.browser);
          if (browserIds.length > 0) {
            dispatch({
              type: SETTINGS_AVAILABLE_BROWSERS_UPDATED,
              payload: browserIds
            });
          }
          resolve(browsers);
        } catch (error) {
          console.error('Error detecting browsers:', error);
          resolve([]);
        }
      }, 2000);
    });
  }
  return browserLoadPromise;
};

/**
 * Launches a URL in the selected browser from settings or falls back to system default
 *
 * @param url The URL to open
 * @returns Promise that resolves when the browser is launched
 */
const openExternal = async url => {
  // Get the selected browser from settings
  const selectedBrowser = readSetting('selectedBrowser');

  // If no specific browser is selected, use the system default
  if (!selectedBrowser) {
    return electron.shell.openExternal(url);
  }
  try {
    // Lazy load browsers when needed
    const browsers = await loadBrowsersLazy();

    // Find the selected browser in the available browsers
    const browser = browsers.find(browser => browser.browser === selectedBrowser);
    if (browser) {
      // Launch the selected browser with the URL
      return detectBrowsers.launchBrowser(browser, url);
    }
    // If the selected browser isn't available, fall back to system default
    console.warn(`Selected browser "${selectedBrowser}" not found, using system default.`);
    return electron.shell.openExternal(url);
  } catch (error) {
    console.error('Error launching browser:', error);
    // Fall back to shell.openExternal on error
    return electron.shell.openExternal(url);
  }
};

const getInternalVideoChatWindowEnabled = () => select(({
  isInternalVideoChatWindowEnabled
}) => ({
  isInternalVideoChatWindowEnabled
})).isInternalVideoChatWindowEnabled;
const openInternalVideoChatWindow = (url, options) => {
  const validUrl = new URL(url);
  const allowedProtocols = ['http:', 'https:'];
  if (!allowedProtocols.includes(validUrl.protocol)) {
    return;
  }
  if (!process.mas && getInternalVideoChatWindowEnabled()) {
    switch (options === null || options === void 0 ? void 0 : options.providerName) {
      case 'jitsi':
        // window.open(validUrl.href, 'Video Call', 'scrollbars=true');
        // We will open Jitsi on browser instead of opening a new window for compatibility from their side
        electron.ipcRenderer.invoke('video-call-window/open-window', validUrl.href, options);
        break;
      case 'googlemeet':
        openExternal(validUrl.href);
        break;
      default:
        electron.ipcRenderer.invoke('video-call-window/open-window', validUrl.href, options);
        break;
    }
  } else {
    openExternal(validUrl.href);
  }
};

const reloadServer = () => {
  const url = getServerUrl();

  // Dispatch action to trigger force reload with cache clear
  dispatch({
    type: WEBVIEW_FORCE_RELOAD_WITH_CACHE_CLEAR,
    payload: url
  });
};

let timer;
let prevBackground;
let prevColor;
let prevBorder;
let serverVersion;
function versionIsGreaterOrEqualsTo(version1, version2) {
  var _version1$match, _version2$match;
  const v1 = ((_version1$match = version1.match(/\d+/g)) === null || _version1$match === void 0 ? void 0 : _version1$match.map(Number)) || [];
  const v2 = ((_version2$match = version2.match(/\d+/g)) === null || _version2$match === void 0 ? void 0 : _version2$match.map(Number)) || [];
  for (let i = 0; i < 3; i++) {
    const n1 = v1[i] || 0;
    const n2 = v2[i] || 0;
    if (n1 > n2) {
      return true;
    }
    if (n1 < n2) {
      return false;
    }
  }
  return true;
}
const pollSidebarStyle = (referenceElement, emit) => {
  clearTimeout(timer);
  document.body.append(referenceElement);
  const {
    background,
    color,
    border
  } = window.getComputedStyle(referenceElement);
  referenceElement.remove();
  const newBgg = prevBackground !== background ? background : prevBackground;
  const newColor = prevColor !== color ? color : prevColor;
  const newBorder = prevBorder !== border ? border : prevBorder;
  if (prevBackground !== background || prevColor !== color || newBorder !== border) {
    emit({
      background: newBgg,
      color: newColor,
      border: newBorder
    });
    prevBackground = background;
    prevColor = color;
    prevBorder = border;
  }
  timer = setTimeout(() => pollSidebarStyle(referenceElement, emit), 5000);
};
let element;
const getElement = () => {
  if (!element) {
    element = document.createElement('div');
    element.style.backgroundColor = 'var(--sidebar-background)';
    element.style.color = 'var(--sidebar-item-text-color)';
    element.style.display = 'none';
    if (versionIsGreaterOrEqualsTo(serverVersion, '6.3.0')) {
      element.classList.add('rcx-sidebar--main');
      element.style.border = '1px solid var(--sidebar-border-color)';
    } else {
      element.classList.add('sidebar');
    }
  }
  return element;
};
const setServerVersionToSidebar = version => {
  serverVersion = version;
};
const setBackground = imageUrl => {
  const element = getElement();
  element.style.backgroundImage = imageUrl ? `url(${JSON.stringify(getAbsoluteUrl(imageUrl))})` : 'none';
  pollSidebarStyle(element, sideBarStyle => {
    dispatch({
      type: WEBVIEW_SIDEBAR_STYLE_CHANGED,
      payload: {
        url: getServerUrl(),
        style: sideBarStyle
      }
    });
  });
};
const setSidebarCustomTheme = customTheme => {
  dispatch({
    type: WEBVIEW_SIDEBAR_CUSTOM_THEME_CHANGED,
    payload: {
      url: getServerUrl(),
      customTheme
    }
  });
};

const setUserThemeAppearance = themeAppearance => {
  dispatch({
    type: WEBVIEW_USER_THEME_APPEARANCE_CHANGED,
    payload: {
      url: getServerUrl(),
      themeAppearance
    }
  });
};

const setTitle = title => {
  if (typeof title !== 'string') {
    return;
  }
  const url = getServerUrl();
  if (title === 'Rocket.Chat' && new URL(url).host !== 'open.rocket.chat') {
    dispatch({
      type: WEBVIEW_TITLE_CHANGED,
      payload: {
        url,
        title: `${title} - ${url}`
      }
    });
    return;
  }
  dispatch({
    type: WEBVIEW_TITLE_CHANGED,
    payload: {
      url,
      title
    }
  });
};

const setUserLoggedIn = userLoggedIn => {
  dispatch({
    type: WEBVIEW_USER_LOGGED_IN,
    payload: {
      url: getServerUrl(),
      userLoggedIn
    }
  });
};

let serverInfo;
let cb = _serverInfo => undefined;
const RocketChatDesktop = {
  onReady: c => {
    if (serverInfo) {
      c(serverInfo);
    }
    cb = c;
  },
  setServerInfo: _serverInfo => {
    serverInfo = _serverInfo;
    cb(_serverInfo);
    setServerVersionToSidebar(_serverInfo.version);
  },
  setUrlResolver,
  setBadge,
  setFavicon,
  setBackground,
  setTitle,
  setUserPresenceDetection,
  setUserLoggedIn,
  setUserThemeAppearance,
  createNotification,
  destroyNotification,
  dispatchCustomNotification,
  closeCustomNotification,
  getInternalVideoChatWindowEnabled,
  openInternalVideoChatWindow,
  setGitCommitHash,
  writeTextToClipboard,
  getOutlookEvents,
  setOutlookExchangeUrl,
  hasOutlookCredentials,
  clearOutlookCredentials,
  setUserToken,
  setSidebarCustomTheme,
  openDocumentViewer,
  reloadServer
};

function debounce(cb, wait = 20) {
  let h;
  const callable = (...args) => {
    h && clearTimeout(h);
    h = setTimeout(() => cb(...args), wait);
  };
  return callable;
}

let focusedMessageBoxInput = null;
const handleFocusEvent = event => {
  if (!(event.target instanceof Element)) {
    return;
  }
  if (!event.target.classList.contains('js-input-message')) {
    return;
  }
  focusedMessageBoxInput = event.target;
  dispatch({
    type: WEBVIEW_MESSAGE_BOX_FOCUSED
  });
};
const handleBlurEvent = event => {
  if (!(event.target instanceof Element)) {
    return;
  }
  if (!event.target.classList.contains('js-input-message')) {
    return;
  }
  focusedMessageBoxInput = null;
  dispatch({
    type: WEBVIEW_MESSAGE_BOX_BLURRED
  });
};
const listenToMessageBoxEvents = () => {
  listen(TOUCH_BAR_FORMAT_BUTTON_TOUCHED, action => {
    if (!focusedMessageBoxInput) {
      return;
    }
    const {
      payload: buttonId
    } = action;
    const ancestor = focusedMessageBoxInput.closest('.rc-message-box');
    const button = ancestor === null || ancestor === void 0 ? void 0 : ancestor.querySelector(`[data-id='${buttonId}']`);
    button === null || button === void 0 || button.click();
  });
  document.addEventListener('focus', handleFocusEvent, true);
  document.addEventListener('blur', handleBlurEvent, true);
};

const selectIsSideBarVisible = ({
  servers,
  isSideBarEnabled
}) => servers.length > 0 && isSideBarEnabled;
const handleTrafficLightsSpacing = () => {
  if (process.platform !== 'darwin') {
    return;
  }
  const style = document.getElementById('sidebar-padding') || document.createElement('style');
  style.id = 'sidebar-padding';
  document.head.append(style);
  watch(selectIsSideBarVisible, isSideBarVisible => {
    style.innerHTML = `
      .sidebar {
        padding-top: ${isSideBarVisible ? 0 : '10px'} !important;
        transition: padding-top 230ms ease-in-out !important;
      }
    `;
  });
};

const whenReady = () => new Promise(resolve => {
  if (document.readyState === 'complete') {
    resolve();
    return;
  }
  const handleReadyStateChange = () => {
    if (document.readyState !== 'complete') {
      return;
    }
    document.removeEventListener('readystatechange', handleReadyStateChange);
    resolve();
  };
  document.addEventListener('readystatechange', handleReadyStateChange);
});

console.log('[Rocket.Chat Desktop] Preload.ts');
electron.contextBridge.exposeInMainWorld('JitsiMeetElectron', JitsiMeetElectron);
electron.contextBridge.exposeInMainWorld('RocketChatDesktop', RocketChatDesktop);
let retryCount = 0;
const start = async () => {
  console.log('[Rocket.Chat Desktop] Preload.ts start fired');
  const serverUrl = await invoke('server-view/get-url');
  if (retryCount > 5) return;
  if (!serverUrl) {
    console.log('[Rocket.Chat Desktop] serverUrl is not defined');
    console.log('[Rocket.Chat Desktop] Preload start - retrying in 1 seconds');
    setTimeout(start, 1000);
    retryCount += 1;
    return;
  }
  window.removeEventListener('load', start);
  setServerUrl(serverUrl);
  await whenReady();
  await createRendererReduxStore();
  await invoke('server-view/ready');
  console.log('[Rocket.Chat Desktop] waiting for RocketChatDesktop.onReady');
  RocketChatDesktop.onReady(() => {
    console.log('[Rocket.Chat Desktop] RocketChatDesktop.onReady fired');
    listen(WEBVIEW_DID_NAVIGATE, debounce(() => {
      const resources = electron.webFrame.getResourceUsage();
      // TODO: make this configurable
      if (resources.images.size > 50 * 1024 * 1024) {
        electron.webFrame.clearCache();
      }
    }, 1000 * 30));
    listenToNotificationsRequests();
    listenToScreenSharingRequests();
    listenToMessageBoxEvents();
    handleTrafficLightsSpacing();
  });
};
console.log('[Rocket.Chat Desktop] waiting for window load');
window.addEventListener('load', start);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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
