'use strict';

var main = require('./main.js');
var electron = require('electron');
require('child_process');
require('fs');
require('rimraf');
require('redux');
require('path');
require('i18next');
require('reselect');
require('electron-store');
require('detect-browsers');
require('semver');
require('os');
require('electron-dl');
require('@bugsnag/js');
require('axios');
require('@ewsjs/xhr');
require('ews-javascript-api');
require('node:fs/promises');
require('node:path');
require('jsonwebtoken');
require('moment');
require('process');
require('electron-updater');

/**
 * Internal picker provider that uses the React UI and desktopCapturer cache.
 * This provider works in both main and renderer processes:
 * - handleDisplayMediaRequest: called from main process, uses ipc.ts state
 * - initialize: called from renderer process, preloads UI and cache
 *
 * Note: This provider needs access to state in ipc.ts. The factory will
 * pass a handler function that encapsulates the internal picker logic.
 */
class InternalPickerProvider {
  constructor() {
    main._defineProperty(this, "type", 'internal');
    main._defineProperty(this, "requiresInternalUI", true);
    main._defineProperty(this, "requiresCacheWarming", true);
    main._defineProperty(this, "isInitialized", false);
    main._defineProperty(this, "handleRequestFn", null);
    main._defineProperty(this, "initializeFn", null);
  }
  /**
   * Set the handler function from main process (ipc.ts)
   * This encapsulates all the internal picker logic with access to ipc.ts state
   */
  setHandleRequestHandler(handler) {
    this.handleRequestFn = handler;
  }

  /**
   * Set the initialize function from renderer process (video-call-window.ts)
   * This encapsulates the UI preload and cache warming logic
   */
  setInitializeHandler(handler) {
    this.initializeFn = handler;
  }
  handleDisplayMediaRequest(callback) {
    if (this.handleRequestFn) {
      this.handleRequestFn(callback);
    } else {
      console.error('InternalPickerProvider: handleRequest handler not set. This should be set by ipc.ts');
      callback({
        video: false
      });
    }
  }
  async initialize() {
    if (this.isInitialized) return;
    console.log('Screen picker [internal]: initializing cache and UI');
    if (this.initializeFn) {
      await this.initializeFn();
    } else {
      console.warn('InternalPickerProvider: initialize handler not set. Call setInitializeHandler() first.');
    }
    this.isInitialized = true;
  }
  cleanup() {
    // Clear caches, reset state
    this.isInitialized = false;
  }
}

class PortalPickerProvider {
  constructor() {
    main._defineProperty(this, "type", 'portal');
    main._defineProperty(this, "requiresInternalUI", false);
    main._defineProperty(this, "requiresCacheWarming", false);
  }
  handleDisplayMediaRequest(callback) {
    // On Linux/Wayland, calling getSources() triggers the XDG portal picker.
    // The portal typically returns exactly one source on selection or an empty array
    // on cancellation; we defensively check for > 0 and use only the first source.
    console.log('Screen picker [portal]: triggering XDG portal picker via getSources()');
    electron.desktopCapturer.getSources({
      types: ['screen', 'window']
    }).then(sources => {
      if (sources.length > 0) {
        // User selected a source via portal picker
        callback({
          video: sources[0]
        });
      } else {
        // User cancelled or no source available
        console.warn('Screen picker [portal]: No source selected by user');
        callback({
          video: false
        });
      }
    }).catch(error => {
      console.error('Screen picker [portal]: Failed to get source from XDG portal:', error);
      callback({
        video: false
      });
    });
  }
  async initialize() {
    // Portal picker requires no initialization
    console.log('Screen picker [portal]: initialized (no-op)');
  }
  cleanup() {
    // Portal picker requires no cleanup
    console.log('Screen picker [portal]: cleanup (no-op)');
  }
}

function detectPickerType() {
  if (process.platform === 'linux') {
    const sessionType = process.env.XDG_SESSION_TYPE;
    const desktop = process.env.XDG_CURRENT_DESKTOP || '';
    const isWayland = sessionType === 'wayland';
    const hasPortal = /GNOME|KDE|XFCE|Cinnamon|MATE|Pantheon|Budgie|Unity/i.test(desktop);
    if (isWayland || hasPortal) {
      return 'portal';
    }
  }

  // Future: Add macOS native picker detection here
  // if (process.platform === 'darwin' && hasMacOSNativePicker()) {
  //   return 'native-macos';
  // }

  return 'internal';
}
let cachedProvider = null;
function createScreenPicker() {
  if (cachedProvider) return cachedProvider;
  const type = detectPickerType();
  switch (type) {
    case 'portal':
      cachedProvider = new PortalPickerProvider();
      break;
    case 'internal':
    default:
      cachedProvider = new InternalPickerProvider();
  }
  console.log(`Screen picker: using ${cachedProvider.type} provider`);
  return cachedProvider;
}
function getScreenPicker() {
  if (!cachedProvider) {
    throw new Error('Screen picker not initialized. Call createScreenPicker() first.');
  }
  return cachedProvider;
}
function resetScreenPicker() {
  var _cachedProvider;
  (_cachedProvider = cachedProvider) === null || _cachedProvider === void 0 || _cachedProvider.cleanup();
  cachedProvider = null;
}

exports.InternalPickerProvider = InternalPickerProvider;
exports.PortalPickerProvider = PortalPickerProvider;
exports.createScreenPicker = createScreenPicker;
exports.getScreenPicker = getScreenPicker;
exports.resetScreenPicker = resetScreenPicker;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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
